package com.kontakt.sdk.android.cloud.api.executor.devices;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.DevicesApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.DevicesService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import retrofit2.Call;

/**
 * Request executor provided by {@link DevicesApi}. Use this class if you want to assign
 * devices either to manager or to venue through fluent API in chained fashion. Example of use:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   kontaktCloud.devices().assign(IDs)
 *      .toManager(managerId)
 *      .execute();
 *   </code>
 * </pre>
 * Keep in mind that {@code toManager} cannot be used with {@code toVenue} and so on.
 * In such a situation an exception will be thrown at runtime.
 */
public class AssignDeviceRequestExecutor extends RequestExecutor<String> {

  private final DevicesService devicesService;

  private UUID venueId;
  private UUID managerId;
  private UUID[] deviceIds;
  private String[] uniqueIds;

  /**
   * Constructs request executor initialized with corresponding service class and device unique IDs.
   * @param devicesService the devices API service.
   * @param deviceIds device unique identifiers.
   */
  public AssignDeviceRequestExecutor(final DevicesService devicesService, final UUID... deviceIds) {
    this.devicesService = devicesService;
    this.deviceIds = deviceIds;
  }

  /**
   * Constructs request executor initialized with corresponding service class and device unique IDs.
   * @param devicesService the devices API service.
   * @param uniqueIds device unique identifiers.
   */
  public AssignDeviceRequestExecutor(final DevicesService devicesService, final String... uniqueIds) {
    this.devicesService = devicesService;
    this.uniqueIds = uniqueIds;
  }

  /**
   * Specifies the venue. Cannot be used with {@code toManager} simultaneously.
   * @param venueId venue unique identifier.
   * @return this request executor.
   */
  public AssignDeviceRequestExecutor toVenue(final UUID venueId) {
    SDKPreconditions.checkNotNull(venueId, "venue ID cannot be null");
    SDKPreconditions.checkState(managerId == null, "toVenue cannot be used with toManager");
    this.venueId = venueId;
    return this;
  }

  /**
   * Specifies the manager. Cannot be used with {@code toVenue} simultaneously.
   * @param managerId manager unique identifier.
   * @return this request executor.
   */
  public AssignDeviceRequestExecutor toManager(final UUID managerId) {
    SDKPreconditions.checkNotNull(managerId, "manager ID cannot be null");
    SDKPreconditions.checkState(venueId == null, "toManager cannot be used with toVenue");
    this.managerId = managerId;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public String execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override public void execute(final CloudCallback<String> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Call<String> prepareCall() {
    return devicesService.assignDevice(params());
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(venueId != null || managerId != null, "cannot assign device - specify venue or manager");
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    if (venueId != null) {
      params.put(CloudConstants.Devices.VENUE_ID_PARAMETER, venueId.toString());
    }
    if (managerId != null) {
      params.put(CloudConstants.Devices.MANAGER_ID_PARAMETER, managerId.toString());
    }
    if (deviceIds != null) {
      params.put(CloudConstants.Devices.DEVICE_ID_PARAMETER, StringUtils.join(deviceIds, ","));
    }
    if (uniqueIds != null) {
      params.put(CloudConstants.Common.UNIQUE_ID_PARAMETER, StringUtils.join(uniqueIds, ","));
    }
    return params;
  }
}
