package com.kontakt.sdk.android.cloud.api.executor.devices;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.DevicesApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.DevicesService;
import com.kontakt.sdk.android.cloud.response.paginated.Configs;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.model.Config;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import retrofit2.Call;

/**
 * Request executor provided by {@link DevicesApi}. Use this class if you want to apply
 * secure configs through fluent API in chained fashion. Example of use:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   // define secure configs
 *   kontaktCloud.devices().applySecureConfigs(secureConfigs).execute();
 *   </code>
 * </pre>
 * Note that every config object passed to {@code applySecureConfigs} method should contain
 * non-null unique ID and secure response with timestamp. Otherwise an exception will be thrown at runtime.
 */
public class ApplySecureConfigRequestExecutor extends RequestExecutor<Configs> {

  private final DevicesService devicesService;

  private Config[] configs;

  /**
   * Constructs request executor initialized with corresponding service class and configs to apply.
   * @param devicesService the devices API service.
   * @param configs secure configs to apply.
   */
  public ApplySecureConfigRequestExecutor(final DevicesService devicesService, final Config... configs) {
    this.devicesService = devicesService;
    this.configs = configs;
  }

  /**
   * Constructs request executor initialized with corresponding service class and configs to apply.
   * @param devicesService the devices API service.
   * @param configs secure configs to apply.
   */
  public ApplySecureConfigRequestExecutor(final DevicesService devicesService, final List<Config> configs) {
    this.devicesService = devicesService;
    final int size = configs.size();
    this.configs = configs.toArray(new Config[size]);
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Call<Configs> prepareCall() {
    return devicesService.applySecureConfigs(params());
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Common.UNIQUE_ID_PARAMETER, extractUniqueIds());
    params.put(CloudConstants.Devices.RESPONSE_PARAMETER, extractResponses());
    params.put(CloudConstants.Devices.UPDATED_AT_PARAMETER, extractTimestamps());
    return params;
  }

  private String extractUniqueIds() {
    final List<String> uniqueIds = new ArrayList<>();
    for (Config config : configs) {
      SDKPreconditions.checkState(config.getUniqueId() != null, "config doesn't contain the unique ID");
      uniqueIds.add(config.getUniqueId());
    }
    return StringUtils.join(uniqueIds, ",");
  }

  private String extractResponses() {
    final List<String> responses = new ArrayList<>();
    for (Config config : configs) {
      SDKPreconditions.checkState(config.getSecureResponse() != null, "config doesn't contain the secure response");
      responses.add(config.getSecureResponse());
    }
    return StringUtils.join(responses, ",");
  }

  private String extractTimestamps() {
    final List<Long> timestamps = new ArrayList<>();
    for (Config config : configs) {
      SDKPreconditions.checkState(config.getSecureResponseTime() != 0, "config doesn't contain the secure response timestamp");
      timestamps.add(config.getSecureResponseTime());
    }
    return StringUtils.join(timestamps, ",");
  }
}
