package com.kontakt.sdk.android.cloud.api.executor.configs;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.ConfigsApi;
import com.kontakt.sdk.android.cloud.api.executor.PaginatedRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ConfigsService;
import com.kontakt.sdk.android.cloud.response.paginated.Configs;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.Order;
import com.kontakt.sdk.android.common.model.OrderBy;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.List;
import java.util.Map;
import retrofit2.Call;

/**
 * Request executor provided by {@link ConfigsApi}. Use this class if you want to obtain
 * secure configs through fluent API in chained fashion. Here is an example of how to use this executor:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   Configs configs = kontaktCloud.configs().secure()
 *      .withIds("7HuT", "2Fge", "dFGh")
 *      .maxResult(10)
 *      .execute();
 *   </code>
 * </pre>
 */
public class SecureConfigsRequestExecutor extends PaginatedRequestExecutor<Configs> {

  private final ConfigsService configsService;

  private String[] uniqueIds;

  /**
   * Constructs request executor initialized with corresponding service class.
   * @param configsService the configs API service.
   */
  public SecureConfigsRequestExecutor(final ConfigsService configsService) {
    this.configsService = configsService;
  }

  /**
   * Specifies config unique IDs. Use that method if you want to fetch particular configs by ID.
   * @param uniqueIds config unique identifiers.
   * @return this request executor.
   */
  public SecureConfigsRequestExecutor withIds(final String... uniqueIds) {
    this.uniqueIds = SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    return this;
  }

  /**
   * Specifies config unique IDs. Use that method if you want to fetch particular configs by ID.
   * @param uniqueIds config unique identifiers.
   * @return this request executor.
   */
  public SecureConfigsRequestExecutor withIds(final List<String> uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    final int size = uniqueIds.size();
    this.uniqueIds = uniqueIds.toArray(new String[size]);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public SecureConfigsRequestExecutor startIndex(int startIndex) {
    super.startIndex(startIndex);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public SecureConfigsRequestExecutor maxResult(int maxResult) {
    super.maxResult(maxResult);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public SecureConfigsRequestExecutor filter(String query) {
    super.filter(query);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public SecureConfigsRequestExecutor orderBy(OrderBy orderBy, Order order) {
    super.orderBy(orderBy, order);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public SecureConfigsRequestExecutor eTag(String eTag) {
    super.eTag(eTag);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Call<Configs> prepareCall() {
    return configsService.getSecureConfigs(params());
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Map<String, String> params() {
    final Map<String, String> params = super.params();
    if (uniqueIds != null) {
      params.put(CloudConstants.Common.UNIQUE_ID_PARAMETER, StringUtils.join(uniqueIds, ","));
    }
    return params;
  }
}
