package com.kontakt.sdk.android.cloud.api.executor.actions;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.ActionsApi;
import com.kontakt.sdk.android.cloud.api.service.ActionsService;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.Map;
import java.util.UUID;

/**
 * Request executor provided by {@link ActionsApi}. Use this class if you want to update
 * browser actions through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   kontaktCloud.actions().updateBrowserAction(id)
 *      .withProximity(Proximity.IMMEDIATE)
 *      .withUrl("https://kontakt.io")
 *      .execute();
 *   </code>
 * </pre>
 */
public class UpdateBrowserActionRequestExecutor extends UpdateActionRequestExecutor {

  private String url;

  /**
   * Constructs request executor initialized with corresponding service class and action's id.
   * @param actionsService teh actions API service.
   * @param id the action's unique identifier.
   */
  public UpdateBrowserActionRequestExecutor(final ActionsService actionsService, final UUID id) {
    super(actionsService, id);
  }

  /**
   * Updates the action's URL.
   * @param url the URL.
   * @return this request executor.
   */
  public UpdateBrowserActionRequestExecutor withUrl(final String url) {
    SDKPreconditions.checkNotNullOrEmpty(url, "url cannot be null or empty");
    this.url = url;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public UpdateBrowserActionRequestExecutor withProximity(Proximity proximity) {
    super.withProximity(proximity);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override protected void checkPreconditions() {
    SDKPreconditions.checkState(url != null, "cannot update action - specify url");
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Map<String, String> params() {
    final Map<String, String> params = super.params();
    params.put(CloudConstants.Actions.URL_PARAMETER, url);

    return params;
  }
}
