package com.kontakt.sdk.android.cloud.api.executor.actions;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ActionsService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.model.Action;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import retrofit2.Call;

abstract class CreateActionRequestExecutor extends RequestExecutor<Action> {

  protected final ActionsService actionsService;

  protected String[] uniqueIds;
  protected Action.Type actionType;
  protected Proximity proximity;

  public CreateActionRequestExecutor(final ActionsService actionsService) {
    this.actionsService = actionsService;
  }

  /**
   * Specifies associated devices. The method invocation is obligatory while using this request executor.
   * @param uniqueIds the device unique identifiers.
   * @return this request executor.
   */
  public CreateActionRequestExecutor forDevices(final String... uniqueIds) {
    this.uniqueIds = SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    return this;
  }

  /**
   * Specifies associated devices. The method invocation is obligatory while using this request executor.
   * @param uniqueIds the device unique identifiers.
   * @return this request executor.
   */
  public CreateActionRequestExecutor forDevices(final List<String> uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    final int size = uniqueIds.size();
    this.uniqueIds = uniqueIds.toArray(new String[size]);
    return this;
  }

  /**
   * Specifies the proximity of newly created action.
   * @param proximity the proximity.
   * @return this request executor.
   */
  public CreateActionRequestExecutor withProximity(final Proximity proximity) {
    this.proximity = SDKPreconditions.checkNotNull(proximity, "proximity cannot be null");
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public Action execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override public void execute(final CloudCallback<Action> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Call<Action> prepareCall() {
    return actionsService.createAction(params());
  }

  /**
   * Checks if all pre conditions are satisfied before sending the request.
   */
  protected abstract void checkPreconditions();

  /**
   * {@inheritDoc}
   */
  @Override protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Common.UNIQUE_ID_PARAMETER, StringUtils.join(uniqueIds, ","));
    params.put(CloudConstants.Actions.ACTION_TYPE_PARAMETER, actionType.name());
    params.put(CloudConstants.Actions.PROXIMITY_PARAMETER, proximity.name());

    return params;
  }
}
