package com.kontakt.sdk.android.cloud.api;

import com.kontakt.sdk.android.cloud.api.executor.firmwares.FirmwareFileRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.firmwares.FirmwareRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.firmwares.FirmwaresRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.firmwares.ScheduleFirmwareUpdateRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.firmwares.UnscheduleFirmwareUpdateRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.FirmwaresService;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

/**
 * An abstract representation of Firmwares API endpoint. Provides executors that can be
 * used to handle all firmware-related requests.
 */
public class FirmwaresApi {

  private FirmwaresService firmwaresService;

  /**
   * Constructs Firmwares API endpoint initialized with corresponding service class.
   * @param firmwaresService the firmwares API service.
   */
  public FirmwaresApi(final FirmwaresService firmwaresService) {
    this.firmwaresService = firmwaresService;
  }

  /**
   * Provides executor that can be used to obtain single firmware by name.
   * @param name the firmware's name.
   * @return the request executor.
   */
  public FirmwareRequestExecutor fetch(final String name) {
    SDKPreconditions.checkNotNull(name, "name cannot be null");
    return new FirmwareRequestExecutor(firmwaresService, name);
  }

  /**
   * Provides executor that can be used to obtain firmwares.
   * @return the request executor.
   */
  public FirmwaresRequestExecutor fetch() {
    return new FirmwaresRequestExecutor(firmwaresService);
  }

  /**
   * Provides executor responsible for loading firmware file.
   * @param name the firmware's name.
   * @return the request executor.
   */
  public FirmwareFileRequestExecutor fileOf(final String name) {
    SDKPreconditions.checkNotNull(name, "name cannot be null");
    return new FirmwareFileRequestExecutor(firmwaresService, name);
  }

  /**
   * Provides executor that can be used to schedule firmware update.
   * @return the request executor.
   */
  public ScheduleFirmwareUpdateRequestExecutor scheduleUpdate() {
    return new ScheduleFirmwareUpdateRequestExecutor(firmwaresService);
  }

  /**
   * Provides executor that can be used to unschedule firmware update.
   * @return the request executor.
   */
  public UnscheduleFirmwareUpdateRequestExecutor unscheduleUpdate() {
    return new UnscheduleFirmwareUpdateRequestExecutor(firmwaresService);
  }
}
