package com.kontakt.sdk.android.cloud.api;

import com.kontakt.sdk.android.cloud.api.executor.configs.ConfigsRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.configs.CreateConfigRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.configs.DeletePendingConfigsRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.configs.SecureConfigsRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ConfigsService;
import com.kontakt.sdk.android.common.model.Config;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.List;

/**
 * An abstract representation of Configs API endpoint. Provides executors that can be
 * used to handle all config-related requests.
 */
public class ConfigsApi {

  private final ConfigsService configsService;

  /**
   * Constructs Configs API endpoint initialized with corresponding service class.
   * @param configsService the configs API service.
   */
  public ConfigsApi(final ConfigsService configsService) {
    this.configsService = configsService;
  }

  /**
   * Provides executor that can be used to obtain configs.
   * @return the request executor.
   */
  public ConfigsRequestExecutor fetch() {
    return new ConfigsRequestExecutor(configsService);
  }

  /**
   * Provides executor that can be used to obtain secure configs.
   * @return the request executor.
   */
  public SecureConfigsRequestExecutor secure() {
    return new SecureConfigsRequestExecutor(configsService);
  }

  /**
   * Provides executor that can be used to create new config.
   * @param config the new config.
   * @return the request executor.
   */
  public CreateConfigRequestExecutor create(final Config config) {
    SDKPreconditions.checkNotNull(config, "config cannot be null");
    return new CreateConfigRequestExecutor(configsService, config);
  }

  /**
   * Provides executor that can be used to delete pending configs.
   * @param uniqueIds config unique identifiers.
   * @return the request executor.
   */
  public DeletePendingConfigsRequestExecutor delete(final String... uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    return new DeletePendingConfigsRequestExecutor(configsService, uniqueIds);
  }

  /**
   * Provides executor that can be used to delete pending configs.
   * @param uniqueIds config unique identifiers.
   * @return the request executor.
   */
  public DeletePendingConfigsRequestExecutor delete(final List<String> uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    return new DeletePendingConfigsRequestExecutor(configsService, uniqueIds);
  }
}
