package com.kontakt.sdk.android.cloud.adapter;

import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonToken;
import com.google.gson.stream.JsonWriter;
import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.common.model.Config;
import com.kontakt.sdk.android.common.model.PacketType;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.util.EddystoneUtils;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

/**
 * Converts {@link Config} objects to and from JSON.
 */
public class ConfigTypeAdapter extends BaseTypeAdapter<Config> {

  /**
   * {@inheritDoc}
   */
  @Override public void write(JsonWriter out, Config config) throws IOException {
    writeUUID(out, CloudConstants.Configs.PROXIMITY_PARAMETER, config.getProximity());
    writeInteger(out, CloudConstants.Configs.MAJOR_PARAMETER, config.getMajor());
    writeInteger(out, CloudConstants.Configs.MINOR_PARAMETER, config.getMinor());
    writeInteger(out, CloudConstants.Configs.TX_POWER_PARAMETER, config.getTxPower());
    writeInteger(out, CloudConstants.Configs.INTERVAL_PARAMETER, config.getInterval());
    writeString(out, CloudConstants.Configs.NAMESPACE_PARAMETER, config.getNamespace());
    writeString(out, CloudConstants.Configs.INSTANCE_ID_PARAMETER, config.getInstanceId());
    final String url = config.getUrl();
    if (url != null) {
      final String encodedUrl = EddystoneUtils.toHexString(EddystoneUtils.serializeUrl(url));
      writeString(out, CloudConstants.Configs.URL_PARAMETER, encodedUrl);
    }
    writeProfiles(out, config.getProfiles());
    writePackets(out, config.getPackets());
    writeBoolean(out, CloudConstants.Configs.SHUFFLED_PARAMETER, config.isShuffled());
    writeString(out, CloudConstants.Configs.NAME_PARAMETER, config.getName());
    writeString(out, CloudConstants.Configs.PASSWORD_PARAMETER, config.getPassword());

    // write secure config parameters
    writeString(out, CloudConstants.Configs.SECURE_REQUEST_PARAMETER, config.getSecureRequest());
    writeString(out, CloudConstants.Configs.SECURE_RESPONSE_PARAMETER, config.getSecureResponse());
    writeLong(out, CloudConstants.Configs.SECURE_RESPONSE_TIME_PARAMETER, config.getSecureResponseTime());
  }

  /**
   * {@inheritDoc}
   */
  @Override public Config read(JsonReader in) throws IOException {
    final Config.Builder configBuilder = new Config.Builder();

    in.beginObject();
    while (in.hasNext()) {
      String fieldName = in.nextName();
      if (in.peek() == JsonToken.NULL) {
        in.skipValue();
        continue;
      }
      switch (fieldName) {
        case CloudConstants.Common.UNIQUE_ID_PARAMETER:
          final String uniqueId = in.nextString();
          configBuilder.uniqueId(uniqueId);
          break;
        case CloudConstants.Configs.PROXIMITY_PARAMETER:
          final UUID proximity = UUID.fromString(in.nextString());
          configBuilder.proximity(proximity);
          break;
        case CloudConstants.Configs.MAJOR_PARAMETER:
          final int major = in.nextInt();
          configBuilder.major(major);
          break;
        case CloudConstants.Configs.MINOR_PARAMETER:
          final int minor = in.nextInt();
          configBuilder.minor(minor);
          break;
        case CloudConstants.Configs.TX_POWER_PARAMETER:
          final int txPower = in.nextInt();
          configBuilder.txPower(txPower);
          break;
        case CloudConstants.Configs.INTERVAL_PARAMETER:
          final int interval = in.nextInt();
          configBuilder.interval(interval);
          break;
        case CloudConstants.Configs.NAMESPACE_PARAMETER:
          final String namespace = in.nextString();
          configBuilder.namespace(namespace);
          break;
        case CloudConstants.Configs.URL_PARAMETER:
          final String url = in.nextString();
          configBuilder.url(url);
          break;
        case CloudConstants.Configs.INSTANCE_ID_PARAMETER:
          final String instanceId = in.nextString();
          configBuilder.instanceId(instanceId);
          break;
        case CloudConstants.Configs.PROFILES_PARAMETER:
          final List<DeviceProfile> profiles = readProfiles(in);
          configBuilder.profiles(profiles);
          break;
        case CloudConstants.Configs.PACKETS_PARAMETER:
          final List<PacketType> packets = readPackets(in);
          configBuilder.packets(packets);
          break;
        case CloudConstants.Configs.SHUFFLED_PARAMETER:
          final boolean shuffled = in.nextBoolean();
          configBuilder.shuffled(shuffled);
          break;
        case CloudConstants.Configs.NAME_PARAMETER:
          final String name = in.nextString();
          configBuilder.name(name);
          break;
        case CloudConstants.Configs.PASSWORD_PARAMETER:
          final String password = in.nextString();
          configBuilder.password(password);
          break;
        case CloudConstants.Configs.SECURE_REQUEST_PARAMETER:
          final String secureRequest = in.nextString();
          configBuilder.secureRequest(secureRequest);
          break;
        default:
          in.skipValue();
          break;
      }
    }
    in.endObject();

    return configBuilder.build();
  }

  private void writeProfiles(JsonWriter out, List<DeviceProfile> profiles) throws IOException {
    out.name(CloudConstants.Configs.PROFILES_PARAMETER);
    if (profiles == null) {
      out.nullValue();
      return;
    }

    out.beginArray();
    for (DeviceProfile profile : profiles) {
      out.value(profile.name());
    }
    out.endArray();
  }

  private void writePackets(JsonWriter out, List<PacketType> packets) throws IOException {
    out.name(CloudConstants.Configs.PACKETS_PARAMETER);
    if (packets == null) {
      out.nullValue();
      return;
    }

    out.beginArray();
    for (PacketType packet : packets) {
      out.value(packet.name());
    }
    out.endArray();
  }

  List<DeviceProfile> readProfiles(JsonReader in) throws IOException {
    in.beginArray();
    List<DeviceProfile> profiles = new ArrayList<>();
    while (in.hasNext()) {
      profiles.add(DeviceProfile.valueOf(in.nextString()));
    }
    in.endArray();
    return profiles;
  }

  List<PacketType> readPackets(JsonReader in) throws IOException {
    in.beginArray();
    List<PacketType> packets = new ArrayList<>();
    while (in.hasNext()) {
      packets.add(PacketType.valueOf(in.nextString()));
    }
    in.endArray();
    return packets;
  }
}
