package com.kontakt.sdk.android.ble.spec;

import java.util.UUID;

/**
 * Service Model describes service propagated by kontakt.io beacon after
 * connecting to it.
 * For more information refer to kontakt.io Beacon specification - version 2.0.
 *
 * @see <a href="http://docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon Datasheet - version 2.0.</a>
 */
public enum KontaktDeviceService {
  /**
   * Generic Access Service model .
   */
  GENERIC_ACCESS("00001800-0000-1000-8000-00805f9b34fb", "Generic Access"),
  /**
   * Generic Attribute Service model.
   */
  GENERIC_ATTRIBUTE("00001801-0000-1000-8000-00805f9b34fb", "Generic Attribute"),
  /**
   * Beacon Information Service model.
   */
  DEVICE_INFORMATION("0000180a-0000-1000-8000-00805f9b34fb", "Device Information"),
  /**
   * Beacon Measured Power Service model.
   */
  TX_POWER("00001804-0000-1000-8000-00805f9b34fb", "TxPower"),
  /**
   * Beacon Battery Service Model.
   */
  BATTERY_LEVEL("0000180f-0000-1000-8000-00805f9b34fb", "Battery Service"),
  /**
   * Beacon Proximity Service Model.
   */
  PROXIMITY("a1ea8110-0e1b-d4a1-b840-63f88c8da1ea", "Proximity Service"),
  /**
   * Beacon Timers Service Model.
   */
  TIMERS("a1ea8120-0e1b-d4a1-b840-63f88c8da1ea", "Timers"),
  /**
   * Beacon Control Service Model.
   */
  CONTROL("a1ea8130-0e1b-d4a1-b840-63f88c8da1ea", "Control Service"),
  /**
   * The Firmware Upgrade Service.
   */
  DFU_SERVICE("00001530-1212-efde-1523-785feabcd123", "Firmware upgrade Service"),
  /**
   * The Sensors service.
   */
  SENSORS_SERVICE("a1ea8210-0e1b-d4a1-b840-63f88c8da1ea", "Sensors service"),

  /**
   * The Secure config service introduced in 4.0 firmware
   */
  SECURE_CONFIGURATION_SERVICE("A1EA81F0-0E1B-D4A1-B840-63F88C8DA1EA", "Secure config service");

  private final UUID id;

  private final String name;

  /**
   * Instantiates a new Service record.
   *
   * @param uuidString the Service Id string
   * @param name the Service name
   */
  KontaktDeviceService(final String uuidString, String name) {
    this.name = name;
    this.id = UUID.fromString(uuidString);
  }

  /**
   * Gets Service id.
   *
   * @return the service id
   */
  public UUID getId() {
    return id;
  }

  /**
   * Gets Service name.
   *
   * @return the name
   */
  public String getName() {
    return name;
  }

  /**
   * Returns Service Model by given Service Id. If no service is found, then
   * IllegalArgumentException is thrown.
   *
   * @param id the id
   * @return the service model
   */
  public static KontaktDeviceService valueOf(final UUID id) {
    for (final KontaktDeviceService value : values()) {
      if (value.getId().equals(id)) {
        return value;
      }
    }
    return null;
  }
}
