package com.kontakt.sdk.android.ble.service;

import android.app.Service;
import android.content.Intent;
import android.os.Binder;
import android.os.Handler;
import android.os.IBinder;
import android.os.Message;
import android.os.Messenger;
import com.kontakt.sdk.android.ble.configuration.scan.ScanContext;
import com.kontakt.sdk.android.ble.manager.internal.InternalProximityManager;
import com.kontakt.sdk.android.ble.manager.listeners.InternalProximityListener;
import com.kontakt.sdk.android.common.interfaces.SDKSupplier;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.util.Closeables;
import java.io.Closeable;
import java.io.IOException;

/**
 * Proximity service executes requests scheduled by {@link InternalProximityManager}.
 */
public class ProximityService extends Service implements ScanContextAccessor {

  private static final String TAG = ProximityService.class.getSimpleName();

  /**
   * Message code informing backing service that BLE devices scan has started.
   */
  public static final int MESSAGE_SCAN_STARTED = 1;

  /**
   * Message code informing backing service that BLE devices scan has stopped.
   */
  public static final int MESSAGE_SCAN_STOPPED = 2;

  /**
   * Message code informing backing service that BLE devices scan has restarted.
   */
  public static final int MESSAGE_RESTART_SCAN = 3;

  /**
   * Message code informing backing service to intialize scan.
   */
  public static final int MESSAGE_INITIALIZE_SCAN = 4;

  /**
   * Message code informing backing service to finish scan.
   */
  public static final int MESSAGE_FINISH_SCAN = 5;

  /**
   * Message code informing about attaching monitoring listener.
   * Attaching listeners should take place while {@link InternalProximityManager}
   * is scanning.
   */
  public static final int MESSAGE_ATTACH_MONITORING_LISTENER = 6;

  /**
   * Message code informing about detaching monitoring listener.
   */
  public static final int MESSAGE_DETACH_MONITORING_LISTENER = 7;

  /**
   * Message code informing service that scan has finished.
   */
  public static final int MESSAGE_WORK_FINISHED = 8;

  /**
   * The response message notifying that {@link InternalProximityManager}'s request was handled successfully.
   */
  public static final int MESSAGE_SERVICE_RESPONSE_OK = 201;

  private final ServiceScanConfiguration configuration = new ServiceScanConfiguration();

  private Messenger serviceMessenger;
  private ServiceBinder serviceBinder;
  private Handler messagingHandler;
  private ScanCompat scanCompat;

  @Override
  public void onCreate() {
    super.onCreate();
    messagingHandler = new MessagingHandler(this);
    serviceMessenger = new Messenger(messagingHandler);
    serviceBinder = new ServiceBinder(serviceMessenger);
  }

  @Override
  public int onStartCommand(Intent intent, int flags, int startId) {
    super.onStartCommand(intent, flags, startId);
    return START_NOT_STICKY;
  }

  @Override
  public IBinder onBind(Intent intent) {
    return serviceBinder;
  }

  @Override
  public void onDestroy() {
    super.onDestroy();
    configuration.clear();
    messagingHandler = null;
    serviceMessenger = null;
  }

  /**
   * Gets scan context connected to proximity manager
   *
   * @param proximityManagerId the proximity manager id
   * @return the scan context
   */
  @Override
  public ScanContext getScanContext(final int proximityManagerId) {
    return configuration.getScanContext(proximityManagerId);
  }

  /**
   * Gets internal messaging handler.
   *
   * @return the messaging handler
   */
  public final Handler getMessagingHandler() {
    return messagingHandler;
  }

  /**
   * Gets Service scan configuration.
   *
   * @return the configuration
   */
  ServiceScanConfiguration getConfiguration() {
    return configuration;
  }

  private void onStartScan(final ScanContext scanContext, final boolean forceConfigurationRebuild, final int proximityManagerId) {

    final boolean isNewScanContextRequested = (scanContext != configuration.getScanContext(proximityManagerId)) && !isScanning(proximityManagerId);
    createScanCompat();

    if (forceConfigurationRebuild || isNewScanContextRequested) {
      onStopScan(proximityManagerId);
      final ScanConfiguration previousConfiguration = configuration.getScanConfiguration(proximityManagerId);
      ScanConfiguration newConfiguration = scanCompat.createScanConfiguration(previousConfiguration.getScanCallback(), scanContext, serviceMessenger);
      ensureClosed(previousConfiguration);
      ForceScanScheduler forceScanScheduler = scanCompat.createForceScanScheduler(newConfiguration);
      ScanController scanController = scanCompat.createScanController(newConfiguration, forceScanScheduler);
      ServiceScanConfiguration.Item item = new ServiceScanConfiguration.Item(scanContext, newConfiguration, forceScanScheduler, scanController);
      configuration.add(proximityManagerId, item);
    }

    configuration.getScanController(proximityManagerId).start();
    scanCompat.onScanStart(configuration.getScanConfiguration(proximityManagerId));
    updateState(proximityManagerId, State.SCANNING);
  }

  private void onStopScan(final int proximityManagerId) {
    if (isScanning(proximityManagerId)) {
      createScanCompat();
      configuration.getScanController(proximityManagerId).stop();

      scanCompat.onScanStop(configuration.getScanConfiguration(proximityManagerId).getScanCallback());

      updateState(proximityManagerId, State.IDLE);
    } else {
      Logger.d(TAG + ": Stop Ranging method requested but BeaconService is not is Ranging state.");
    }
  }

  private void onAttachListener(final int proximityManagerId, final InternalProximityListener proximityListener) {
    configuration.addListener(proximityManagerId, proximityListener);
  }

  private void onDetachListener(final int proximityManagerId, final InternalProximityListener proximityListener) {
    configuration.removeListener(proximityManagerId, proximityListener);
  }

  /**
   * Provides information whether BeaconService is in ranging.
   *
   * @return the boolean
   */
  protected boolean isScanning(final int proximityManagerId) {
    return configuration.getState(proximityManagerId) == State.SCANNING;
  }

  /**
   * Updates BeaconService state.
   *
   * @param proximityManagerId the proximity manager id
   * @param newState the new state
   */
  protected void updateState(final int proximityManagerId, final State newState) {
    configuration.updateState(proximityManagerId, newState);
  }

  private void onCleanUp(final int proximityManagerId) {
    final ServiceScanConfiguration.Item item = configuration.remove(proximityManagerId);
    ensureClosed(item.scanConfiguration);

    ForceScanScheduler forceScanScheduler = item.forceScanScheduler;

    if (forceScanScheduler != null) {
      forceScanScheduler.finish();
    }

    item.scanController.stop();
  }

  private void createScanCompat() {
    if(scanCompat == null) {
      scanCompat = ScanCompatFactory.createScanCompat();
    }
  }

  /**
   * The type Messaging handler.
   */
  private static void ensureClosed(final Closeable closeable) {
    try {
      Closeables.close(closeable, true);
    } catch (IOException ignored) {
    }
  }

  public static class ServiceBinder extends Binder implements SDKSupplier<Messenger> {
    private final Messenger serviceMessenger;

    private ServiceBinder(Messenger serviceMessenger) {
      this.serviceMessenger = serviceMessenger;
    }

    @Override
    public Messenger get() {
      return serviceMessenger;
    }
  }

  @SuppressWarnings("unchecked")
  private static class MessagingHandler extends Handler {

    private ProximityService service;

    private MessagingHandler(ProximityService abstractBluetoothDeviceService) {
      super();
      service = abstractBluetoothDeviceService;
    }

    @Override
    public void handleMessage(Message msg) {
      try {
        final int proximityManagerId = msg.arg1;
        switch (msg.what) {
          case MESSAGE_SCAN_STARTED:
            Logger.d(TAG + ": Message received - scan started");
            break;
          case MESSAGE_SCAN_STOPPED:
            Logger.d(TAG + ": Message received - scan stopped");
            break;
          case MESSAGE_INITIALIZE_SCAN:
            Logger.d(TAG + ": Message received - start ranging");
            service.onStartScan((ScanContext) msg.obj, true, proximityManagerId);
            msg.replyTo.send(Message.obtain(null, msg.what, MESSAGE_SERVICE_RESPONSE_OK, 0));
            break;
          case MESSAGE_FINISH_SCAN:
            Logger.d(TAG + ": Message received - stop ranging");
            service.onStopScan(proximityManagerId);
            msg.replyTo.send(Message.obtain(null, msg.what, MESSAGE_SERVICE_RESPONSE_OK, 0));
            break;
          case MESSAGE_RESTART_SCAN:
            Logger.d(TAG + ": Message received - restart ranging");
            service.onStopScan(proximityManagerId);
            service.onStartScan((ScanContext) msg.obj, true, proximityManagerId);
            msg.replyTo.send(Message.obtain(null, msg.what, MESSAGE_SERVICE_RESPONSE_OK, 0));
            break;
          case MESSAGE_WORK_FINISHED:
            Logger.d(TAG + ": Message received - work finished");
            service.onCleanUp(proximityManagerId);
            service.stopSelf();
            msg.replyTo.send(Message.obtain(null, msg.what, MESSAGE_SERVICE_RESPONSE_OK, 0));
            break;
          case MESSAGE_ATTACH_MONITORING_LISTENER:
            Logger.d(TAG + ": Attaching listener");
            service.onAttachListener(proximityManagerId, (InternalProximityListener) msg.obj);
            break;
          case MESSAGE_DETACH_MONITORING_LISTENER:
            Logger.d(TAG + ": Detaching listener");
            service.onDetachListener(proximityManagerId, (InternalProximityListener) msg.obj);
            break;
          default:
            throw new IllegalStateException("Unsupported message code: " + msg.what);
        }
      } catch (Exception ignored) {
      }
    }
  }

  enum State {
    IDLE,
    SCANNING
  }
}
