/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kontakt.sdk.android.ble.math;

import com.kontakt.sdk.android.common.util.SDKPreconditions;

/**
 * Abstract base class for all implementations of the
 * {@link UnivariateStatistic} interface.
 *
 * Provides a default implementation of evaluate(double[]),
 * delegating to evaluate(double[], int, int) in the natural way.
 *
 * Also includes a test method that performs generic parameter
 * validation for the evaluate methods.
 */
abstract class AbstractUnivariateStatistic implements UnivariateStatistic {

  /** Stored data. */
  private double[] storedData;

  /**
   * Set the data array.
   * <p>
   * The stored value is a copy of the parameter array, not the array itself.
   *
   *
   * @param values data array to store (may be null to remove stored data)
   * @see #evaluate()
   */
  public void setData(final double[] values) {
    storedData = (values == null) ? null : values.clone();
  }

  /**
   * Get a copy of the stored data array.
   *
   * @return copy of the stored data array (may be null)
   */
  public double[] getData() {
    return (storedData == null) ? null : storedData.clone();
  }

  /**
   * Get a reference to the stored data array.
   *
   * @return reference to the stored data array (may be null)
   */
  protected double[] getDataRef() {
    return storedData;
  }

  /**
   * Set the data array.  The input array is copied, not referenced.
   *
   * @param values data array to store
   * @param begin the index of the first element to include
   * @param length the number of elements to include
   * are not valid
   * @see #evaluate()
   */
  public void setData(final double[] values, final int begin, final int length) {

    SDKPreconditions.checkState(values != null);

    SDKPreconditions.checkState(begin >= 0);

    SDKPreconditions.checkState(length >= 0);

    SDKPreconditions.checkState(begin + length <= values.length);

    storedData = new double[length];
    System.arraycopy(values, begin, storedData, 0, length);
  }

  /**
   * Returns the result of evaluating the statistic over the stored data.
   * <p>
   * The stored array is the one which was set by previous calls to {@link #setData(double[])}.
   *
   *
   * @return the value of the statistic applied to the stored data
   * @throws IllegalStateException if the stored data array is null
   */
  public double evaluate() {
    return evaluate(storedData);
  }

  /**
   * {@inheritDoc}
   */
  public double evaluate(final double[] values) {
    test(values, 0, 0);
    return evaluate(values, 0, values.length);
  }

  /**
   * {@inheritDoc}
   */
  public abstract double evaluate(final double[] values, final int begin, final int length);

  /**
   * This method is used by evaluate(double[], int, int) methods
   * to verify that the input parameters designate a subarray of positive length.
   * <p>
   * <ul>
   * <li>returns true iff the parameters designate a subarray of
   * positive length</li>
   * <li>throws MathIllegalArgumentException if the array is null or
   * or the indices are invalid</li>
   * <li>returns false if the array is non-null, but
   * length is 0.</li>
   * </ul>
   *
   * @param values the input array
   * @param begin index of the first array element to include
   * @param length the number of elements to include
   * @return true if the parameters are valid and designate a subarray of positive length
   */
  protected boolean test(final double[] values, final int begin, final int length) {
    return verifyValues(values, begin, length, false);
  }

  /**
   * This method is used by evaluate(double[], int, int) methods
   * to verify that the input parameters designate a subarray of positive length.
   * <p>
   * <ul>
   * <li>returns true iff the parameters designate a subarray of
   * non-negative length</li>
   * <li>throws IllegalArgumentException if the array is null or
   * or the indices are invalid</li>
   * <li>returns false if the array is non-null, but
   * length is 0 unless allowEmpty is true</li>
   * </ul>
   *
   * @param values the input array
   * @param begin index of the first array element to include
   * @param length the number of elements to include
   * @param allowEmpty if true then zero length arrays are allowed
   * @return true if the parameters are valid
   * @throws NullPointerException or IllegalStateException
   * if the indices are invalid or the array is null
   */
  protected boolean test(final double[] values, final int begin, final int length, final boolean allowEmpty) {
    return verifyValues(values, begin, length, allowEmpty);
  }

  /**
   * This method is used by evaluate(double[], double[], int, int) methods
   * to verify that the begin and length parameters designate a subarray of positive length
   * and the weights are all non-negative, non-NaN, finite, and not all zero.
   * <p>
   * <ul>
   * <li>returns true iff the parameters designate a subarray of
   * positive length and the weights array contains legitimate values.</li>
   * <li>throws IllegalArgumentException if any of the following are true:
   * <ul><li>the values array is null</li>
   * <li>the weights array is null</li>
   * <li>the weights array does not have the same length as the values array</li>
   * <li>the weights array contains one or more infinite values</li>
   * <li>the weights array contains one or more NaN values</li>
   * <li>the weights array contains negative values</li>
   * <li>the start and length arguments do not determine a valid array</li></ul>
   * </li>
   * <li>returns false if the array is non-null, but
   * length is 0.</li>
   * </ul>
   *
   * @param values the input array
   * @param weights the weights array
   * @param begin index of the first array element to include
   * @param length the number of elements to include
   * @return true if the parameters are valid and designate a subarray of positive length
   * @throws NullPointerException or java.lang.IllegalStateException if
   * the indices are invalid or the array is null
   */
  protected boolean test(final double[] values, final double[] weights, final int begin, final int length) {
    return verifyValues(values, weights, begin, length, false);
  }

  /**
   * This method is used
   * to verify that the begin and length parameters designate a subarray of positive length
   * and the weights are all non-negative, non-NaN, finite, and not all zero.
   * <ul>
   * <li>returns true iff the parameters designate a subarray of
   * non-negative length and the weights array contains legitimate values.</li>
   * <li>throws MathIllegalArgumentException if any of the following are true:
   * <ul><li>the values array is null</li>
   * <li>the weights array is null</li>
   * <li>the weights array does not have the same length as the values array</li>
   * <li>the weights array contains one or more infinite values</li>
   * <li>the weights array contains one or more NaN values</li>
   * <li>the weights array contains negative values</li>
   * <li>the start and length arguments do not determine a valid array</li></ul>
   * </li>
   * <li>returns false if the array is non-null, but
   * length is 0 unless allowEmpty is true.</li>
   * </ul>
   *
   * @param values the input array.
   * @param weights the weights array.
   * @param begin index of the first array element to include.
   * @param length the number of elements to include.
   * @param allowEmpty if {@code true} than allow zero length arrays to pass.
   * @return {@code true} if the parameters are valid.
   * @throws NullPointerException if either of the arrays are null
   * @throws IllegalStateException if the array indices are not valid,
   * the weights array contains NaN, infinite or negative elements, or there
   * are no positive weights.
   */
  public static boolean verifyValues(final double[] values, final double[] weights, final int begin, final int length, final boolean allowEmpty) {

    SDKPreconditions.checkNotNull(weights);
    SDKPreconditions.checkNotNull(values);

    SDKPreconditions.checkState(weights.length == values.length);

    boolean containsPositiveWeight = false;
    for (int i = begin; i < begin + length; i++) {
      final double weight = weights[i];

      SDKPreconditions.checkState(!Double.isNaN(weight));

      SDKPreconditions.checkState(!Double.isInfinite(weight));

      SDKPreconditions.checkState(weight >= 0);

      if (!containsPositiveWeight && weight > 0.0) {
        containsPositiveWeight = true;
      }
    }

    SDKPreconditions.checkState(containsPositiveWeight);

    return verifyValues(values, begin, length, allowEmpty);
  }

  /**
   * This method is used by evaluate(double[], double[], int, int) methods
   * to verify that the begin and length parameters designate a subarray of positive length
   * and the weights are all non-negative, non-NaN, finite, and not all zero.
   * <p>
   * <ul>
   * <li>returns true iff the parameters designate a subarray of
   * non-negative length and the weights array contains legitimate values.</li>
   * <li>throws MathIllegalArgumentException if any of the following are true:
   * <ul><li>the values array is null</li>
   * <li>the weights array is null</li>
   * <li>the weights array does not have the same length as the values array</li>
   * <li>the weights array contains one or more infinite values</li>
   * <li>the weights array contains one or more NaN values</li>
   * <li>the weights array contains negative values</li>
   * <li>the start and length arguments do not determine a valid array</li></ul>
   * </li>
   * <li>returns false if the array is non-null, but
   * length is 0 unless allowEmpty is true.</li>
   * </ul>
   *
   * @param values the input array.
   * @param weights the weights array.
   * @param begin index of the first array element to include.
   * @param length the number of elements to include.
   * @param allowEmpty if {@code true} than allow zero length arrays to pass.
   * @return {@code true} if the parameters are valid.
   * @throws NullPointerException if either of the arrays are null
   * @throws IllegalStateException if the array indices are not valid,
   * the weights array contains NaN, infinite or negative elements, or there
   * are no positive weights.
   */
  protected boolean test(final double[] values, final double[] weights, final int begin, final int length, final boolean allowEmpty) {

    return verifyValues(values, weights, begin, length, allowEmpty);
  }

  public static boolean verifyValues(final double[] values, final int begin, final int length, final boolean allowEmpty) {

    SDKPreconditions.checkNotNull(values);

    SDKPreconditions.checkState(begin >= 0);

    SDKPreconditions.checkState(length >= 0);

    SDKPreconditions.checkState(begin + length <= values.length);

    if (length == 0 && !allowEmpty) {
      return false;
    }

    return true;
  }
}

