package com.kontakt.sdk.android.ble.manager.internal;

import android.Manifest;
import android.annotation.TargetApi;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.Build;
import android.os.Handler;
import android.os.IBinder;
import android.os.Message;
import android.os.Messenger;
import android.os.RemoteException;
import com.kontakt.sdk.android.ble.cache.FutureShufflesCache;
import com.kontakt.sdk.android.ble.configuration.InternalProximityManagerConfiguration;
import com.kontakt.sdk.android.ble.configuration.scan.ScanContext;
import com.kontakt.sdk.android.ble.connection.OnServiceReadyListener;
import com.kontakt.sdk.android.ble.exception.ScanError;
import com.kontakt.sdk.android.ble.manager.listeners.InternalProximityListener;
import com.kontakt.sdk.android.ble.manager.listeners.KontaktProximityListener;
import com.kontakt.sdk.android.ble.manager.service.AbstractServiceConnector;
import com.kontakt.sdk.android.ble.monitoring.EventCollector;
import com.kontakt.sdk.android.ble.monitoring.IEventCollector;
import com.kontakt.sdk.android.ble.service.ProximityService;
import com.kontakt.sdk.android.cloud.IKontaktCloud;
import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.common.interfaces.SDKSupplier;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

public class InternalProximityManager extends AbstractServiceConnector {

  private static final String TAG = InternalProximityManager.class.getSimpleName();

  private static final String[] PERMISSIONS = new String[] {
      Manifest.permission.BLUETOOTH, Manifest.permission.BLUETOOTH_ADMIN, Manifest.permission.INTERNET,
  };
  private static final String[] PERMISSIONS_MARSHMALLOW =
      new String[] { Manifest.permission.ACCESS_FINE_LOCATION, Manifest.permission.ACCESS_COARSE_LOCATION };

  private final int id;
  private final IKontaktCloud kontaktCloud;
  private final ShuffledSpacesManager shuffledSpacesManager;
  private FutureShufflesCache futureShufflesCache;
  private IEventCollector eventCollector;
  private Context context;
  private ServiceConnection serviceConnection;
  private Messenger serviceMessenger;
  private Messenger managerMessenger;
  private KontaktProximityListener kontaktProximityListener;
  private boolean isScanning;

  public InternalProximityManager(Context context) {
    this(context, KontaktCloud.newInstance(), InternalProximityManagerConfiguration.DEFAULT);
  }

  public InternalProximityManager(final Context context, IKontaktCloud kontaktCloud, InternalProximityManagerConfiguration configuration) {
    super(context, PERMISSIONS, PERMISSIONS_MARSHMALLOW);
    this.context = context.getApplicationContext();
    this.kontaktCloud = kontaktCloud;
    this.managerMessenger = new Messenger(new ManagerHandler(this));
    this.id = System.identityHashCode(this);
    this.futureShufflesCache = new FutureShufflesCache(context, kontaktCloud, configuration);
    this.eventCollector = new EventCollector(kontaktCloud, configuration);
    this.shuffledSpacesManager = new ShuffledSpacesManager(kontaktCloud);
  }

  public int getId() {
    return id;
  }

  @TargetApi(Build.VERSION_CODES.ICE_CREAM_SANDWICH_MR1)
  @Override
  public synchronized void connect(final OnServiceReadyListener onServiceReadyListener) {
    SDKPreconditions.checkNotNull(onServiceReadyListener, "OnServiceBoundListener is null.");
    checkPermissions();

    if (isConnected()) {
      onServiceReadyListener.onServiceReady();
      Logger.d("InternalProximityManager already connected to BeaconService.");
      return;
    }

    serviceConnection = new ServiceConnection() {
      @Override
      public void onServiceConnected(ComponentName name, IBinder serviceBinder) {
        final SDKSupplier<Messenger> messengerSupplier = (SDKSupplier<Messenger>) serviceBinder;
        serviceMessenger = messengerSupplier.get();
        onServiceReadyListener.onServiceReady();
        Logger.d(TAG + ": Beacon Service connected.");
      }

      @Override
      public void onServiceDisconnected(ComponentName name) {
        Logger.e(TAG + ": disconnected from Beacon Service");
      }
    };
    bindService();
  }

  @Override
  public synchronized void disconnect() {
    if (!isConnected()) {
      Logger.d(TAG + ": BeaconManager already disconnected.");
      return;
    } else if (isScanning()) {
      finishScan();
    }

    try {
      if (serviceConnection != null) {
        final Message message = createMessage(ProximityService.MESSAGE_WORK_FINISHED, null);
        serviceMessenger.send(message);
        context.unbindService(serviceConnection);
        serviceConnection = null;
        serviceMessenger = null;
      }
      super.disconnect();
    } catch (RemoteException e) {
      Logger.e(TAG + ": unexpected exception thrown while disconnecting from Beacon Service ", e);
      throw new IllegalStateException(e);
    }

    eventCollector.stop();
    eventCollector.clear();
    futureShufflesCache.unregisterProximityManager(getId());
    futureShufflesCache.clearBuffers();
    shuffledSpacesManager.clearCache();
    shuffledSpacesManager.onDestroy();
    kontaktProximityListener = null;
  }

  @Override
  public synchronized boolean isConnected() {
    return serviceConnection != null && serviceMessenger != null;
  }

  public synchronized void initializeScan(final ScanContext scanContext, InternalProximityManagerConfiguration configuration,
      final InternalProximityListener proximityListener) {
    SDKPreconditions.checkNotNull(scanContext, "ScanContext cannot be null");
    SDKPreconditions.checkNotNull(configuration, "InternalProximityManagerConfiguration cannot be null");
    SDKPreconditions.checkNotNull(proximityListener, "InternalProximityListener cannot be null");
    updateConfiguration(configuration);
    resolveShuffledSpaces(scanContext, proximityListener, false);
  }

  public synchronized void finishScan() {
    if (!isConnected()) {
      Logger.d("BeaconManger not connected");
      return;
    }

    if (!isScanning()) {
      Logger.d("BeaconManager is not scanning");
      return;
    }

    if (kontaktProximityListener != null) {
      kontaktProximityListener.onScanStop();
    }
    detachListener();
    final Message message = createMessage(ProximityService.MESSAGE_FINISH_SCAN, null);
    sendMessage(message);
  }

  public synchronized void restartScan(final ScanContext scanContext, InternalProximityManagerConfiguration configuration,
      final InternalProximityListener proximityListener) {
    SDKPreconditions.checkNotNull(scanContext, "ScanContext cannot be null");
    SDKPreconditions.checkNotNull(proximityListener, "InternalProximityListener cannot be null");
    updateConfiguration(configuration);
    resolveShuffledSpaces(scanContext, proximityListener, true);
  }

  public synchronized boolean isScanning() {
    return isScanning;
  }

  public synchronized void clearCache() {
    futureShufflesCache.clear();
    eventCollector.clear();
  }

  public synchronized void clearBuffers() {
    futureShufflesCache.clearBuffers();
    eventCollector.clear();
  }

  private synchronized void attachListener(final InternalProximityListener proximityListener) {
    SDKPreconditions.checkNotNull(proximityListener, "Proximity listener is null");
    if (kontaktProximityListener == null) {
      kontaktProximityListener = new KontaktProximityListener(getId(), proximityListener, futureShufflesCache, eventCollector);
      futureShufflesCache.registerProximityManager(getId());
      futureShufflesCache.addProximityListener(kontaktProximityListener);
    }
    final Message message = createMessage(ProximityService.MESSAGE_ATTACH_MONITORING_LISTENER, kontaktProximityListener);
    sendMessage(message);
  }

  private synchronized void detachListener() {
    if (kontaktProximityListener == null) {
      return;
    }
    futureShufflesCache.removeProximityListener(kontaktProximityListener);
    final Message message = createMessage(ProximityService.MESSAGE_DETACH_MONITORING_LISTENER, kontaktProximityListener);
    kontaktProximityListener = null;
    sendMessage(message);
  }

  private void bindService() {
    final Intent serviceIntent = new Intent(context, ProximityService.class);
    final boolean isBindRequestSent = context.bindService(serviceIntent, serviceConnection, Context.BIND_AUTO_CREATE);
    if (!isBindRequestSent) {
      final String serviceName = ProximityService.class.getSimpleName();
      throw new RuntimeException(
          String.format("Could not connect to %s. Please check if the %s is registered in AndroidManifest.xml", serviceName, serviceName));
    }
  }

  private void updateConfiguration(InternalProximityManagerConfiguration configuration) {
    eventCollector.updateConfiguration(configuration);
    if (kontaktProximityListener != null) {
      futureShufflesCache.removeProximityListener(kontaktProximityListener);
    }
    futureShufflesCache.unregisterProximityManager(getId());
    futureShufflesCache = new FutureShufflesCache(context, kontaktCloud, configuration);
  }

  private void resolveShuffledSpaces(ScanContext scanContext, final InternalProximityListener proximityListener, final boolean restartScan) {
    shuffledSpacesManager.resolve(scanContext, new ShuffledSpacesManager.OnSpacesResolvedListener() {
      @Override
      public void onSpacesResolved(ScanContext scanContext) {
        startScanIfConnected(scanContext, restartScan, proximityListener);
      }

      @Override
      public void onError(ScanError exception) {
        proximityListener.onScanError(exception);
      }
    });
  }

  private void startScanIfConnected(final ScanContext scanContext, boolean restartScan, InternalProximityListener proximityListener) {
    if (isScanning() && !restartScan) {
      Logger.d(TAG + ": BeaconManager is already scanning");
      return;
    }
    futureShufflesCache.registerProximityManager(getId());
    eventCollector.start();
    int messageCode = restartScan ? ProximityService.MESSAGE_RESTART_SCAN : ProximityService.MESSAGE_INITIALIZE_SCAN;
    final Message message = createMessage(messageCode, scanContext);
    sendMessage(message);
    attachListener(proximityListener);
    proximityListener.onScanStart();
  }

  private boolean sendMessage(final Message message) {
    if (isConnected()) {
      try {
        serviceMessenger.send(message);
        return true;
      } catch (RemoteException e) {
        return false;
      }
    } else {
      Logger.i("BeaconManager already disconnected");
    }

    return false;
  }

  private Message createMessage(final int messageCode, final Object obj) {
    final Message message = Message.obtain(null, messageCode, id, -1, obj);
    message.replyTo = managerMessenger;
    return message;
  }

  private static class ManagerHandler extends Handler {

    private final InternalProximityManager manager;

    private ManagerHandler(InternalProximityManager manager) {
      this.manager = manager;
    }

    @Override
    public void handleMessage(Message msg) {
      switch (msg.what) {
        case ProximityService.MESSAGE_INITIALIZE_SCAN:
        case ProximityService.MESSAGE_RESTART_SCAN:
          manager.isScanning = (msg.arg1 == ProximityService.MESSAGE_SERVICE_RESPONSE_OK);
          break;

        case ProximityService.MESSAGE_FINISH_SCAN:
        case ProximityService.MESSAGE_WORK_FINISHED:
          manager.isScanning = false;
          break;

        default:
          throw new IllegalArgumentException("Unsupported response code: " + msg.what);
      }
    }
  }
}
