package com.kontakt.sdk.android.ble.manager.internal;

import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceEvent;
import com.kontakt.sdk.android.ble.discovery.eddystone.EddystoneDeviceEvent;
import com.kontakt.sdk.android.ble.discovery.ibeacon.IBeaconDeviceEvent;
import com.kontakt.sdk.android.ble.exception.ScanError;
import com.kontakt.sdk.android.ble.manager.listeners.EddystoneListener;
import com.kontakt.sdk.android.ble.manager.listeners.IBeaconListener;
import com.kontakt.sdk.android.ble.manager.listeners.InternalProximityListener;
import com.kontakt.sdk.android.ble.manager.listeners.ScanStatusListener;
import com.kontakt.sdk.android.ble.manager.listeners.SpaceListener;
import com.kontakt.sdk.android.common.profile.IBeaconDevice;
import com.kontakt.sdk.android.common.profile.IEddystoneDevice;
import java.util.List;

/**
 * Simple class that helps with invoking proper callbacks based on {@link BluetoothDeviceEvent} type.
 */
public class EventObserver implements InternalProximityListener {

  private final ScanStatusListener scanStatusListener;
  private final SpaceListener spaceListener;
  private final EddystoneListener eddystoneListener;
  private final IBeaconListener iBeaconListener;

  public EventObserver(EddystoneListener eddystoneListener, IBeaconListener iBeaconListener, ScanStatusListener scanStatusListener, SpaceListener spaceListener) {
    this.iBeaconListener = iBeaconListener;
    this.eddystoneListener = eddystoneListener;
    this.scanStatusListener = scanStatusListener;
    this.spaceListener = spaceListener;
  }

  @Override
  public void onScanStart() {
    if (scanStatusListener != null) {
      scanStatusListener.onScanStart();
    }
  }

  @Override
  public void onScanStop() {
    if (scanStatusListener != null) {
      scanStatusListener.onScanStop();
    }
  }

  @Override
  public void onMonitoringCycleStart() {
    if(scanStatusListener != null) {
      scanStatusListener.onMonitoringCycleStart();
    }
  }

  @Override
  public void onMonitoringCycleStop() {
    if(scanStatusListener != null) {
      scanStatusListener.onMonitoringCycleStop();
    }
  }

  @Override
  public void onScanError(ScanError exception) {
    if (scanStatusListener != null) {
      scanStatusListener.onScanError(exception);
    }
  }

  @Override
  public void onEvent(BluetoothDeviceEvent event) {
    if (eddystoneListener != null && event instanceof EddystoneDeviceEvent) {
      onEddystoneEvent((EddystoneDeviceEvent) event);
    } else if (iBeaconListener != null && event instanceof IBeaconDeviceEvent) {
      onIBeaconEvent((IBeaconDeviceEvent) event);
    }
  }

  private void onEddystoneEvent(EddystoneDeviceEvent event) {
    switch (event.getEventType()) {
      case DEVICE_DISCOVERED:
      case DEVICES_UPDATE:
      case DEVICE_LOST:
        invokeDeviceCallbacks(event);
        break;
      case SPACE_ENTERED:
      case SPACE_ABANDONED:
        invokeSpaceCallbacks(event);
        break;
    }
  }

  private void onIBeaconEvent(IBeaconDeviceEvent event) {
    switch (event.getEventType()) {
      case DEVICE_DISCOVERED:
      case DEVICES_UPDATE:
      case DEVICE_LOST:
        invokeDeviceCallbacks(event);
        break;
      case SPACE_ENTERED:
      case SPACE_ABANDONED:
        invokeSpaceCallbacks(event);
        break;
    }
  }

  private void invokeDeviceCallbacks(EddystoneDeviceEvent event) {
    List<IEddystoneDevice> deviceList = event.getDeviceList();
    switch (event.getEventType()) {
      case DEVICE_DISCOVERED:
        eddystoneListener.onEddystoneDiscovered(deviceList.get(0), event.getNamespace());
        break;
      case DEVICES_UPDATE:
        eddystoneListener.onEddystonesUpdated(deviceList, event.getNamespace());
        break;
      case DEVICE_LOST:
        eddystoneListener.onEddystoneLost(deviceList.get(0), event.getNamespace());
        break;
    }
  }

  private void invokeDeviceCallbacks(IBeaconDeviceEvent event) {
    List<IBeaconDevice> deviceList = event.getDeviceList();
    switch (event.getEventType()) {
      case DEVICE_DISCOVERED:
        iBeaconListener.onIBeaconDiscovered(deviceList.get(0), event.getRegion());
        break;
      case DEVICES_UPDATE:
        iBeaconListener.onIBeaconsUpdated(deviceList, event.getRegion());
        break;
      case DEVICE_LOST:
        iBeaconListener.onIBeaconLost(deviceList.get(0), event.getRegion());
        break;
    }
  }

  private void invokeSpaceCallbacks(EddystoneDeviceEvent event) {
    if (spaceListener == null) {
      return;
    }
    switch (event.getEventType()) {
      case SPACE_ENTERED:
        spaceListener.onNamespaceEntered(event.getNamespace());
        break;
      case SPACE_ABANDONED:
        spaceListener.onNamespaceAbandoned(event.getNamespace());
        break;
    }
  }

  private void invokeSpaceCallbacks(IBeaconDeviceEvent event) {
    if (spaceListener == null) {
      return;
    }
    switch (event.getEventType()) {
      case SPACE_ENTERED:
        spaceListener.onRegionEntered(event.getRegion());
        break;
      case SPACE_ABANDONED:
        spaceListener.onRegionAbandoned(event.getRegion());
        break;
    }
  }
}
