package com.kontakt.sdk.android.ble.manager.configuration;

import com.kontakt.sdk.android.ble.configuration.ActivityCheckConfiguration;
import com.kontakt.sdk.android.ble.configuration.ForceScanConfiguration;
import com.kontakt.sdk.android.ble.configuration.InternalProximityManagerConfiguration;
import com.kontakt.sdk.android.ble.configuration.ScanPeriod;
import com.kontakt.sdk.android.ble.configuration.scan.EddystoneScanContext;
import com.kontakt.sdk.android.ble.configuration.scan.IBeaconScanContext;
import com.kontakt.sdk.android.ble.configuration.scan.ScanContext;
import com.kontakt.sdk.android.ble.configuration.scan.ScanMode;
import com.kontakt.sdk.android.ble.filter.eddystone.TLMFilter;
import com.kontakt.sdk.android.ble.filter.eddystone.UIDFilter;
import com.kontakt.sdk.android.ble.filter.eddystone.URLFilter;
import com.kontakt.sdk.android.ble.filter.ibeacon.IBeaconFilter;
import com.kontakt.sdk.android.ble.rssi.RssiCalculator;
import com.kontakt.sdk.android.ble.spec.EddystoneFrameType;
import com.kontakt.sdk.android.common.profile.IBeaconRegion;
import com.kontakt.sdk.android.common.profile.IEddystoneNamespace;
import java.util.Collection;
import java.util.Collections;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkArgument;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

/**
 * This class is responsible for ranging/monitoring configuration.
 */
public class Configuration implements GeneralConfigurator, SpacesConfigurator, FiltersConfigurator {

  private ScanPeriod scanPeriod;
  private ScanMode scanMode;
  private ForceScanConfiguration forceScanConfiguration;
  private ActivityCheckConfiguration activityCheckConfiguration;
  private RssiCalculator rssiCalculator;
  private Collection<EddystoneFrameType> eddystoneFrameTypes;

  private String cacheFileName;
  private boolean monitoringEnabled;
  private int monitoringSyncIntervalSeconds;
  private int resolveShuffledInterval;
  private long deviceUpdateCallbackInterval;

  private Collection<IBeaconRegion> iBeaconRegions;
  private Collection<IEddystoneNamespace> eddystoneNamespaces;
  private Collection<IBeaconFilter> iBeaconFilters;
  private Collection<UIDFilter> uidFilters;
  private Collection<URLFilter> urlFilters;
  private Collection<TLMFilter> tlmFilters;

  public Configuration(ScanContext scanContext, InternalProximityManagerConfiguration proximityManagerConfiguration) {
    checkNotNull(scanContext);
    checkNotNull(proximityManagerConfiguration);

    this.scanPeriod = scanContext.getScanPeriod();
    this.forceScanConfiguration = scanContext.getForceScanConfiguration();
    this.activityCheckConfiguration = scanContext.getActivityCheckConfiguration();
    this.scanMode = scanContext.getScanMode();
    this.deviceUpdateCallbackInterval = scanContext.getDevicesUpdateCallbackInterval();
    this.rssiCalculator = scanContext.getRssiCalculator();
    this.eddystoneFrameTypes = Collections.unmodifiableCollection(scanContext.getEddystoneScanContext().getTriggerFrameTypes());

    this.cacheFileName = proximityManagerConfiguration.getCacheFileName();
    this.monitoringEnabled = proximityManagerConfiguration.isMonitoringEnabled();
    this.monitoringSyncIntervalSeconds = proximityManagerConfiguration.getMonitoringSyncInterval();
    this.resolveShuffledInterval = proximityManagerConfiguration.getResolveInterval();

    this.iBeaconRegions = Collections.unmodifiableCollection(scanContext.getIBeaconScanContext().getSpaces());
    this.eddystoneNamespaces = Collections.unmodifiableCollection(scanContext.getEddystoneScanContext().getSpaces());
    this.iBeaconFilters = Collections.unmodifiableCollection(scanContext.getIBeaconScanContext().getFilters());
    this.uidFilters = Collections.unmodifiableCollection(scanContext.getEddystoneScanContext().getUIDFilters());
    this.urlFilters = Collections.unmodifiableCollection(scanContext.getEddystoneScanContext().getURLFilters());
    this.tlmFilters = Collections.unmodifiableCollection(scanContext.getEddystoneScanContext().getTLMFilters());
  }

  public ScanContext createScanContext() {
    return new ScanContext.Builder().setScanPeriod(scanPeriod)
        .setScanMode(scanMode)
        .setDevicesUpdateCallbackInterval(deviceUpdateCallbackInterval)
        .setForceScanConfiguration(forceScanConfiguration)
        .setActivityCheckConfiguration(activityCheckConfiguration)
        .setRssiCalculator(rssiCalculator)
        .setIBeaconScanContext(new IBeaconScanContext.Builder()
            .setIBeaconRegions(iBeaconRegions)
            .setIBeaconFilters(iBeaconFilters)
            .build())
        .setEddystoneScanContext(new EddystoneScanContext.Builder()
            .setEddystoneNamespaces(eddystoneNamespaces)
            .setTriggerFrameTypes(eddystoneFrameTypes)
            .setUIDFilters(uidFilters)
            .setURLFilters(urlFilters)
            .setTLMFilters(tlmFilters)
            .build())
        .build();
  }

  public InternalProximityManagerConfiguration createKontaktManagerConfiguration() {
    return new InternalProximityManagerConfiguration.Builder().setMonitoringEnabled(monitoringEnabled)
        .setMonitoringSyncInterval(monitoringSyncIntervalSeconds)
        .setCacheFileName(cacheFileName)
        .setResolveInterval(resolveShuffledInterval)
        .build();
  }

  @Override
  public GeneralConfigurator scanPeriod(ScanPeriod scanPeriod) {
    this.scanPeriod = checkNotNull(scanPeriod);
    return this;
  }

  @Override
  public GeneralConfigurator scanMode(ScanMode scanMode) {
    this.scanMode = checkNotNull(scanMode);
    return this;
  }

  @Override
  public GeneralConfigurator forceScanConfiguration(ForceScanConfiguration forceScanConfiguration) {
    this.forceScanConfiguration = checkNotNull(forceScanConfiguration);
    return this;
  }

  @Override
  public GeneralConfigurator activityCheckConfiguration(ActivityCheckConfiguration activityCheckConfiguration) {
    this.activityCheckConfiguration = checkNotNull(activityCheckConfiguration);
    return this;
  }

  @Override
  public GeneralConfigurator cacheFileName(String name) {
    this.cacheFileName = checkNotNull(name);
    return this;
  }

  @Override
  public GeneralConfigurator monitoringEnabled(boolean enabled) {
    this.monitoringEnabled = checkNotNull(enabled);
    return this;
  }

  @Override
  public GeneralConfigurator monitoringSyncInterval(int intervalInSeconds) {
    checkArgument(intervalInSeconds > 0);
    this.monitoringSyncIntervalSeconds = intervalInSeconds;
    return this;
  }

  @Override
  public GeneralConfigurator resolveShuffledInterval(int intervalInSeconds) {
    checkArgument(intervalInSeconds > 0);
    this.resolveShuffledInterval = intervalInSeconds;
    return this;
  }

  @Override
  public GeneralConfigurator deviceUpdateCallbackInterval(int intervalInMillis) {
    checkArgument(intervalInMillis > 0);
    this.deviceUpdateCallbackInterval = intervalInMillis;
    return this;
  }

  @Override
  public GeneralConfigurator rssiCalculator(RssiCalculator rssiCalculator) {
    this.rssiCalculator = checkNotNull(rssiCalculator);
    return this;
  }

  @Override
  public GeneralConfigurator eddystoneFrameTypes(Collection<EddystoneFrameType> eddystoneFrameTypes) {
    checkNotNull(eddystoneFrameTypes);
    this.eddystoneFrameTypes = Collections.unmodifiableCollection(eddystoneFrameTypes);
    return this;
  }

  @Override
  public SpacesConfigurator iBeaconRegion(IBeaconRegion region) {
    checkNotNull(region);
    iBeaconRegions(Collections.singletonList(region));
    return this;
  }

  @Override
  public SpacesConfigurator iBeaconRegions(Collection<IBeaconRegion> regions) {
    checkNotNull(regions);
    this.iBeaconRegions = Collections.unmodifiableCollection(regions);
    return this;
  }

  @Override
  public SpacesConfigurator eddystoneNamespace(IEddystoneNamespace namespace) {
    checkNotNull(namespace);
    eddystoneNamespaces(Collections.singletonList(namespace));
    return this;
  }

  @Override
  public SpacesConfigurator eddystoneNamespaces(Collection<IEddystoneNamespace> namespaces) {
    checkNotNull(namespaces);
    this.eddystoneNamespaces = Collections.unmodifiableCollection(namespaces);
    return this;
  }

  @Override
  public FiltersConfigurator iBeaconFilter(IBeaconFilter filter) {
    checkNotNull(filter);
    iBeaconFilters(Collections.singletonList(filter));
    return this;
  }

  @Override
  public FiltersConfigurator iBeaconFilters(Collection<IBeaconFilter> filters) {
    checkNotNull(filters);
    this.iBeaconFilters = Collections.unmodifiableCollection(filters);
    return this;
  }

  @Override
  public FiltersConfigurator eddystoneUidFilter(UIDFilter filter) {
    checkNotNull(filter);
    eddystoneUidFilters(Collections.singletonList(filter));
    return this;
  }

  @Override
  public FiltersConfigurator eddystoneUidFilters(Collection<UIDFilter> filters) {
    checkNotNull(filters);
    this.uidFilters = Collections.unmodifiableCollection(filters);
    return this;
  }

  @Override
  public FiltersConfigurator eddystoneUrlFilter(URLFilter filter) {
    checkNotNull(filter);
    eddystoneUrlFilters(Collections.singletonList(filter));
    return this;
  }

  @Override
  public FiltersConfigurator eddystoneUrlFilters(Collection<URLFilter> filters) {
    checkNotNull(filters);
    this.urlFilters = Collections.unmodifiableCollection(filters);
    return this;
  }

  @Override
  public FiltersConfigurator eddystoneTlmFilter(TLMFilter filter) {
    checkNotNull(filter);
    eddystoneTlmFilters(Collections.singletonList(filter));
    return this;
  }

  @Override
  public FiltersConfigurator eddystoneTlmFilters(Collection<TLMFilter> filters) {
    checkNotNull(filters);
    this.tlmFilters = Collections.unmodifiableCollection(filters);
    return this;
  }
}
