package com.kontakt.sdk.android.ble.discovery.ibeacon;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothDevice;
import android.os.Build;
import android.util.SparseArray;
import com.kontakt.sdk.android.ble.configuration.ActivityCheckConfiguration;
import com.kontakt.sdk.android.ble.configuration.scan.IBeaconScanContext;
import com.kontakt.sdk.android.ble.configuration.scan.ScanContext;
import com.kontakt.sdk.android.ble.device.BeaconDevice;
import com.kontakt.sdk.android.ble.discovery.AbstractBluetoothDeviceDiscoverer;
import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceEvent;
import com.kontakt.sdk.android.ble.discovery.DiscoveryContract;
import com.kontakt.sdk.android.ble.discovery.EventType;
import com.kontakt.sdk.android.ble.discovery.Validator;
import com.kontakt.sdk.android.ble.util.ReplacingArrayList;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.profile.IBeaconDevice;
import com.kontakt.sdk.android.common.profile.IBeaconRegion;
import java.util.ArrayList;

/**
 * Performs IBeacon devices discovery and sends specific events via
 * {@link DiscoveryContract}. The discoverer adjusts its behaviour according to
 * the specified {@link IBeaconScanContext} and {@link ActivityCheckConfiguration}.
 */
@TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
public class IBeaconDiscoverer extends AbstractBluetoothDeviceDiscoverer<IBeaconRegion, IBeaconDevice> {

  private final IBeaconAdvertisingDataController advertisingDataController;
  private final Validator<IBeaconAdvertisingPacket, IBeaconRegion> validator;

  public IBeaconDiscoverer(ScanContext scanContext, DiscoveryContract discoveryContract) {
    super(discoveryContract, scanContext.getIBeaconScanContext().getDistanceSort(), scanContext.getIBeaconScanContext().getEventTypes(),
        scanContext.getActivityCheckConfiguration(), scanContext.getIBeaconScanContext().getSpaces(), scanContext.getDevicesUpdateCallbackInterval());

    this.advertisingDataController = new IBeaconAdvertisingDataController(scanContext);
    this.validator = new RegionValidator();
  }

  @Override
  protected BluetoothDeviceEvent createEvent(EventType eventType, IBeaconRegion iBeaconRegion, ArrayList<IBeaconDevice> deviceList) {
    return new IBeaconDeviceEvent(eventType, iBeaconRegion, deviceList);
  }

  @Override
  protected void onBeforeDeviceLost(IBeaconDevice device) {
    advertisingDataController.clearRssiCalculation(device.hashCode());
  }

  @Override
  public boolean performDiscovery(BluetoothDevice device, int rssi, byte[] scanRecord) {
    if (!advertisingDataController.isEnabled()) {
      Logger.d("AdvertisingDataController not enabled.");
      return PROFILE_UNRECOGNIZED;
    }

    SparseArray<byte[]> parsedScan = IBeaconAdvertisingDataController.parseScanRecord(scanRecord);
    if (parsedScan == null) {
      return PROFILE_UNRECOGNIZED;
    }

    //Parsing advertising
    IBeaconAdvertisingPacket advertisingPacket = advertisingDataController.getOrCreateAdvertisingPackage(device, rssi, parsedScan);
    if (advertisingPacket == null) {
      return PROFILE_UNRECOGNIZED;
    }

    final IBeaconRegion iBeaconRegion = extractRegion(advertisingPacket);
    if (iBeaconRegion == null) {
      return PROFILE_RECOGNIZED_NO_BELONGING_SPACE_FOUND;
    }

    if (advertisingPacket.getProximity() != Proximity.UNKNOWN && advertisingDataController.filter(advertisingPacket)) {

      notifySpacePresent(iBeaconRegion.hashCode(), System.currentTimeMillis());

      ReplacingArrayList<IBeaconDevice> iBeaconList = getDevicesInSpace(iBeaconRegion);

      if (iBeaconList == null) {
        iBeaconList = new ReplacingArrayList<IBeaconDevice>();

        insertDevicesIntoSpace(iBeaconRegion, iBeaconList);

        onSpaceEnteredEvent(iBeaconRegion);

        return true;
      }

      BeaconDevice discoveredDevice = new BeaconDevice(advertisingPacket);

      if (iBeaconList.addOrReplace(discoveredDevice)) {
        onDeviceDiscoveredEvent(iBeaconRegion, discoveredDevice);
      } else {
        sortIfEnabled(iBeaconList);

        onDevicesUpdatedEvent(iBeaconRegion, iBeaconList);
      }

      return true;
    }

    return PROFILE_RECOGNIZED_FILTERING_NOT_PASSED;
  }

  @Override
  public void disable() {
    advertisingDataController.disable();
  }

  /**
   * Extracts region from advertising package provided that the region was specified
   * during BeaconManager configuration. If no BeaconRegion has been applied to {@link IBeaconScanContext}
   * then always {@link com.kontakt.sdk.android.ble.device.BeaconRegion#EVERYWHERE} is returned.
   *
   * @param advertisingPacket the advertising package
   * @return the Matched region or null if no match is found
   */
  private IBeaconRegion extractRegion(final IBeaconAdvertisingPacket advertisingPacket) {
    for (final IBeaconRegion iBeaconRegion : getSpaceSet()) {
      if (validator.isValid(advertisingPacket, iBeaconRegion)) {
        return iBeaconRegion;
      }
    }
    return null;
  }
}
