package com.kontakt.sdk.android.ble.discovery.ibeacon;

import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.ble.device.BeaconRegion;
import com.kontakt.sdk.android.ble.discovery.AbstractBluetoothDeviceEvent;
import com.kontakt.sdk.android.ble.discovery.EventType;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.IBeaconDevice;
import com.kontakt.sdk.android.common.profile.IBeaconRegion;
import java.util.ArrayList;
import java.util.List;

/**
 * This event is notified once the {@link com.kontakt.sdk.android.ble.configuration.scan.IBeaconScanContext}
 * has been included to the global {@link com.kontakt.sdk.android.ble.configuration.scan.ScanContext}
 * and the InternalProximityManager scans within this global Scan Context.
 */
public class IBeaconDeviceEvent extends AbstractBluetoothDeviceEvent {

    private static final ArrayList<IBeaconDevice> EMPTY_DEVICE_LIST = new ArrayList<IBeaconDevice>(0);

    /**
     * The Parcelable constant CREATOR.
     */
    public static final Parcelable.Creator<IBeaconDeviceEvent>
        CREATOR = new Parcelable.Creator<IBeaconDeviceEvent>() {
        @Override
        public IBeaconDeviceEvent createFromParcel(Parcel source) {
            Bundle bundle = source.readBundle(getClass().getClassLoader());

            BeaconRegion iBeaconRegion = bundle.getParcelable("region");
            EventType eventType = (EventType) bundle.getSerializable(PARCELABLE_EVENT_TYPE);
            ArrayList<IBeaconDevice> deviceList = bundle.getParcelableArrayList(PARCELABLE_DEVICE_LIST);
            long timestamp = bundle.getLong(PARCELABLE_TIMESTAMP);

            return new IBeaconDeviceEvent(eventType, iBeaconRegion, deviceList, timestamp);
        }

        @Override
        public IBeaconDeviceEvent[] newArray(int size) {
            return new IBeaconDeviceEvent[size];
        }
    };

    /**
     * Creates IBeaconDeviceEvent from other event with provided list of devices
     *
     * @param event   the event
     * @param devices the devices
     * @return new event
     */
    public static IBeaconDeviceEvent of(IBeaconDeviceEvent event, List<IBeaconDevice> devices) {
        return new IBeaconDeviceEvent(event.getEventType(), event.getRegion(), devices, event.getTimestamp());
    }

    /**
     * Creates IBeaconDeviceEvent from resolved {@link IBeaconDevice} and {@link IBeaconRegion}
     *
     * @param device    the real device
     * @param region    the region
     * @param timestamp the timestamp
     * @return new event
     */
    public static IBeaconDeviceEvent createNewDiscovered(IBeaconDevice device, IBeaconRegion region, long timestamp) {
        List<IBeaconDevice> discovered = new ArrayList<IBeaconDevice>();
        discovered.add(device);
        return new IBeaconDeviceEvent(EventType.DEVICE_DISCOVERED, region, discovered, timestamp);
    }

    private final IBeaconRegion region;

    /**
     * Instantiates a new iBeacon device event.
     *
     * @param eventType the event type
     * @param iBeaconRegion    the iBeaconRegion
     */
    public IBeaconDeviceEvent(EventType eventType,
                              BeaconRegion iBeaconRegion) {
        this(eventType, iBeaconRegion, EMPTY_DEVICE_LIST);
    }

    /**
     * Instantiates a new iBeacon device event.
     *
     * @param eventType         the event type
     * @param iBeaconRegion            the iBeaconRegion
     * @param iBeaconDeviceList the iBeacon device list
     */
    public IBeaconDeviceEvent(
            EventType eventType,
            IBeaconRegion iBeaconRegion,
            List<IBeaconDevice> iBeaconDeviceList) {
        this(eventType,
                iBeaconRegion,
                iBeaconDeviceList,
                System.currentTimeMillis());
    }

    private IBeaconDeviceEvent(
            EventType eventType,
            IBeaconRegion region,
            List<IBeaconDevice> iBeaconDevices,
            long timestamp) {
        super(eventType,
                DeviceProfile.IBEACON,
                iBeaconDevices,
                timestamp);

        this.region = region;
    }

    @Override
    public List<IBeaconDevice> getDeviceList() {
        return (List<IBeaconDevice>) super.getDeviceList();
    }

    /**
     * Gets iBeacon region.
     *
     * @return the region
     */
    public IBeaconRegion getRegion() {
        return region;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        Bundle bundle = new Bundle(getClass().getClassLoader());
        bundle.putSerializable(PARCELABLE_EVENT_TYPE, eventType);
        bundle.putLong(PARCELABLE_TIMESTAMP, timestamp);
        bundle.putParcelableArrayList(PARCELABLE_DEVICE_LIST, (ArrayList<? extends Parcelable>) deviceList);
        bundle.putParcelable("region", region);

        dest.writeBundle(bundle);
    }
}
