package com.kontakt.sdk.android.ble.discovery.ibeacon;

import android.bluetooth.BluetoothDevice;
import android.util.SparseArray;
import com.kontakt.sdk.android.ble.configuration.ForceScanConfiguration;
import com.kontakt.sdk.android.ble.configuration.scan.ScanContext;
import com.kontakt.sdk.android.ble.discovery.DiscoveryUtils;
import com.kontakt.sdk.android.ble.discovery.ScanResponse;
import com.kontakt.sdk.android.ble.filter.ibeacon.IBeaconFilter;
import com.kontakt.sdk.android.ble.rssi.RssiCalculator;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.LimitedLinkedHashMap;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.UUID;

/**
 * AdvertisingDataController extracts necessary data from propagated Beacon's
 * advertising package frame and describes devices .
 * For more information see Advertising Packet structure and
 * {@link ForceScanConfiguration}.
 *
 * @see <a href="https://kontaktio.zendesk.com/hc/en-gb/articles/201492492-Advertising-packet-structure" target="_blank">kontakt.io knowledge base - Advertising packet structure</a>
 * @see <a href="http://code.google.com/p/android/issues/detail?id=65863" target="_blank">Bug no 65863 - Andoroid Open Source Project</a>
 */
public final class IBeaconAdvertisingDataController {

    /**
     * Specifies byte value of the type of manufacturer-specific data.
     */
    public static final int TYPE_MANUFACTURER_SPECIFIC_DATA = 0xFF;

    public static final int TX_POWER_BLOCK = 0x0a;

    private static final byte[] MANUFACTURER_DATA_IBEACON_PREFIX = new byte[]{0x4C, 0x00, 0x02, 0x15};

    private static final byte[] EMPTY_BYTE_ARRAY = new byte[0];

    private static final int CACHE_SIZE = 20;

    private static final int SERVICE_UUID_MSB = 13;
    private static final int SERVICE_UUID_LSB = -48;
    private static final int MANUFACTURER_BLOCK_LENGTH = 25;
    private static final int COMPANY_ID = -3;
    private static final int COMPANY_ID_BEGIN = 1;
    private static final int PAYLOAD_VERSION_1 = 1;

    private final Map<Integer, AdvertisingPacketImpl> cache;

    private final HashCodeBuilder hashCodeBuilder;
    private AdvertisingPacketImpl.Builder builder;
    private RssiCalculator rssiCalculator;
    private Collection<IBeaconFilter> filters;

    private boolean isEnabled;

    /**
     * AdvertisingDataController handles advertising data
     * ({@link android.bluetooth.BluetoothAdapter.LeScanCallback}).
     *
     * @param scanContext the IBeacon Scan Context
     */
    public IBeaconAdvertisingDataController(final ScanContext scanContext) {
        this.rssiCalculator = scanContext.getRssiCalculator();
        this.filters = scanContext.getIBeaconScanContext().getFilters();
        this.isEnabled = true;
        this.hashCodeBuilder = HashCodeBuilder.init();
        this.builder = new AdvertisingPacketImpl.Builder();
        this.cache = new LimitedLinkedHashMap<Integer, AdvertisingPacketImpl>(CACHE_SIZE);
    }

    /**
     * Parses scan record sent by kontakt.io IBeacon device.
     *
     * @param scanRecord the scan record
     * @return parsed scan record if the scan record matches kontakt.io's iBeacon, null otherwise.
     */
    public static SparseArray<byte[]> parseScanRecord(final byte[] scanRecord) {
        SparseArray<byte[]> frameArray = DiscoveryUtils.extractMetaData(scanRecord);
        final byte[] manufacturerData = frameArray.get(TYPE_MANUFACTURER_SPECIFIC_DATA, EMPTY_BYTE_ARRAY);
        final byte[] serviceData = frameArray.get(ScanResponse.TYPE_SERVICE_DATA, EMPTY_BYTE_ARRAY);

        boolean standard = ConversionUtils.doesArrayBeginWith(manufacturerData, MANUFACTURER_DATA_IBEACON_PREFIX)
                && manufacturerData.length >= 25
                && serviceData.length == 9
                && serviceData[0] == SERVICE_UUID_MSB
                && serviceData[1] == SERVICE_UUID_LSB;

        boolean shuffled = ConversionUtils.doesArrayBeginWith(manufacturerData, MANUFACTURER_DATA_IBEACON_PREFIX)
                && serviceData.length >= 25
                && serviceData[0] == SERVICE_UUID_MSB
                && serviceData[1] == SERVICE_UUID_LSB
                && serviceData[2] == ScanResponse.PAYLOAD_VERSION_1;

        if (standard || shuffled) {
            return frameArray;
        }

        return null;
    }

    /**
     * Creates or retrieves cached advertising package.
     *
     * @param device     the device
     * @param rssi       the rssi
     * @param parsedScan initially parsed scan record
     * @return the or create advertising package
     */
    public IBeaconAdvertisingPacket getOrCreateAdvertisingPackage(final BluetoothDevice device,
                                                                  final int rssi,
                                                                  final SparseArray<byte[]> parsedScan) {

        int deviceHashCode = hashCodeBuilder.append(device.getAddress())
                .append(parsedScan.get(TYPE_MANUFACTURER_SPECIFIC_DATA))
                .build();

        AdvertisingPacketImpl advertisingPackage = cache.get(deviceHashCode);

        if (advertisingPackage != null) {
            update(advertisingPackage, parsedScan, deviceHashCode, rssi);
            return advertisingPackage;
        }

        byte[] manufacturerData = parsedScan.get(IBeaconAdvertisingDataController.TYPE_MANUFACTURER_SPECIFIC_DATA);
        int txPower = manufacturerData[24];
        final double rssiValue = rssiCalculator.calculateRssi(deviceHashCode, rssi);

        byte[] serviceData = parsedScan.get(ScanResponse.TYPE_SERVICE_DATA);
        final ScanResponse scanResponse = ScanResponse.fromScanResponseBytes(serviceData);

        final double distance = DiscoveryUtils.calculateDistance(txPower, rssiValue);
        final UUID proximityUUID = ConversionUtils.toUUID(Arrays.copyOfRange(manufacturerData, 4, 20));

        final int major = ConversionUtils.asInt(Arrays.copyOfRange(manufacturerData, 20, 22));
        final int minor = ConversionUtils.asInt(Arrays.copyOfRange(manufacturerData, 22, 24));

        advertisingPackage = builder.setAdvertisingData(parsedScan)
                .setRssi(rssi)
                .setProximityUUID(proximityUUID)
                .setMajor(major)
                .setMinor(minor)
                .setBeaconUniqueId(scanResponse.getUniqueId())
                .setFirmwareVersion(scanResponse.getFirmwareVersion())
                .setBatteryPercentagePower(scanResponse.getBatteryPower())
                .setShuffled(scanResponse.isShuffled())
                .setAddress(device.getAddress())
                .setName(device.getName())
                .setTxPower(txPower)
                .setDistance(distance)
                .setProximity(Proximity.fromDistance(distance))
                .setTimestamp(System.currentTimeMillis())
                .build();

        cache.put(deviceHashCode, advertisingPackage);
        return advertisingPackage;
    }

    private void update(AdvertisingPacketImpl advertisingPacket, SparseArray<byte[]> advertisingData, int deviceHashCode, int rssi) {
        double rssiValue = rssiCalculator.calculateRssi(deviceHashCode, rssi);
        int txPower = advertisingData.get(IBeaconAdvertisingDataController.TYPE_MANUFACTURER_SPECIFIC_DATA)[24];
        double distance = DiscoveryUtils.calculateDistance(txPower, rssiValue);

        advertisingPacket.setDistance(distance);
        advertisingPacket.setRssi(rssiValue);
        advertisingPacket.setProximity(Proximity.fromDistance(distance));
        advertisingPacket.setTimestamp(System.currentTimeMillis());
    }

    /**
     * Filters Advertising Package with specfied {@link IBeaconFilter}s.
     *
     * @param advertisingPacket the advertising package
     * @return true if there are no filters specified or at least one matches
     */
    public boolean filter(final IBeaconAdvertisingPacket advertisingPacket) {
        if (filters.isEmpty()) {
            return true;
        }

        for (final IBeaconFilter filter : filters) {
            if (filter.apply(advertisingPacket)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Is enabled.
     *
     * @return the boolean
     */
    boolean isEnabled() {
        return isEnabled;
    }

    /**
     * Disable void.
     */
    void disable() {
        if (isEnabled) {
            isEnabled = false;
            cache.clear();
            filters = Collections.emptyList();
            rssiCalculator.clear();
            rssiCalculator = null;
        }
    }

    /**
     * Clear rssi calculation.
     *
     * @param deviceHashCode the device hash code
     */
    void clearRssiCalculation(int deviceHashCode) {
        if (rssiCalculator != null) {
            rssiCalculator.clear(deviceHashCode);
        }
    }
}
