package com.kontakt.sdk.android.ble.discovery.eddystone;

import android.util.SparseArray;
import com.kontakt.sdk.android.common.util.ConversionUtils;

final class URLResolver implements EddystonePropertyResolver<String> {

    private static final SparseArray<String> URL_SCHEME_PREFIXES;

    private static final SparseArray<String> URL_SCHEME_SUFFIXES;

    private static final int SCAN_RECORD_EDDYSTONE_PDU_LENGTH_INDEX = 7;

    static final int SCAN_RECORD_EDDYSTONE_URL_START_INDEX = 13;

    private static final int URL_PREFIX_MASK = 0x000F;

    private static final int URL_SUFFIX_MASK = 0x000F;

    private final StringBuilder urlBuilder = new StringBuilder();

    static {
        URL_SCHEME_PREFIXES = new SparseArray<String>(4);

        URL_SCHEME_PREFIXES.put(0x00, "http://www.");
        URL_SCHEME_PREFIXES.put(0x01, "https://www.");
        URL_SCHEME_PREFIXES.put(0x02, "http://");
        URL_SCHEME_PREFIXES.put(0x03, "https://");

        URL_SCHEME_SUFFIXES = new SparseArray<String>(13);

        URL_SCHEME_SUFFIXES.put(0x01, ".com/");
        URL_SCHEME_SUFFIXES.put(0x02, ".org/");
        URL_SCHEME_SUFFIXES.put(0x03, ".edu/");
        URL_SCHEME_SUFFIXES.put(0x04, ".net/");
        URL_SCHEME_SUFFIXES.put(0x05, ".info/");
        URL_SCHEME_SUFFIXES.put(0x06, ".biz/");
        URL_SCHEME_SUFFIXES.put(0x07, ".gov/");
        URL_SCHEME_SUFFIXES.put(0x08, ".org");
        URL_SCHEME_SUFFIXES.put(0x09, ".edu");
        URL_SCHEME_SUFFIXES.put(0x0a, ".net");
        URL_SCHEME_SUFFIXES.put(0x0b, ".info");
        URL_SCHEME_SUFFIXES.put(0x0c, ".biz");
        URL_SCHEME_SUFFIXES.put(0x0d, ".gov");
    }

    @Override
    public byte[] extract(byte[] scanRecord) {
        final int urlLength = scanRecord[SCAN_RECORD_EDDYSTONE_PDU_LENGTH_INDEX] - 5;

        return ConversionUtils.extractPayload(
                scanRecord,
                SCAN_RECORD_EDDYSTONE_URL_START_INDEX,
                urlLength);
    }

    @Override
    public String parse(byte[] source) {
        if(source == null) {
            return null;
        }

        urlBuilder.setLength(0);

        final int prefix = source[0] & URL_PREFIX_MASK;

        urlBuilder.append(URL_SCHEME_PREFIXES.get(prefix));

        for(int i = 1, size = source.length - 1; i < size; i++) {
            urlBuilder.append((char)source[i]);
        }

        final String urlSuffix = URL_SCHEME_SUFFIXES.get(source[source.length - 1] & URL_SUFFIX_MASK);
        if(urlSuffix != null) {
            urlBuilder.append(urlSuffix);
        }


        return urlBuilder.toString();
    }
}
