package com.kontakt.sdk.android.ble.discovery.eddystone;

import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.ble.discovery.AbstractBluetoothDeviceEvent;
import com.kontakt.sdk.android.ble.discovery.EventType;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.IEddystoneDevice;
import com.kontakt.sdk.android.common.profile.IEddystoneNamespace;
import com.kontakt.sdk.android.common.util.Constants;
import java.util.ArrayList;
import java.util.List;

/**
 * This event is notified once the {@link com.kontakt.sdk.android.ble.configuration.scan.EddystoneScanContext}
 * has been included to the global {@link com.kontakt.sdk.android.ble.configuration.scan.ScanContext}
 * and the InternalProximityManager scans within this global Scan Context.
 */
public class EddystoneDeviceEvent extends AbstractBluetoothDeviceEvent {

    /**
     * The parcelable constant CREATOR.
     */
    public static final Parcelable.Creator<EddystoneDeviceEvent>
        CREATOR = new Parcelable.Creator<EddystoneDeviceEvent>() {
        @Override
        public EddystoneDeviceEvent createFromParcel(Parcel source) {
            Bundle bundle = source.readBundle(getClass().getClassLoader());

            EventType eventType = (EventType) bundle.getSerializable(PARCELABLE_EVENT_TYPE);
            IEddystoneNamespace IEddystoneNamespace = bundle.getParcelable(Constants.Eddystone.NAMESPACE);
            ArrayList<IEddystoneDevice> deviceList = bundle.getParcelableArrayList(PARCELABLE_DEVICE_LIST);
            long timestamp = bundle.getLong(PARCELABLE_TIMESTAMP);

            return new EddystoneDeviceEvent(
                    eventType,
                    IEddystoneNamespace,
                    deviceList,
                    timestamp);
        }

        @Override
        public EddystoneDeviceEvent[] newArray(int size) {
            return new EddystoneDeviceEvent[size];
        }
    };

    private final IEddystoneNamespace iEddystoneNamespace;

    /**
     * Instantiates a new Eddystone device event.
     *
     * @param eventType           the event type
     * @param iEddystoneNamespace the i eddystone namespace
     * @param deviceList          the device list
     */
    public EddystoneDeviceEvent(final EventType eventType,
                                final IEddystoneNamespace iEddystoneNamespace,
                                final ArrayList<IEddystoneDevice> deviceList) {
        this(eventType,
                iEddystoneNamespace,
                deviceList,
                System.currentTimeMillis());

    }

    private EddystoneDeviceEvent(final EventType eventType,
                                 final IEddystoneNamespace iEddystoneNamespace,
                                 final List<IEddystoneDevice> deviceList,
                                 final long timestamp) {
        super(eventType,
                DeviceProfile.EDDYSTONE,
                deviceList,
                timestamp);
        this.iEddystoneNamespace = iEddystoneNamespace;
    }

    /**
     * Provides namespace that the Eddystone devices list belongs to.
     *
     * @return the namespace
     */
    public IEddystoneNamespace getNamespace() {
        return iEddystoneNamespace;
    }

    /**
     * Gets device list
     *
     * @return List of IEddystoneDevice
     */
    @Override
    public List<IEddystoneDevice> getDeviceList() {
        return (List<IEddystoneDevice>) super.getDeviceList();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        Bundle bundle = new Bundle(getClass().getClassLoader());
        bundle.putParcelable(Constants.Eddystone.NAMESPACE, iEddystoneNamespace);
        bundle.putSerializable(PARCELABLE_EVENT_TYPE, getEventType());
        bundle.putParcelableArrayList(PARCELABLE_DEVICE_LIST, (ArrayList<? extends Parcelable>) deviceList);
        bundle.putLong(PARCELABLE_TIMESTAMP, timestamp);

        dest.writeBundle(bundle);
    }

    /**
     * Creates EddystoneDeviceEvent from {@link IEddystoneDevice} and {@link IEddystoneNamespace}
     *
     * @param resolvedDevice the real device resolved through cloud
     * @param namespace      the namespace
     * @param timestamp      the timestamp
     * @return eddystoneDeviceEvent
     */
    public static EddystoneDeviceEvent createNewDiscovered(IEddystoneDevice resolvedDevice, IEddystoneNamespace namespace, long timestamp) {
        ArrayList<IEddystoneDevice> eddystoneDevices = new ArrayList<IEddystoneDevice>();
        eddystoneDevices.add(resolvedDevice);
        return new EddystoneDeviceEvent(EventType.DEVICE_DISCOVERED, namespace, eddystoneDevices, timestamp);
    }

    /**
     * Creates EddystoneDeviceEvent from other event with provided device list
     *
     * @param eddystoneDeviceEvent the event
     * @param eddystoneDevices     the devices
     * @return eddystoneDeviceEvent
     */
    public static EddystoneDeviceEvent of(EddystoneDeviceEvent eddystoneDeviceEvent, List<IEddystoneDevice> eddystoneDevices) {
        return new EddystoneDeviceEvent(eddystoneDeviceEvent.getEventType(),
                eddystoneDeviceEvent.getNamespace(),
                eddystoneDevices,
                eddystoneDeviceEvent.getTimestamp());
    }
}
