package com.kontakt.sdk.android.ble.discovery.eddystone;

import android.bluetooth.BluetoothDevice;
import android.util.SparseIntArray;
import com.kontakt.sdk.android.ble.configuration.scan.ScanContext;
import com.kontakt.sdk.android.ble.device.EddystoneDevice;
import com.kontakt.sdk.android.ble.device.EddystoneNamespace;
import com.kontakt.sdk.android.ble.discovery.DiscoveryUtils;
import com.kontakt.sdk.android.ble.discovery.ScanResponse;
import com.kontakt.sdk.android.ble.filter.eddystone.TLMFilter;
import com.kontakt.sdk.android.ble.filter.eddystone.UIDFilter;
import com.kontakt.sdk.android.ble.filter.eddystone.URLFilter;
import com.kontakt.sdk.android.ble.rssi.RssiCalculator;
import com.kontakt.sdk.android.ble.spec.EddystoneFrameType;
import com.kontakt.sdk.android.ble.spec.Telemetry;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import com.kontakt.sdk.android.common.util.EddystoneUtils;
import com.kontakt.sdk.android.common.util.LimitedLinkedHashMap;
import java.util.Collection;
import java.util.Map;

final class EddystoneAdvertisingDataController {

  private static final int HEADER_LENGTH = 12;
  private static final int HEADER_BLE_FLAGS_LENGTH_OF_FLAGS_BLOCK = 0x02;
  private static final int HEADER_BLE_FLAGS_PDU_DATA_TYPE = 0x01;
  private static final int HEADER_BLE_FLAGS_DATA = 0x06;
  private static final int HEADER_SERVICE_UUID_LENGTH_OF_SERVICE_UUID_BLOCK = 0x03;
  private static final int HEADER_SERVICE_UUID_PDU_DATA_TYPE = 0x03;
  private static final byte HEADER_SERVICE_UUID_HIGH_ORDER_VALUE = (byte) 0xAA;
  private static final byte HEADER_SERVICE_UUID_LOW_ORDER_VALUE = (byte) 0xFE;
  private static final int SCAN_RECORD_TX_POWER_INDEX = 12;
  private static final int CACHE_SIZE = 25;

  private static final InstanceIdResolver INSTANCE_ID_RESOLVER = new InstanceIdResolver();
  private static final NamespaceIdResolver NAMESPACE_ID_RESOLVER = new NamespaceIdResolver();
  private static final TLMResolver TLM_RESOLVER = new TLMResolver();
  private static final URLResolver URL_RESOLVER = new URLResolver();

  private static byte[] EDDYSTONE_SPECIFIC_HEADER = new byte[] {
      HEADER_BLE_FLAGS_LENGTH_OF_FLAGS_BLOCK, HEADER_BLE_FLAGS_PDU_DATA_TYPE, HEADER_BLE_FLAGS_DATA, HEADER_SERVICE_UUID_LENGTH_OF_SERVICE_UUID_BLOCK,
      HEADER_SERVICE_UUID_PDU_DATA_TYPE, HEADER_SERVICE_UUID_HIGH_ORDER_VALUE, HEADER_SERVICE_UUID_LOW_ORDER_VALUE
  };

  static boolean isEddystoneSpecificFrame(final byte[] scanRecord) {

    return !(scanRecord == null || scanRecord.length < HEADER_LENGTH) &&
        ConversionUtils.doesArrayBeginWith(scanRecord, EDDYSTONE_SPECIFIC_HEADER) &&
        scanRecord[9] == HEADER_SERVICE_UUID_HIGH_ORDER_VALUE &&
        scanRecord[10] == HEADER_SERVICE_UUID_LOW_ORDER_VALUE;
  }

  private final Map<String, UIDAdvertisingPacket> UID_CACHE;
  private final Map<String, TLMAdvertisingPacket> TLM_CACHE;
  private final Map<String, URLAdvertisingPacket> URL_CACHE;
  private final Map<String, EddystoneDevice> BUILT_DEVICES_CACHE;
  private final Map<String, ScanResponse> SCAN_RESPONSE_CACHE;
  private final SparseIntArray TX_POWER_CACHE;
  private final Collection<UIDFilter> uidFilters;
  private final Collection<TLMFilter> tlmFilters;
  private final Collection<URLFilter> urlFilters;
  private final boolean areUIDFiltersEmpty;
  private final boolean areTLMFiltersEmpty;
  private final boolean areURLFiltersEmpty;
  private final RssiCalculator rssiCalculator;
  private final Collection<EddystoneFrameType> triggerFrameTypes;
  private boolean isEnabled = true;

  EddystoneAdvertisingDataController(final ScanContext scanContext) {
    this.UID_CACHE = new LimitedLinkedHashMap<>(CACHE_SIZE);
    this.TLM_CACHE = new LimitedLinkedHashMap<>(CACHE_SIZE);
    this.URL_CACHE = new LimitedLinkedHashMap<>(CACHE_SIZE);
    this.SCAN_RESPONSE_CACHE = new LimitedLinkedHashMap<>(CACHE_SIZE);
    this.BUILT_DEVICES_CACHE = new LimitedLinkedHashMap<>(CACHE_SIZE);

    this.TX_POWER_CACHE = new SparseIntArray();

    this.uidFilters = scanContext.getEddystoneScanContext().getUIDFilters();
    this.areUIDFiltersEmpty = uidFilters.isEmpty();

    this.tlmFilters = scanContext.getEddystoneScanContext().getTLMFilters();
    this.areTLMFiltersEmpty = tlmFilters.isEmpty();

    this.urlFilters = scanContext.getEddystoneScanContext().getURLFilters();
    this.areURLFiltersEmpty = urlFilters.isEmpty();

    this.rssiCalculator = scanContext.getRssiCalculator();
    this.triggerFrameTypes = scanContext.getEddystoneScanContext().getTriggerFrameTypes();
  }

  void cacheUID(String deviceAddress, byte[] scanResult) {

    updateTxPower(deviceAddress, scanResult[SCAN_RECORD_TX_POWER_INDEX]);

    UIDAdvertisingPacket eddystoneUIDPacket = (UIDAdvertisingPacket) getUIDAdvertisingPacket(deviceAddress);

    boolean isNewUIDPacket = eddystoneUIDPacket == null;

    boolean isNamespaceChanged = !isNewUIDPacket && !ConversionUtils.doesArrayContainSubset(scanResult, eddystoneUIDPacket.getNamespaceIdBytes(),
        NamespaceIdResolver.NAMESPACE_ID_START_INDEX);

    boolean isInstanceChanged = !isNewUIDPacket && !ConversionUtils.doesArrayContainSubset(scanResult, eddystoneUIDPacket.getInstanceIdBytes(),
        InstanceIdResolver.INSTANCE_ID_START_INDEX);

    if (isNewUIDPacket) {
      putUIDAdvertisingPacket(deviceAddress, new UIDAdvertisingPacket(scanResult, getScanResponse(deviceAddress)));
      resetDeviceCache(deviceAddress);
      return;
    }

    if (isNamespaceChanged) {
      eddystoneUIDPacket.swapNamespaceId(scanResult);
      resetDeviceCache(deviceAddress);
    }

    if (isInstanceChanged) {
      eddystoneUIDPacket.swapInstanceId(scanResult);
      resetDeviceCache(deviceAddress);
    }
  }

  void cacheUrl(String deviceAddress, byte[] scanResult) {
    updateTxPower(deviceAddress, scanResult[SCAN_RECORD_TX_POWER_INDEX]);

    URLAdvertisingPacket urlAdvertisingPacket = (URLAdvertisingPacket) getURLAdvertisingPacket(deviceAddress);

    boolean isNewUrlPacket = urlAdvertisingPacket == null;

    boolean isUrlChanged = !isNewUrlPacket && !ConversionUtils.doesArrayContainSubset(scanResult, urlAdvertisingPacket.getUrlBytes(),
        URLResolver.SCAN_RECORD_EDDYSTONE_URL_START_INDEX);

    if (isNewUrlPacket) {
      putURLAdvertisingPacket(deviceAddress, new URLAdvertisingPacket(scanResult));
      return;
    }

    if (isUrlChanged) {
      urlAdvertisingPacket.swap(scanResult);
      resetDeviceCache(deviceAddress);
    }
  }

  void cacheTelemetry(String deviceAddress, byte[] scanResult) {
    TLMAdvertisingPacket tlmAdvertisingPacket = (TLMAdvertisingPacket) getTLMAdvertisingPacket(deviceAddress);

    boolean isNewTLMAdvertisingPacket = tlmAdvertisingPacket == null;

    boolean isTLMChanged = !isNewTLMAdvertisingPacket && !ConversionUtils.doesArrayContainSubset(scanResult, tlmAdvertisingPacket.getTLMBytes(),
        TLMResolver.TLM_START_INDEX);

    if (isNewTLMAdvertisingPacket) {
      putTLMAdvertisingPacket(deviceAddress, new TLMAdvertisingPacket(scanResult));
      return;
    }

    if (isTLMChanged) {
      tlmAdvertisingPacket.swap(scanResult);
      resetDeviceCache(deviceAddress);
    }
  }

  void cacheScanResponse(final String deviceAddress, final byte[] scanResult) {
    if (!SCAN_RESPONSE_CACHE.containsKey(deviceAddress)) {
      ScanResponse scanResponse = ScanResponse.fromScanRecordBytes(scanResult);
      SCAN_RESPONSE_CACHE.put(deviceAddress, scanResponse);
    }
  }

  void disable() {
    isEnabled = false;
  }

  void clearResources() {
    UID_CACHE.clear();
    TLM_CACHE.clear();
    BUILT_DEVICES_CACHE.clear();
    URL_CACHE.clear();
    SCAN_RESPONSE_CACHE.clear();
    TX_POWER_CACHE.clear();
    rssiCalculator.clear();
  }

  void cacheProperty(EddystoneFrameType frameType, String deviceAddress, byte[] scanResult) {
    cacheScanResponse(deviceAddress, scanResult);
    switch (frameType) {
      case UID:
        cacheUID(deviceAddress, scanResult);
        break;
      case URL:
        cacheUrl(deviceAddress, scanResult);
        break;
      case TLM:
        cacheTelemetry(deviceAddress, scanResult);
        break;
      default:
        Logger.e("Unknown Eddystone frame type parsed for device with address: " + deviceAddress);
        break;
    }
  }

  boolean isEnabled() {
    return isEnabled;
  }

  EddystoneDevice getOrCreateDevice(BluetoothDevice bluetoothDevice, int rssi) {

    String deviceAddress = bluetoothDevice.getAddress();

    EddystoneDevice eddystoneDevice = getEddystoneDevice(deviceAddress);

    if (!areTriggerFramesCached(deviceAddress)) {
      return null;
    }

    int txPower = getTxPower(deviceAddress.hashCode());
    double rssiValue = rssiCalculator.calculateRssi(deviceAddress.hashCode(), rssi);
    double distance = DiscoveryUtils.calculateDistance(txPower, rssiValue);
    Proximity proximity = Proximity.fromDistance(distance);

    if (eddystoneDevice == null) {
      String name = bluetoothDevice.getName();
      EddystoneUIDAdvertisingPacket uidPacket = getUIDAdvertisingPacket(deviceAddress);
      EddystoneTLMAdvertisingPacket tlmPacket = getTLMAdvertisingPacket(deviceAddress);
      EddystoneURLAdvertisingPacket urlPacket = getURLAdvertisingPacket(deviceAddress);
      ScanResponse scanResponse = getScanResponse(deviceAddress);

      eddystoneDevice = new EddystoneDevice.Builder().setName(name)
          .setTxPower(txPower)
          .setTLMAdvertisingPacket(tlmPacket)
          .setUIDAdvertisingPacket(uidPacket)
          .setURLAdvertisingPacket(urlPacket)
          .setTxPower(txPower)
          .setDistance(distance)
          .setAddress(deviceAddress)
          .setProximity(proximity)
          .setUniqueId(scanResponse.getUniqueId())
          .setFirmwareVersion(scanResponse.getFirmwareVersion())
          .setBatteryPower(scanResponse.getBatteryPower())
          .setShuffled(scanResponse.isShuffled())
          .setRssi(rssiValue)
          .setTimestamp(System.currentTimeMillis())
          .build();
    } else {
      eddystoneDevice = new EddystoneDevice.Builder()
          .setEddystoneDevice(eddystoneDevice)
          .setDistance(distance)
          .setRssi(rssi)
          .setProximity(proximity)
          .setTimestamp(System.currentTimeMillis())
          .build();
    }
    putEddystoneDevice(deviceAddress, eddystoneDevice);

    return eddystoneDevice;
  }

  boolean areTriggerFramesCached(final String deviceAddress) {
    final boolean isUIDCached = getUIDAdvertisingPacket(deviceAddress) != null;
    if (triggerFrameTypes.contains(EddystoneFrameType.UID) && !isUIDCached) {
      return false;
    }

    final boolean isURLCached = getURLAdvertisingPacket(deviceAddress) != null;
    if (triggerFrameTypes.contains(EddystoneFrameType.URL) && !isURLCached) {
      return false;
    }

    final boolean isTLMCached = getTLMAdvertisingPacket(deviceAddress) != null;
    return !(triggerFrameTypes.contains(EddystoneFrameType.TLM) && !isTLMCached);
  }

  void clearRssiCalculation(int key) {
    rssiCalculator.clear(key);
  }

  boolean filter(final String deviceAddress) {

    if (areUIDFiltersEmpty && areURLFiltersEmpty && areTLMFiltersEmpty) {
      return true;
    }

    final EddystoneUIDAdvertisingPacket uidPacket = getUIDAdvertisingPacket(deviceAddress);

    if (!areUIDFiltersEmpty && uidPacket != null) {
      for (UIDFilter uidFilter : uidFilters) {
        if (uidFilter.apply(uidPacket)) {
          return true;
        }
      }
    }

    final EddystoneURLAdvertisingPacket urlPacket = getURLAdvertisingPacket(deviceAddress);

    if (!areURLFiltersEmpty && urlPacket != null) {
      for (URLFilter urlFilter : urlFilters) {
        if (urlFilter.apply(urlPacket)) {
          return true;
        }
      }
    }

    final EddystoneTLMAdvertisingPacket tlmPacket = getTLMAdvertisingPacket(deviceAddress);

    if (!areTLMFiltersEmpty && tlmPacket != null) {
      for (TLMFilter tlmFilter : tlmFilters) {
        if (tlmFilter.apply(tlmPacket)) {
          return true;
        }
      }
    }

    return false;
  }

  public String getNamespaceIfAllTriggersAreSatisfied(String deviceAddress, EddystoneFrameType frameType) {
    if (areTriggerFramesCached(deviceAddress)) {
      switch (frameType) {
        case UID:
          return getUIDAdvertisingPacket(deviceAddress).getNamespaceId();
        case URL:
        case TLM:
          return EddystoneNamespace.EVERYWHERE.getNamespace();
      }
    }
    return null;
  }

  ScanResponse getScanResponse(String deviceAddress) {
    return SCAN_RESPONSE_CACHE.get(deviceAddress);
  }

  EddystoneUIDAdvertisingPacket getUIDAdvertisingPacket(String deviceAddress) {
    return UID_CACHE.get(deviceAddress);
  }

  EddystoneURLAdvertisingPacket getURLAdvertisingPacket(String deviceAddress) {
    return URL_CACHE.get(deviceAddress);
  }

  EddystoneTLMAdvertisingPacket getTLMAdvertisingPacket(String deviceAddress) {
    return TLM_CACHE.get(deviceAddress);
  }

  EddystoneDevice getEddystoneDevice(String deviceAddress) {
    return BUILT_DEVICES_CACHE.get(deviceAddress);
  }

  int getTxPowerCacheSize() {
    return TX_POWER_CACHE.size();
  }

  int getTxPower(int deviceAddressHashCode) {
    return TX_POWER_CACHE.get(deviceAddressHashCode);
  }

  private void putUIDAdvertisingPacket(String deviceAddress, UIDAdvertisingPacket uidAdvertisingPacket) {
    UID_CACHE.put(deviceAddress, uidAdvertisingPacket);
  }

  private void putURLAdvertisingPacket(String deviceAddress, URLAdvertisingPacket urlAdvertisingPacket) {
    URL_CACHE.put(deviceAddress, urlAdvertisingPacket);
  }

  private void putTLMAdvertisingPacket(String deviceAddress, TLMAdvertisingPacket tlmAdvertisingPacket) {
    TLM_CACHE.put(deviceAddress, tlmAdvertisingPacket);
  }

  private void putEddystoneDevice(String address, EddystoneDevice eddystoneDevice) {
    BUILT_DEVICES_CACHE.put(address, eddystoneDevice);
  }

  private void updateTxPower(final String deviceAddress, final int txPower) {
    TX_POWER_CACHE.put(deviceAddress.hashCode(), txPower);
  }

  private void resetDeviceCache(final String deviceAddress) {
    BUILT_DEVICES_CACHE.remove(deviceAddress);
  }

  private static class TLMAdvertisingPacket implements EddystoneTLMAdvertisingPacket {

    private TLMAdvertisingPacket(final byte[] scanRecord) {
      swap(scanRecord);
    }

    private byte[] tlmBytes;

    private Telemetry telemetry;

    private void swap(final byte[] scanRecord) {
      this.tlmBytes = TLM_RESOLVER.extract(scanRecord);
      this.telemetry = TLM_RESOLVER.parse(this.tlmBytes);
    }

    @Override
    public byte[] getTLMBytes() {
      return tlmBytes;
    }

    @Override
    public int getBatteryVoltage() {
      return telemetry.getBatteryVoltage();
    }

    @Override
    public double getTemperature() {
      return telemetry.getTemperature();
    }

    @Override
    public int getPduCount() {
      return telemetry.getPduCount();
    }

    @Override
    public int getTimeSincePowerUp() {
      return telemetry.getTimeSincePowerUp();
    }

    @Override
    public int getTelemetryVersion() {
      return telemetry.getVersion();
    }
  }

  private static class UIDAdvertisingPacket implements EddystoneUIDAdvertisingPacket {

    private final ScanResponse scanResponse;

    private UIDAdvertisingPacket(final byte[] scanRecord, final ScanResponse scanResponse) {
      swapNamespaceId(scanRecord);
      swapInstanceId(scanRecord);
      this.shuffled = scanResponse.isShuffled();
      this.scanResponse = scanResponse;
    }

    private void swapNamespaceId(final byte[] scanRecord) {
      this.namespaceIdBytes = NAMESPACE_ID_RESOLVER.extract(scanRecord);
      this.namespaceId = NAMESPACE_ID_RESOLVER.parse(this.namespaceIdBytes);
    }

    private void swapInstanceId(final byte[] scanRecord) {
      this.instanceIdBytes = INSTANCE_ID_RESOLVER.extract(scanRecord);
      this.instanceId = INSTANCE_ID_RESOLVER.parse(this.instanceIdBytes);
    }

    private byte[] namespaceIdBytes;

    private String namespaceId;

    private byte[] instanceIdBytes;

    private String instanceId;

    private boolean shuffled;

    @Override
    public byte[] getNamespaceIdBytes() {
      return namespaceIdBytes;
    }

    @Override
    public String getNamespaceId() {
      return namespaceId;
    }

    @Override
    public byte[] getInstanceIdBytes() {
      return instanceIdBytes;
    }

    @Override
    public String getInstanceId() {
      return instanceId;
    }

    @Override
    public String getEddystoneUniqueId() {
      return scanResponse.getUniqueId();
    }

    @Override
    public String getFirmwareVersion() {
      return scanResponse.getFirmwareVersion();
    }

    @Override
    public boolean isShuffled() {
      return shuffled;
    }
  }

  private static class URLAdvertisingPacket implements EddystoneURLAdvertisingPacket {

    private byte[] urlBytes;

    private String url;

    private URLAdvertisingPacket(final byte[] scanRecord) {
      swap(scanRecord);
    }

    private void swap(final byte[] scanRecord) {
      this.urlBytes = URL_RESOLVER.extract(scanRecord);
      this.url = EddystoneUtils.deserializeUrl(urlBytes);
    }

    @Override
    public byte[] getUrlBytes() {
      return urlBytes;
    }

    @Override
    public String getUrl() {
      return url;
    }
  }
}
