package com.kontakt.sdk.android.ble.device;

import android.os.Bundle;
import android.os.Parcel;
import android.text.TextUtils;
import com.kontakt.sdk.android.ble.discovery.eddystone.EddystoneTLMAdvertisingPacket;
import com.kontakt.sdk.android.ble.discovery.eddystone.EddystoneUIDAdvertisingPacket;
import com.kontakt.sdk.android.ble.discovery.eddystone.EddystoneURLAdvertisingPacket;
import com.kontakt.sdk.android.ble.spec.Telemetry;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.model.EddystoneUid;
import com.kontakt.sdk.android.common.model.ResolvedId;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.IEddystoneDevice;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

/**
 * {@link IEddystoneDevice} implementation.
 */
public class EddystoneDevice implements IEddystoneDevice {

  /**
   * Parcelable constant CREATOR.
   */
  public static final Creator<EddystoneDevice> CREATOR = new Creator<EddystoneDevice>() {
    @Override
    public EddystoneDevice createFromParcel(Parcel source) {
      return new EddystoneDevice(source);
    }

    @Override
    public EddystoneDevice[] newArray(int size) {
      return new EddystoneDevice[size];
    }
  };

  private final long timestamp;
  private final String namespaceId;
  private final String instanceId;
  private final int txPower;
  private final String url;
  private final int hashCode;
  private final double distance;
  private final String address;
  private final Proximity proximity;
  private final double rssi;
  private final Telemetry telemetry;
  private final String firmwareVersion;
  private final String uniqueId;
  private final int batteryPower;
  private final String name;
  private byte[] password;
  private final boolean shuffled;

  private EddystoneDevice(Parcel source) {
    Bundle bundle = source.readBundle(getClass().getClassLoader());

    this.name = bundle.getString(Constants.Devices.NAME);
    this.timestamp = bundle.getLong(Constants.TIMESTAMP);
    this.namespaceId = bundle.getString(Constants.Eddystone.NAMESPACE_ID);
    this.instanceId = bundle.getString(Constants.Eddystone.INSTANCE_ID);
    this.txPower = bundle.getInt(Constants.Devices.TX_POWER);
    this.url = bundle.getString(Constants.Eddystone.URL);
    this.distance = bundle.getDouble(Constants.Devices.ACCURACY);
    this.address = bundle.getString(Constants.Devices.ADDRESS);
    this.proximity = (Proximity) bundle.getSerializable(Constants.Devices.PROXIMITY);
    this.rssi = bundle.getDouble(Constants.Devices.RSSI);
    this.telemetry = bundle.getParcelable(Constants.Eddystone.TELEMETRY);
    this.password = bundle.getByteArray(Constants.Devices.PASSWORD);

    this.firmwareVersion = bundle.getString(Constants.FIRMWARE);
    this.uniqueId = bundle.getString(Constants.UNIQUE_ID);
    this.batteryPower = bundle.getInt(Constants.Devices.BATTERY);
    this.shuffled = bundle.getBoolean(Constants.Devices.SHUFFLED);

    this.hashCode = HashCodeBuilder.init().append(namespaceId).append(instanceId).append(txPower).append(url).build();
  }

  private EddystoneDevice(final Builder builder) {
    this.name = builder.name;
    this.timestamp = builder.timestamp;
    this.namespaceId = builder.namespaceId;
    this.instanceId = builder.instanceId;
    this.txPower = builder.txPower;
    this.url = builder.url;
    this.distance = builder.distance;
    this.address = builder.address;
    this.proximity = builder.proximity;
    this.rssi = builder.rssi;
    this.telemetry = builder.telemetry;

    this.firmwareVersion = builder.firmwareVersion;
    this.uniqueId = builder.uniqueId;
    this.batteryPower = builder.batteryPower;
    this.shuffled = builder.shuffled;
    this.hashCode = HashCodeBuilder.init().append(address).build();
  }

  public static Builder builder() {
    return new Builder();
  }

  @Override
  public int hashCode() {
    return hashCode;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (o == null || !(o instanceof EddystoneDevice)) {
      return false;
    }

    final EddystoneDevice device = (EddystoneDevice) o;

    return TextUtils.equals(address, device.address);
  }

  @Override
  public int compareTo(IEddystoneDevice another) {
    if (this == another) {
      return 0;
    }

    final int namespaceComparisonResult = namespaceId.compareTo(another.getNamespaceId());

    if (namespaceComparisonResult == 0) {

      final int instanceComparisonResult = instanceId.compareTo(another.getInstanceId());

      if (instanceComparisonResult == 0) {

        return url.compareTo(another.getUrl());
      } else {
        return instanceComparisonResult;
      }
    } else {
      return namespaceComparisonResult;
    }
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    final Bundle bundle = new Bundle(getClass().getClassLoader());
    bundle.putString(Constants.Eddystone.NAMESPACE_ID, namespaceId);
    bundle.putString(Constants.Eddystone.INSTANCE_ID, instanceId);
    bundle.putString(Constants.Eddystone.URL, url);
    bundle.putInt(Constants.Devices.TX_POWER, txPower);
    bundle.putLong(Constants.TIMESTAMP, timestamp);
    bundle.putDouble(Constants.Devices.ACCURACY, distance);
    bundle.putSerializable(Constants.Devices.PROXIMITY, proximity);
    bundle.putDouble(Constants.Devices.RSSI, rssi);
    bundle.putString(Constants.Devices.ADDRESS, address);
    bundle.putParcelable(Constants.Eddystone.TELEMETRY, telemetry);
    bundle.putByteArray(Constants.Devices.PASSWORD, password);
    bundle.putString(Constants.Devices.NAME, name);

    bundle.putString(Constants.FIRMWARE, firmwareVersion);
    bundle.putString(Constants.UNIQUE_ID, uniqueId);
    bundle.putInt(Constants.Devices.BATTERY, batteryPower);
    bundle.putBoolean(Constants.Devices.SHUFFLED, shuffled);
    dest.writeBundle(bundle);
  }

  @Override
  public double getDistance() {
    return distance;
  }

  @Override
  public long getTimestamp() {
    return timestamp;
  }

  @Override
  public String getAddress() {
    return address;
  }

  @Override
  public Proximity getProximity() {
    return proximity;
  }

  @Override
  public double getRssi() {
    return rssi;
  }

  @Override
  public void setPassword(byte[] password) {
    SDKPreconditions.checkNotNull(password, "Password is null");

    final int passwordLength = password.length;
    this.password = new byte[passwordLength];

    System.arraycopy(password, 0, this.password, 0, passwordLength);
  }

  @Override
  public byte[] getPassword() {
    return password;
  }

  @Override
  public String getFirmwareVersion() {
    return firmwareVersion;
  }

  @Override
  public String getName() {
    return name;
  }

  @Override
  public String getUniqueId() {
    return uniqueId;
  }

  @Override
  public int getBatteryPower() {
    return batteryPower;
  }

  @Override
  public String getNamespaceId() {
    return namespaceId;
  }

  @Override
  public String getInstanceId() {
    return instanceId;
  }

  @Override
  public int getTxPower() {
    return txPower;
  }

  @Override
  public DeviceProfile getProfile() {
    return DeviceProfile.EDDYSTONE;
  }

  @Override
  public String getUrl() {
    return url;
  }

  @Override
  public int getBatteryVoltage() {
    return telemetry.getBatteryVoltage();
  }

  @Override
  public double getTemperature() {
    return telemetry.getTemperature();
  }

  @Override
  public int getPduCount() {
    return telemetry.getPduCount();
  }

  @Override
  public int getTimeSincePowerUp() {
    return telemetry.getTimeSincePowerUp();
  }

  @Override
  public int getTelemetryVersion() {
    return telemetry.getVersion();
  }

  @Override
  public boolean isShuffled() {
    return shuffled;
  }

  public static class Builder {

    private String namespaceId;
    private String instanceId;
    private int txPower;
    private String url;
    private double distance;
    private String address;
    private Proximity proximity;
    private double rssi;
    private Telemetry telemetry;
    private long timestamp;
    private String firmwareVersion;
    private String uniqueId;
    private int batteryPower;
    private String name;
    private boolean shuffled;

    /**
     * Build eddystone device.
     *
     * @return the eddystone device
     */
    public EddystoneDevice build() {
      return new EddystoneDevice(this);
    }

    /**
     * Sets IEddystoneDevice
     *
     * @param eddystoneDevice the eddystone device
     * @return the builder
     */
    public Builder setEddystoneDevice(IEddystoneDevice eddystoneDevice) {
      this.name = eddystoneDevice.getName();
      this.timestamp = eddystoneDevice.getTimestamp();
      this.namespaceId = eddystoneDevice.getNamespaceId();
      this.instanceId = eddystoneDevice.getInstanceId();
      this.txPower = eddystoneDevice.getTxPower();
      this.url = eddystoneDevice.getUrl();
      this.distance = eddystoneDevice.getDistance();
      this.address = eddystoneDevice.getAddress();
      this.proximity = eddystoneDevice.getProximity();
      this.rssi = eddystoneDevice.getRssi();
      this.telemetry = new Telemetry.Builder().setBatteryVoltage(eddystoneDevice.getBatteryVoltage())
          .setTemperature(eddystoneDevice.getTemperature())
          .setPduCount(eddystoneDevice.getPduCount())
          .setTimeSincePowerUp(eddystoneDevice.getTimeSincePowerUp())
          .setVersion(eddystoneDevice.getTelemetryVersion())
          .build();

      this.firmwareVersion = eddystoneDevice.getFirmwareVersion();
      this.uniqueId = eddystoneDevice.getUniqueId();
      this.batteryPower = eddystoneDevice.getBatteryPower();
      this.shuffled = eddystoneDevice.isShuffled();
      return this;
    }

    /**
     * Sets eddystone uid from resolved id
     *
     * @param resolvedId resolved eddystone uid
     * @return the builder
     */
    public Builder setResolvedId(ResolvedId resolvedId) {
      EddystoneUid eddystoneUID = resolvedId.getEddystoneUID();
      this.namespaceId = eddystoneUID.getNamespace();
      this.instanceId = eddystoneUID.getInstanceId();
      this.uniqueId = resolvedId.getUniqueId();
      return this;
    }

    /**
     * Sets uID advertising packet.
     *
     * @param uidPacket the uid packet
     * @return the builder
     */
    public Builder setUIDAdvertisingPacket(EddystoneUIDAdvertisingPacket uidPacket) {
      this.namespaceId = uidPacket != null ? uidPacket.getNamespaceId() : null;
      this.instanceId = uidPacket != null ? uidPacket.getInstanceId() : null;
      return this;
    }

    /**
     * Sets uRL advertising packet.
     *
     * @param urlPacket the url packet
     * @return the builder
     */
    public Builder setURLAdvertisingPacket(EddystoneURLAdvertisingPacket urlPacket) {
      this.url = urlPacket != null ? urlPacket.getUrl() : null;
      return this;
    }

    /**
     * Sets tLM advertising packet.
     *
     * @param tlmPacket the tlm packet
     * @return the builder
     */
    public Builder setTLMAdvertisingPacket(EddystoneTLMAdvertisingPacket tlmPacket) {
      this.telemetry = new Telemetry.Builder().setBatteryVoltage(tlmPacket != null ? tlmPacket.getBatteryVoltage() : 0)
          .setTimeSincePowerUp(tlmPacket != null ? tlmPacket.getTimeSincePowerUp() : 0)
          .setTemperature(tlmPacket != null ? tlmPacket.getTemperature() : 0)
          .setPduCount(tlmPacket != null ? tlmPacket.getPduCount() : 0)
          .setVersion(tlmPacket != null ? tlmPacket.getTelemetryVersion() : -1)
          .build();
      return this;
    }

    /**
     * Sets tx power.
     *
     * @param txPower the tx power
     * @return the builder
     */
    public Builder setTxPower(int txPower) {
      this.txPower = txPower;
      return this;
    }

    /**
     * Sets distance.
     *
     * @param distance the distance
     * @return the builder
     */
    public Builder setDistance(double distance) {
      this.distance = distance;
      return this;
    }

    /**
     * Sets address.
     *
     * @param address the address
     * @return the builder
     */
    public Builder setAddress(String address) {
      this.address = address;
      return this;
    }

    /**
     * Sets proximity.
     *
     * @param proximity the proximity
     * @return the builder
     */
    public Builder setProximity(Proximity proximity) {
      this.proximity = proximity;
      return this;
    }

    /**
     * Sets rssi.
     *
     * @param rssi the rssi
     * @return the builder
     */
    public Builder setRssi(double rssi) {
      this.rssi = rssi;
      return this;
    }

    /**
     * Sets firmware version.
     *
     * @param firmwareVersion the firmware version
     * @return the builder
     */
    public Builder setFirmwareVersion(String firmwareVersion) {
      this.firmwareVersion = firmwareVersion;
      return this;
    }

    /**
     * Sets unique id.
     *
     * @param uniqueId the unique id
     * @return the builder
     */
    public Builder setUniqueId(String uniqueId) {
      this.uniqueId = uniqueId;
      return this;
    }

    /**
     * Sets battery power.
     *
     * @param batteryPower the battery power
     * @return the builder
     */
    public Builder setBatteryPower(int batteryPower) {
      this.batteryPower = batteryPower;
      return this;
    }

    /**
     * Sets name.
     *
     * @param name the name
     * @return the builder
     */
    public Builder setName(String name) {
      this.name = name;
      return this;
    }

    /**
     * Sets boolean value indicating if device is shuffled or not
     *
     * @param shuffled true or false
     * @return the builder
     */
    public Builder setShuffled(boolean shuffled) {
      this.shuffled = shuffled;
      return this;
    }

    /**
     * Sets timestamp
     *
     * @param timestamp in milliseconds
     * @return the builder
     */
    public Builder setTimestamp(long timestamp) {
      this.timestamp = timestamp;
      return this;
    }
  }
}
