package com.kontakt.sdk.android.ble.device;

import android.os.Bundle;
import android.os.Parcel;
import com.kontakt.sdk.android.common.KontaktSDK;
import com.kontakt.sdk.android.common.profile.IBeaconRegion;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.UUID;

/**
 * Object representing BeaconRegion
 * Proximity UUID is requeired for setting region
 * <p>
 * If you want to scan only for Proximity UUID, you can set major to {@link BeaconRegion#ANY_MAJOR} and minor to {@link BeaconRegion#ANY_MINOR}
 *
 * <p>
 * Example usage:
 * <pre>
 *      BeaconRegion region = new BeaconRegion.Builder()
 *          .setProximity(KontaktSDK.DEFAULT_KONTAKT_BEACON_PROXIMITY_UUID)
 *          .setMajor(ANY_MAJOR)
 *          .setMinor(ANY_MINOR)
 *          .setShuffled(false)
 *          .setIdentifier("Everywhere")
 *          .build();
 *     </pre>
 */
public class BeaconRegion implements IBeaconRegion {

  public static final Creator<BeaconRegion> CREATOR = new Creator<BeaconRegion>() {
    @Override
    public BeaconRegion createFromParcel(Parcel source) {
      Bundle bundle = source.readBundle(getClass().getClassLoader());

      return new Builder().setProximity(((UUID) bundle.getSerializable(Constants.Region.UUID)))
          .setMajor(bundle.getInt(Constants.Region.MAJOR))
          .setMinor(bundle.getInt(Constants.Region.MINOR))
          .setShuffled(bundle.getBoolean(Constants.Region.SHUFFLED))
          .setIdentifier(bundle.getString(Constants.Region.IDENTIFIER))
          .build();
    }

    @Override
    public BeaconRegion[] newArray(int size) {
      return new BeaconRegion[size];
    }
  };

  /**
   * Value marking than any major is acceptable
   */
  public static final int ANY_MAJOR = -1;
  /**
   * Value marking than any minor is acceptable
   */
  public static final int ANY_MINOR = -1;

  /**
   * Default region for scan. Accepts all beacons discovered during scan
   */
  public static final BeaconRegion EVERYWHERE = new BeaconRegion.Builder().setProximity(KontaktSDK.DEFAULT_KONTAKT_BEACON_PROXIMITY_UUID)
      .setMajor(ANY_MAJOR)
      .setMinor(ANY_MINOR)
      .setShuffled(false)
      .setIdentifier("Everywhere")
      .build();

  private final UUID proximity;
  private final int major;
  private final int minor;
  private final boolean shuffled;
  private final String identifier;
  private final int hashCode;

  public static Builder builder() {
    return new Builder();
  }

  @Override
  public UUID getProximity() {
    return proximity;
  }

  @Override
  public int getMajor() {
    return major;
  }

  @Override
  public int getMinor() {
    return minor;
  }

  @Override
  public String getIdentifier() {
    return identifier;
  }

  @Override
  public boolean isShuffled() {
    return shuffled;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    Bundle bundle = new Bundle(getClass().getClassLoader());
    bundle.putSerializable(Constants.Region.UUID, proximity);
    bundle.putInt(Constants.Region.MAJOR, major);
    bundle.putInt(Constants.Region.MINOR, minor);
    bundle.putBoolean(Constants.Region.SHUFFLED, shuffled);
    bundle.putString(Constants.Region.IDENTIFIER, identifier);
    dest.writeBundle(bundle);
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (o == null || !(o instanceof BeaconRegion)) {
      return false;
    }

    final BeaconRegion IBeaconRegion = (BeaconRegion) o;

    return IBeaconRegion.getProximity().equals(proximity)
        && IBeaconRegion.getMajor() == major
        && IBeaconRegion.getMinor() == minor
        && IBeaconRegion.isShuffled() == shuffled;
  }

  @Override
  public int hashCode() {
    return hashCode;
  }

  private BeaconRegion(Builder builder) {
    SDKPreconditions.checkNotNull(builder.proximity, "Proximity is not set");
    this.proximity = builder.proximity;
    this.major = builder.major;
    this.minor = builder.minor;
    this.shuffled = builder.shuffled;
    this.identifier = builder.identifier;

    hashCode = HashCodeBuilder.init().append(proximity).append(major).append(minor).append(shuffled).build();
  }

  public static class Builder {

    private UUID proximity;
    private int major = ANY_MAJOR;
    private int minor = ANY_MINOR;
    private boolean shuffled;
    private String identifier;

    /**
     * Sets proximity uuid
     *
     * @param proximity region proximity
     * @return the builder
     */
    public Builder setProximity(final UUID proximity) {
      this.proximity = proximity;
      return this;
    }

    /**
     * Sets major
     *
     * @param major region major
     * @return the builder
     */
    public Builder setMajor(int major) {
      this.major = major;
      return this;
    }

    /**
     * Sets minor
     *
     * @param minor region minor
     * @return the builder
     */
    public Builder setMinor(int minor) {
      this.minor = minor;
      return this;
    }

    /**
     * Sets shuffled value
     *
     * @param shuffled true or false
     * @return the builder
     */
    public Builder setShuffled(boolean shuffled) {
      this.shuffled = shuffled;
      return this;
    }

    /**
     * Sets region identifier
     *
     * @param identifier identifier
     * @return the builder
     */
    public Builder setIdentifier(String identifier) {
      this.identifier = identifier;
      return this;
    }

    /**
     * Builds BeaconRegion
     *
     * @return region
     */
    public BeaconRegion build() {
      return new BeaconRegion(this);
    }
  }
}
