package com.kontakt.sdk.android.ble.device;

import android.os.Bundle;
import android.os.Parcel;
import com.kontakt.sdk.android.ble.discovery.ibeacon.IBeaconAdvertisingPacket;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.model.Device;
import com.kontakt.sdk.android.common.model.IBeaconId;
import com.kontakt.sdk.android.common.model.ResolvedId;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.IBeaconDevice;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.UUID;

/**
 * Class representing a real beacon device.
 * Please, note that in the SDK there is also another Device class
 * ({@link Device}) which represents a JSON model
 * for a device and is accessible while interacting with the REST Client.
 */
public class BeaconDevice implements IBeaconDevice {

  private static final String PARCEL_RSSI = "rssi";
  private static final String PARCEL_ADDRESS = "parcel_address";
  private static final String PARCEL_TIMESTAMP = "parcel_timestamp";
  private static final String PARCEL_NAME = "parcel_name";
  private static final String HASH_CODE = "hashCode";
  private static final String SHUFFLED = "shuffled";

  private final double rssi;
  private final UUID proximityUUID;
  private final int major;
  private final int minor;
  private final int txPower;
  private final String address;
  private final int hashCode;
  private final String name;
  private final String beaconUniqueId;
  private final String firmwareVersion;
  private final int batteryPower;
  private final Proximity proximity;
  private byte[] password;
  private final double accuracy;
  private final long timestamp;
  private final boolean shuffled;

  /**
   * The Parcelable CREATOR constant.
   */
  public static final Creator<BeaconDevice> CREATOR = new Creator<BeaconDevice>() {
    public BeaconDevice createFromParcel(Parcel in) {
      return new BeaconDevice(in);
    }

    public BeaconDevice[] newArray(int size) {
      return new BeaconDevice[size];
    }
  };

  /**
   * Creates BeaconDevice from {@link BeaconDevice} and {@link ResolvedId}
   * <p>
   * Should be used when resolving shuffled beacons manually
   * <br>
   *
   *
   * @param beaconDevice source beacon device
   * @param resolvedId target resolved values
   * @return new beacon device
   */
  public static BeaconDevice of(IBeaconDevice beaconDevice, ResolvedId resolvedId) {
    return new BeaconDevice(beaconDevice, resolvedId);
  }

  private BeaconDevice(IBeaconDevice beaconDevice, ResolvedId resolvedId) {
    IBeaconId iBeaconId = resolvedId.getIBeaconId();
    proximityUUID = iBeaconId.getProximity();
    major = iBeaconId.getMajor();
    minor = iBeaconId.getMinor();
    beaconUniqueId = resolvedId.getUniqueId();

    txPower = beaconDevice.getTxPower();
    accuracy = beaconDevice.getDistance();
    timestamp = beaconDevice.getTimestamp();
    firmwareVersion = beaconDevice.getFirmwareVersion();
    batteryPower = beaconDevice.getBatteryPower();
    proximity = beaconDevice.getProximity();
    rssi = beaconDevice.getRssi();
    address = beaconDevice.getAddress();
    name = beaconDevice.getName();
    shuffled = beaconDevice.isShuffled();
    hashCode = HashCodeBuilder.init().append(address).build();
  }

  /**
   * Instantiates a new Beacon.
   *
   * @param advertisingPacket the advertising package
   */
  public BeaconDevice(final IBeaconAdvertisingPacket advertisingPacket) {
    proximityUUID = advertisingPacket.getProximityUUID();
    major = advertisingPacket.getMajor();
    minor = advertisingPacket.getMinor();
    txPower = advertisingPacket.getTxPower();
    accuracy = advertisingPacket.getDistance();
    timestamp = advertisingPacket.getTimestamp();
    beaconUniqueId = advertisingPacket.getBeaconUniqueId();
    firmwareVersion = advertisingPacket.getFirmwareVersion();
    batteryPower = advertisingPacket.getBatteryPercentagePower();
    proximity = advertisingPacket.getProximity();
    rssi = advertisingPacket.getRssi();
    address = advertisingPacket.getAddress();
    name = advertisingPacket.getName();
    shuffled = advertisingPacket.isShuffled();
    hashCode = HashCodeBuilder.init().append(address).build();
  }

  private BeaconDevice(Parcel input) {
    final Bundle bundle = input.readBundle(getClass().getClassLoader());
    rssi = bundle.getDouble(PARCEL_RSSI);
    name = bundle.getString(PARCEL_NAME);
    address = bundle.getString(PARCEL_ADDRESS);
    proximityUUID = (UUID) bundle.getSerializable(Constants.Devices.PROXIMITY);
    major = bundle.getInt(Constants.Devices.MAJOR);
    minor = bundle.getInt(Constants.Devices.MINOR);
    txPower = bundle.getInt(Constants.Devices.TX_POWER);
    accuracy = bundle.getDouble(Constants.Devices.ACCURACY);
    password = bundle.getByteArray(Constants.Devices.PASSWORD);
    timestamp = bundle.getLong(PARCEL_TIMESTAMP);
    proximity = Proximity.fromDistance(accuracy);
    firmwareVersion = bundle.getString(Constants.FIRMWARE);
    beaconUniqueId = bundle.getString(Constants.UNIQUE_ID);
    batteryPower = bundle.getInt(Constants.Devices.BATTERY);
    hashCode = bundle.getInt(HASH_CODE);
    shuffled = bundle.getBoolean(SHUFFLED, false);
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel output, int flags) {

    final Bundle bundle = new Bundle(getClass().getClassLoader());
    bundle.putDouble(PARCEL_RSSI, rssi);
    bundle.putString(PARCEL_ADDRESS, address);
    bundle.putString(PARCEL_NAME, name);
    bundle.putSerializable(Constants.Devices.PROXIMITY, proximityUUID);
    bundle.putInt(Constants.Devices.MAJOR, major);
    bundle.putInt(Constants.Devices.MINOR, minor);
    bundle.putInt(Constants.Devices.TX_POWER, txPower);
    bundle.putDouble(Constants.Devices.ACCURACY, accuracy);
    bundle.putByteArray(Constants.Devices.PASSWORD, password);
    bundle.putLong(PARCEL_TIMESTAMP, timestamp);
    bundle.putString(Constants.FIRMWARE, firmwareVersion);
    bundle.putString(Constants.UNIQUE_ID, beaconUniqueId);
    bundle.putInt(Constants.Devices.BATTERY, batteryPower);
    bundle.putInt(HASH_CODE, hashCode);
    bundle.putBoolean(SHUFFLED, shuffled);
    output.writeBundle(bundle);
  }

  @Override
  public int compareTo(IBeaconDevice another) {
    SDKPreconditions.checkNotNull(another, "Comparing to null beacon device!");
    return Integer.valueOf(hashCode).compareTo(another.hashCode());
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (o == null || !(o instanceof BeaconDevice)) {
      return false;
    }

    if (o == this) {
      return true;
    }

    final BeaconDevice beaconDevice = (BeaconDevice) o;

    if (this.proximityUUID != null &&
        beaconDevice.proximityUUID != null &&
        !proximityUUID.equals(beaconDevice.proximityUUID)) {
      return false;
    }

    if (this.address != null && beaconDevice.address != null && !this.address.equals(beaconDevice.address)) {
      return false;
    }

    if (this.beaconUniqueId != null && beaconDevice.beaconUniqueId != null && !this.address.equals(beaconDevice.address)) {
      return false;
    }

    if ((this.minor != beaconDevice.minor) || (this.major != beaconDevice.major)) {
      return false;
    }

    return true;
  }

  @Override
  public int hashCode() {
    return hashCode;
  }

  /**
   * Gets timestamp - time when beacon device was discovered
   *
   * @return the timestamp
   */
  @Override
  public long getTimestamp() {
    return timestamp;
  }

  /**
   * Gets Proximity UUID.
   *
   * @return the proximity UUID
   */
  public UUID getProximityUUID() {
    return proximityUUID;
  }

  /**
   * Gets one of three values: IMMEDIATE, NEAR, FAR.
   *
   * @return the proximity
   */
  @Override
  public Proximity getProximity() {
    return proximity;
  }

  /**
   * Returns percentage battery power.
   *
   * @return the battery power
   */
  @Override
  public int getBatteryPower() {
    return batteryPower;
  }

  /**
   * Gets major value.
   *
   * @return the major
   */
  @Override
  public int getMajor() {
    return major;
  }

  /**
   * Gets beacon unique id.
   *
   * @return the beacon unique id
   */
  @Override
  public String getUniqueId() {
    return beaconUniqueId;
  }

  /**
   * Gets minor value.
   *
   * @return the minor
   */
  @Override
  public int getMinor() {
    return minor;
  }

  /**
   * Gets measured power.
   *
   * @return the measured power
   */
  @Override
  public int getTxPower() {
    return txPower;
  }

  @Override
  public DeviceProfile getProfile() {
    return DeviceProfile.IBEACON;
  }

  /**
   * Gets accuracy. The accuracy is calculated according to algorithm
   * described in the Stack Overflow answer.
   *
   * @return the accuracy [m]
   * @see <a href="http://stackoverflow.com/questions/20416218/understanding-ibeacon-distancing" target="_blank">Stack Overflow answer</a>
   */
  @Override
  public double getDistance() {
    return accuracy;
  }

  /**
   * Gets MAC address of beacon.
   *
   * @return the mac address
   */
  @Override
  public String getAddress() {
    return address;
  }

  /**
   * Sets authorization password.
   *
   * @param password the password
   */
  public void setPassword(final byte[] password) {
    this.password = password;
  }

  /**
   * Gets password.
   *
   * @return the byte [ ]
   */
  @Override
  public byte[] getPassword() {
    return password;
  }

  /**
   * Gets rssi.
   *
   * @return the rssi
   */
  @Override
  public double getRssi() {
    return rssi;
  }

  /**
   * Gets beacon name.
   *
   * @return the name
   */
  @Override
  public String getName() {
    return name;
  }

  /**
   * Gets firmware version.
   *
   * @return the firmware version
   */
  @Override
  public String getFirmwareVersion() {
    return firmwareVersion;
  }

  /**
   * Checks is device shuffled
   *
   * @return true or false
   */
  @Override
  public boolean isShuffled() {
    return shuffled;
  }
}
