package com.kontakt.sdk.android.ble.connection;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothGatt;
import android.bluetooth.BluetoothGattCallback;
import android.bluetooth.BluetoothGattCharacteristic;
import android.bluetooth.BluetoothProfile;
import android.content.Context;
import android.os.Build;
import android.os.Handler;
import android.os.RemoteException;
import android.util.Base64;
import com.kontakt.sdk.android.ble.device.KontaktDeviceCharacteristics;
import com.kontakt.sdk.android.ble.exception.ServiceAbsentException;
import com.kontakt.sdk.android.ble.spec.BluetoothDeviceCharacteristic;
import com.kontakt.sdk.android.ble.spec.KontaktDeviceCharacteristic;
import com.kontakt.sdk.android.ble.spec.KontaktDeviceService;
import com.kontakt.sdk.android.ble.util.BluetoothUtils;
import com.kontakt.sdk.android.common.TimestampUtil;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayDeque;
import java.util.Collection;
import java.util.Queue;
import java.util.concurrent.TimeUnit;

@TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2) class SecureGattController extends BluetoothGattCallback implements Closeable,
    GattController {

  private Context context;
  private KontaktDeviceConnection connection;
  private BluetoothGatt gattServer;
  private KontaktDeviceServiceStore secureKontaktDeviceStore;
  private Handler controlPointHandler;
  private Handler readCharacteristicHandler;
  private Queue<BluetoothDeviceCharacteristic> preAuthCharacteristicsQueue;
  private BluetoothDevice bluetoothDevice;
  private boolean readResponse;

  SecureGattController(KontaktDeviceConnection connection, Context context, RemoteBluetoothDevice device) throws RemoteException {
    this.connection = connection;
    this.context = context;
    controlPointHandler = new Handler();
    readCharacteristicHandler = new Handler();
    final BluetoothDevice bluetoothDevice = BluetoothUtils.getBluetoothDevice(device.getAddress());

    if (bluetoothDevice == null) {
      throw new RemoteException("Bluetooth device is null");
    }

    this.bluetoothDevice = bluetoothDevice;
  }

  @Override
  public boolean connect() {
    gattServer = bluetoothDevice.connectGatt(context, false, this);
    return gattServer != null;
  }

  @Override
  public void disconnect() {
    if (gattServer != null) {
      refresh();
      gattServer.disconnect();
    }
  }

  @Override
  public boolean writeCharacteristic(BluetoothGattCharacteristic characteristic) {
    return writeCharacteristic(characteristic, true);
  }

  @Override
  public boolean writeCharacteristic(BluetoothGattCharacteristic characteristic, boolean readResponse) {
    this.readResponse = readResponse;
    return gattServer.writeCharacteristic(characteristic);
  }

  @Override
  public void refresh() {
    BluetoothUtils.refreshGattServer(gattServer);
  }

  @Override
  public void close() throws IOException {
    if (gattServer != null) {
      disconnect();
      gattServer.close();
    }

    if (secureKontaktDeviceStore != null) {
      secureKontaktDeviceStore.clear();
    }

    if (preAuthCharacteristicsQueue != null) {
      preAuthCharacteristicsQueue.clear();
    }

    connection = null;
    controlPointHandler.removeCallbacksAndMessages(controlPointRunnable);
    readCharacteristicHandler.removeCallbacks(characteristicReadRunnable);
  }

  @Override
  public void onConnectionStateChange(BluetoothGatt gatt, int status, int newState) {
    switch (newState) {
      case BluetoothProfile.STATE_CONNECTED:
        if (BluetoothGatt.GATT_SUCCESS == status) {
          connection.onConnectionStateChange(KontaktDeviceConnection.State.CONNECTED);
          connection.onConnected();
          if (!gattServer.discoverServices()) {
            connection.onError(KontaktDeviceConnection.ERROR_SERVICES_DISCOVERY);
          }
        } else {
          connection.onError(KontaktDeviceConnection.toGattError(status));
        }
        break;

      case BluetoothProfile.STATE_DISCONNECTED:
        if (connection != null) {
          if (BluetoothGatt.GATT_SUCCESS != status) {
            connection.onError(KontaktDeviceConnection.toGattError(status));
          }
          if (connection != null) {
            connection.onConnectionStateChange(KontaktDeviceConnection.State.DISCONNECTED);
          }
          if (connection != null) {
            connection.onDisconnected();
          }
        }
        break;

      default:
        throw new IllegalArgumentException("Unsupported connection state change code: " + newState);
    }
  }

  @Override
  public void onServicesDiscovered(BluetoothGatt gatt, int status) {
    Logger.d(String.format("On Services Discovered: %s ", gatt.toString()));

    if (status != BluetoothGatt.GATT_SUCCESS) {
      Logger.v("Services discovered but with no success: " + status);
      return;
    }

    try {
      secureKontaktDeviceStore = new KontaktDeviceServiceStore(gatt.getServices(), KontaktDeviceServiceStore.StoreType.SECURE);
      connection.onServicesDiscovered(secureKontaktDeviceStore);
      if (secureKontaktDeviceStore.contains(KontaktDeviceService.DFU_SERVICE)) {
        connection.onDfuModeEnabled();
      } else {
        connection.onConnectionStateChange(KontaktDeviceConnection.State.AUTHENTICATING);
        readCharacteristicHandler.postDelayed(characteristicReadRunnable, TimeUnit.SECONDS.toMillis(1));
      }
    } catch (RemoteException e) {
      connection.onFailure(KontaktDeviceConnection.FAILURE_UNKNOWN_BEACON);
    }
  }

  @Override
  public void onCharacteristicRead(BluetoothGatt gatt, BluetoothGattCharacteristic characteristic, int status) {
    if (BluetoothGatt.GATT_SUCCESS == status) {

      if (connection.getState() == KontaktDeviceConnection.State.CHARACTERISTICS_REQUESTING) {
        requestOrSetAuthenticated();
        return;
      }
      if (connection.getState() == KontaktDeviceConnection.State.AUTHENTICATED) {
        BluetoothDeviceCharacteristic wrapper = new BluetoothDeviceCharacteristic(characteristic);
        KontaktDeviceCharacteristic kontaktDeviceCharacteristic = wrapper.getKontaktDeviceCharacteristic();
        if (KontaktDeviceCharacteristic.SECURE_CONTROL_POINT == kontaktDeviceCharacteristic) {
          checkControlPointValue(wrapper);
        } else if (KontaktDeviceCharacteristic.SECURE_RESPONSE == kontaktDeviceCharacteristic) {
          checkResponse(wrapper);
        }
      }
    } else if (BluetoothGatt.GATT_READ_NOT_PERMITTED == status) {
      if (connection.getState() == KontaktDeviceConnection.State.CHARACTERISTICS_REQUESTING) {
        requestOrSetAuthenticated();
      }
    }
  }

  @Override
  public void onCharacteristicWrite(BluetoothGatt gatt, BluetoothGattCharacteristic characteristic, int status) {
    if (connection.isConnected()) {
      try {
        secureKontaktDeviceStore.replace(new BluetoothDeviceCharacteristic(characteristic));
      } catch (ServiceAbsentException ignored) {
      }

      if (BluetoothGatt.GATT_SUCCESS == status) {
        BluetoothDeviceCharacteristic bluetoothDeviceCharacteristic = new BluetoothDeviceCharacteristic(characteristic);
        KontaktDeviceCharacteristic kontaktDeviceCharacteristic = bluetoothDeviceCharacteristic.getKontaktDeviceCharacteristic();
        if (KontaktDeviceCharacteristic.SECURE_WRITE == kontaktDeviceCharacteristic) {
          if (shouldReadResponse()) {
            scheduleCheckControlPoint();
          } else {
            if (connection != null) {
              connection.onCharacteristicWritten(true, new WriteListener.WriteResponse(
                  TimestampUtil.getTimestamp(), null));
            }
            if (connection != null) {
              connection.notifyDataSetChanged();
            }
          }
        }
      }
    }
  }

  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  private void checkControlPointValue(BluetoothDeviceCharacteristic bluetoothDeviceCharacteristic) {
    byte[] value = bluetoothDeviceCharacteristic.getValue();
    if (value == null) {
      scheduleCheckControlPoint();
      return;
    }

    if (ConversionUtils.asInt(value) == 1) {
      readResponse();
    } else {
      scheduleCheckControlPoint();
    }
  }

  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  private void checkResponse(BluetoothDeviceCharacteristic bluetoothDeviceCharacteristic) {
    byte[] responseValue = bluetoothDeviceCharacteristic.getValue();
    String response = Base64.encodeToString(responseValue, Base64.DEFAULT);
    //that seems weird but code is executed line by line. So the callback onCharacteristicWritten may eventually cause clos controller
    //and that is why we are checking it seperately
    if (connection != null) {
      connection.onCharacteristicWritten(true, new WriteListener.WriteResponse(TimestampUtil.getTimestamp(), response));
    }
    if (connection != null) {
      connection.notifyDataSetChanged();
    }
  }

  private boolean shouldReadResponse() {
    return readResponse;
  }

  private void scheduleCheckControlPoint() {
    controlPointHandler.postDelayed(controlPointRunnable, 100);
  }

  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  private void readResponse() {
    try {
      BluetoothDeviceCharacteristic responseCharacterisitic = secureKontaktDeviceStore.getSecureResponseCharacteristic();
      if (!gattServer.readCharacteristic(responseCharacterisitic)) {
        connection.onCharacteristicWritten(false, null);
      }
    } catch (Exception e) {
      Logger.e("readResponse SecureGattController", e);
    }
  }

  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  private void request(final BluetoothDeviceCharacteristic wrapper) {
    if (!gattServer.readCharacteristic(wrapper)) {
      if (connection != null && connection.getState() == KontaktDeviceConnection.State.CHARACTERISTICS_REQUESTING) {
        requestOrSetAuthenticated();
      }
    }
  }

  private void requestOrSetAuthenticated() {
    if (preAuthCharacteristicsQueue.isEmpty()) {
      connection.onConnectionStateChange(KontaktDeviceConnection.State.AUTHENTICATED);
      connection.onAuthenticationSuccess(new KontaktDeviceCharacteristics(secureKontaktDeviceStore));
    } else {
      request(preAuthCharacteristicsQueue.poll());
    }
  }

  private void requestCharacteristics() {
    Collection<BluetoothDeviceCharacteristic> readableCharacteristics = secureKontaktDeviceStore.getReadableCharacteristics();
    if (preAuthCharacteristicsQueue == null) {
      preAuthCharacteristicsQueue = new ArrayDeque<BluetoothDeviceCharacteristic>(readableCharacteristics.size());
    } else {
      preAuthCharacteristicsQueue.clear();
    }

    preAuthCharacteristicsQueue.addAll(readableCharacteristics);
    request(preAuthCharacteristicsQueue.poll());
  }

  private Runnable controlPointRunnable = new Runnable() {
    @Override
    public void run() {
      try {
        BluetoothDeviceCharacteristic controlPointCharacteristic = secureKontaktDeviceStore.getSecureControlPointCharacteristic();
        gattServer.readCharacteristic(controlPointCharacteristic);
      } catch (Exception e) {
        Logger.e("controlPointRunnable", e);
      }
    }
  };

  private Runnable characteristicReadRunnable = new Runnable() {
    @Override
    public void run() {
      if (connection == null) {
        return;
      }
      if (connection.getState() == KontaktDeviceConnection.State.AUTHENTICATING) {
        connection.onConnectionStateChange(KontaktDeviceConnection.State.CHARACTERISTICS_REQUESTING);
        requestCharacteristics();
      }
    }
  };
}
