package com.kontakt.sdk.android.ble.connection;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothGattCharacteristic;
import android.content.Context;
import android.os.Build;
import android.os.RemoteException;
import android.util.Base64;
import com.kontakt.sdk.android.ble.device.KontaktDeviceCharacteristics;
import com.kontakt.sdk.android.common.FirmwareRevisions;
import com.kontakt.sdk.android.common.KontaktSDK;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.model.Config;
import com.kontakt.sdk.android.common.model.Credentials;
import com.kontakt.sdk.android.common.model.Preset;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import com.kontakt.sdk.android.common.util.EddystonePropertyValidator;
import com.kontakt.sdk.android.common.util.EddystoneUtils;
import com.kontakt.sdk.android.common.util.IBeaconPropertyValidator;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.EnumSet;
import java.util.Set;
import java.util.UUID;

import static com.kontakt.sdk.android.ble.connection.WriteListener.Cause.FEATURE_NOT_SUPPORTED;
import static com.kontakt.sdk.android.ble.connection.WriteListener.Cause.GATT_FAILURE;
import static com.kontakt.sdk.android.ble.connection.WriteListener.Cause.INCORRECT_VALUE;

/**
 * KontaktDeviceConnection connection allows to establish connection with beacon.<br>
 * Since firmware in version 4.0 device does not to have setup password, but below it still needs set password
 * <br><br>
 * All operation with device should be as quick as possible. You should not open connection and wait long time before taking any action.
 * After doing job connection should be closed immediately
 */
public class KontaktDeviceConnection implements IKontaktDeviceConnection {
  /**
   * Error code informing that Beacon services discovery has failed.
   */
  public static final int ERROR_SERVICES_DISCOVERY = 1;
  /**
   * Error code informing that error has occured during characteristic
   * overwrite process.
   */
  public static final int ERROR_OVERWRITE_REQUEST = 2;
  /**
   * Error code informing that unexpected error has occured while attempting
   * beacon authentication.
   */
  public static final int ERROR_AUTHENTICATION = 3;
  /**
   * Failure code informing that BLE device was not recognised as kontakt.io
   * Beacon.
   */
  public static final int FAILURE_UNKNOWN_BEACON = 4;

  /**
   * Failure code informing that beacon was not authenticated because of
   * wrong password. When authentication failure occurs, beacon change its
   * state to Non-connectable mode lasting about 20 minutes.
   */
  public static final int FAILURE_WRONG_PASSWORD = 7;

  /**
   * Error code informing that Batch write operation failed during Proximity UUID write.
   */
  public static final int ERROR_BATCH_WRITE_PROXIMITY_UUID = 8;

  /**
   * Error code informing that Batch write operation failed during Transmission Power write.
   */
  public static final int ERROR_BATCH_WRITE_TX_POWER = 9;

  /**
   * Error code informing that Batch write operation failed during Major write.
   */
  public static final int ERROR_BATCH_WRITE_MAJOR = 10;

  /**
   * Error code informing that Batch write operation failed during Minor write.
   */
  public static final int ERROR_BATCH_WRITE_MINOR = 11;

  /**
   * Error code informing that Batch write operation failed during Interval write.
   */
  public static final int ERROR_BATCH_WRITE_INTERVAL = 12;

  /**
   * Error code informing that Batch write operation failed during Profile write
   */
  public static final int ERROR_BATCH_WRITE_PROFILE = 13;

  /**
   * Error code informing that Batch write operation failed during Namespace write
   */
  public static final int ERROR_BATCH_WRITE_NAMESPACE = 14;

  /**
   * Error code informing that Batch write operation failed during InstanceId write
   */
  public static final int ERROR_BATCH_WRITE_INSTANCE_ID = 15;

  /**
   * Error code informing that Batch write operation failed during Url write
   */
  public static final int ERROR_BATCH_WRITE_URL = 16;

  /**
   * Error code informing that Batch write operation failed during Name write
   */
  public static final int ERROR_BATCH_WRITE_NAME = 17;

  /**
   * Error code informing that Batch write failed during Password write
   */
  public static final int ERROR_BATCH_WRITE_PASSWORD = 18;

  private static final int GATT_ERROR_BEGIN = 1000;

  static int toGattError(int errorCode) {
    return errorCode + GATT_ERROR_BEGIN;
  }

  /**
   * Metohd for checking is error code indicating GATT error
   *
   * @param errorCode received from {@link com.kontakt.sdk.android.ble.connection.IKontaktDeviceConnection.ConnectionListener#onErrorOccured(int)}
   * @return true if errorCode is gatt error
   */
  public static boolean isGattError(int errorCode) {
    return errorCode > GATT_ERROR_BEGIN;
  }

  /**
   * Method for getting gatt error status code from received errorCode
   * <br><br> throws {@link IllegalAccessException} if errorCode is not gatt error code
   *
   * @param errorCode error code
   * @return Gatt status code, e.g 133
   */
  public static int getGattError(int errorCode) {
    if (!isGattError(errorCode)) {
      throw new IllegalArgumentException("Error code is not gatt error");
    }
    return errorCode - GATT_ERROR_BEGIN;
  }

  private static final Set<State> CONNECTION_AVAILABLE_STATES =
      EnumSet.of(State.CONNECTED, State.AUTHENTICATED, State.AUTHENTICATING, State.CHARACTERISTICS_REQUESTING);

  private final RemoteBluetoothDevice beaconDevice;
  private ConnectionListener connectionListener;
  private volatile State state = State.DISCONNECTED;
  private GattController gattController;
  private KontaktDeviceServiceStore serviceStore;
  private WriteListener writeListener = WriteListener.NULL_LISTENER;
  private Context context;
  private GattControllerFactory gattControllerFactory;

    public KontaktDeviceConnection(final Context context,
                                   final RemoteBluetoothDevice bluetoothDevice,
                                   final ConnectionListener connectionListener) {
        KontaktSDK.getInstance();
        SDKPreconditions.checkNotNull(bluetoothDevice, "Beacon device is null.");
        SDKPreconditions.checkNotNull(connectionListener, "Connection listener is null.");
        validateBeaconPassword(bluetoothDevice);
        this.context = context.getApplicationContext();
        this.beaconDevice = bluetoothDevice;
        this.connectionListener = connectionListener;
    }

  protected KontaktDeviceConnection(final Context context, final RemoteBluetoothDevice bluetoothDevice, final ConnectionListener connectionListener,
      final KontaktDeviceServiceStore serviceStore) {
    this(context, bluetoothDevice, connectionListener);
    this.serviceStore = serviceStore;
  }

  @Override
  /**
   * Gets beacon that the connection is established with.
   *
   * @return the iBeacon device.
   */ public RemoteBluetoothDevice getDevice() {
    return beaconDevice;
  }

  @Override
  /**
   * Connects to Beacon device.
   *
   * @return true, if the connection attempt was initiated successfully
   */
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public synchronized boolean connect() {
    SDKPreconditions.checkState(!isClosed(), "BeaconConnection is closed.");

        if (gattController == null) {
            try {
                gattController = GattControllerFactory.createGattController(this, context, beaconDevice);
                return gattController.connect();
            } catch (RemoteException ignored) {
            }
        }

    return false;
  }

  @Override
  /**
   * Returns true if Connection with Beacon is established.
   * The method is synchronized.
   *
   * @return the boolean flag indicating whether connection is established.
   */ public synchronized boolean isConnected() {
    return CONNECTION_AVAILABLE_STATES.contains(state);
  }

  @Override
  /**
   * Returns true if Beacon was successfully authenticated.
   * The method is synchronized.
   *
   * @return the boolean flag indicating whether beacon was authenticated.
   */ public synchronized boolean isAuthenticated() {
    return state == State.AUTHENTICATED;
  }

  @Override
  /**
   * Closes Beacon connection releases provided resources.
   * The method is synchronized.
   */
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public synchronized void close() {

    if (isConnected()) {
      disconnect();
    }

    Logger.v("Closing connection");

    try {
      if (gattController != null) {
        gattController.close();
      }
    } catch (IOException ignored) {
    }

    gattController = null;
    connectionListener = null;
  }

  /**
   * Disconnect from Beacon device.
   */
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  private void disconnect() {
    if (gattController != null) {
      Logger.v("Disconnecting from beacon device");
      gattController.disconnect();
    } else {
      Logger.v("Connection was not initialized");
    }
  }

  @Override
  public synchronized boolean isClosed() {
    return gattController == null && connectionListener == null;
  }

  /**
   * Applies pending configuration ({@link Config}) for Device.
   * <br>
   * It is possible to encounter problems during batch (the Config in this case)
   * write operation which result in launching error handle callback method.
   * Once the write batch operation fails, the Write Batch Processor
   * will attempt performing rollback write batch operation aiming at
   * restoring original Beacon configuration.
   * The rollback operation however may not succeed as well leaving Beacon device
   * changed partially.
   * Therefore it is strongly advisable to write batches in relatively close distance
   * to Beacon device. Thus, the risk of leaving Beacon in unspecified state may
   * be avoided.
   * <br>
   * The method is synchronized.
   * <br><br>
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or higher
   *
   *
   * @param config             the config
   * @param writeBatchListener the write batch listener
   */
    @Override
    public synchronized void applyConfig(final Config config, final WriteBatchListener<Config> writeBatchListener) {
        SDKPreconditions.checkState(!isClosed(), "Beacon connection is closed");
        SDKPreconditions.checkState(isAuthenticated(), "Beacon connection is not authenticated");
        SDKPreconditions.checkNotNull(writeBatchListener, "Write Batch Listener is null.");
        OperationWrite.CONFIG.validate(beaconDevice);
        if (!isAuthenticated()) {
            writeBatchListener.onWriteFailure();
            return;
        }

        Logger.d("Applying config...");

        final RemoteBluetoothDevice.Characteristics characteristics = new KontaktDeviceCharacteristics(serviceStore);
        KontaktDeviceBatchProcessor.from(KontaktDeviceBatchProcessor.Batch.select(config, characteristics), this)
            .process(createProcessingListener(config, writeBatchListener));
    }

    /**
     * Accepts predefined profile ({@link Preset}) and changes parameters for Beacon.
     * <br>
     * It is possible to encounter problems during batch (the Preset in this case) write operation which result
     * in launching error handle callback method. Once the write batch operation failes,
     * the Write Batch Processor will attempt performing rollback write batch
     * operation aiming at restoring original Beacon configuration.
     * The rollback operation however may not succeed as well leaving Beacon device
     * changed partially.
     * Therefore it is strongly advisable to write batches in relatively close distance
     * to Beacon device. Thus, the risk of leaving Beacon in unspecified state may
     * be avoided.
     * <br>
     * The method is synchronized.
     * <br><br>
     * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or higher
     * 
     *
     * @param profile            the profile
     * @param writeBatchListener the write batch listener
     */
    @Override
    public synchronized void acceptProfile(final Preset profile, final WriteBatchListener<Preset> writeBatchListener) {
        SDKPreconditions.checkState(!isClosed(), "Beacon connection is closed");
        SDKPreconditions.checkState(isAuthenticated(), "Beacon connection is not authenticated");
        SDKPreconditions.checkNotNull(writeBatchListener, "Write Batch Listener is null.");
        OperationWrite.PROFILE.validate(beaconDevice);
        if (!isAuthenticated()) {
            writeBatchListener.onWriteFailure();
            return;
        }

    Logger.d("Accepting profile...");

    final RemoteBluetoothDevice.Characteristics characteristics = new KontaktDeviceCharacteristics(serviceStore);
    KontaktDeviceBatchProcessor.from(
        KontaktDeviceBatchProcessor.Batch.select(profile, characteristics), this)
        .process(createProcessingListener(profile, writeBatchListener));
  }

  /**
   * Writes minor value. The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   * 
   *
   * @param value the value
   * @param writeListener the write listener
   */
  @Override
  public synchronized void overwriteMinor(final int value, final WriteListener writeListener) {
    SDKPreconditions.checkNotNull(writeListener, "WriteListener is null.");
    OperationWrite.MINOR.validate(beaconDevice);
    Logger.d(String.format("Writing new minor value %s ", String.valueOf(value)));

    try {
      IBeaconPropertyValidator.validateMinor(value);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    try {
      overwrite(serviceStore.getMinorCharacteristic(), ConversionUtils.invert(ConversionUtils.to2ByteArray(value)), writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(WriteListener.Cause.FEATURE_NOT_SUPPORTED);
    }
  }

  /**
   * Changes Device mode to non-connectable. During the non-connectable mode
   * no connection can be established with the Device for about 20 minutes.
   * <br>
   * The non-connectable mode is also enabled when authentication failure
   * occured because of providing wrong password.
   * <br>
   * Device Master Password can be obtained via REST Client. The result
   * callback is invoked by WriteListener.
   * <br>
   * For more information concerning Device master password, see
   * Credentials model
   * ({@link Credentials})
   * <br>
   * The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   * 
   *
   * @param masterPassword the beacon master password
   * @param writeListener the write listener
   */
  @Override
  public synchronized void enableNonConnectableMode(final String masterPassword, final WriteListener writeListener) {
    SDKPreconditions.checkNotNull(writeListener, "WriteListener is null.");
    OperationWrite.NON_CONNECTABLE_MODE.validate(beaconDevice);
    try {
      IBeaconPropertyValidator.validateBeaconMasterPassword(masterPassword);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    Logger.d(String.format("Enabling non-connectable mode - master password: %s", masterPassword));

    try {
      overwrite(serviceStore.getNonConnectableCharacteristic(), masterPassword.getBytes(), writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(WriteListener.Cause.FEATURE_NOT_SUPPORTED);
    }
  }

  /**
   * Changes Beacon major value. The result callback is invoked by WriteListener.
   * <br>
   * The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   * 
   *
   * @param value new value
   * @param writeListener the write listener
   */
  @Override
  public synchronized void overwriteMajor(final int value, final WriteListener writeListener) {
    SDKPreconditions.checkNotNull(writeListener, "WriteListener is null.");
    OperationWrite.MAJOR.validate(beaconDevice);
    try {
      IBeaconPropertyValidator.validateMajor(value);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
    }

    Logger.d(String.format("Writing new major value - %s", String.valueOf(value)));

    try {
      overwrite(serviceStore.getMajorCharacteristic(), ConversionUtils.invert(ConversionUtils.to2ByteArray(value)), writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(WriteListener.Cause.FEATURE_NOT_SUPPORTED);
    }
  }

  /**
   * Changes Beacon Proximity UUID value. The result callback is invoked by WriteListener.
   * <br>
   * The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   * 
   *
   * @param proximity the proximity UUID
   * @param writeListener the write listener
   */
  @Override
  public synchronized void overwriteProximityUUID(final UUID proximity, final WriteListener writeListener) {
    SDKPreconditions.checkNotNull(writeListener, "WriteListener is null.");
    OperationWrite.PROXIMITY_UUID.validate(beaconDevice);
    try {
      SDKPreconditions.checkNotNull(proximity);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    try {
      overwrite(serviceStore.getProximityCharacteristic(), ConversionUtils.convert(proximity), writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(WriteListener.Cause.FEATURE_NOT_SUPPORTED);
    }
  }

  /**
   * Changes beacon password. The password must be 4-byte long. For more
   * information see kontakt.io Datasheet - version 2.0.
   * The result callback is invoked by WriteListener.
   * <br>
   * The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   * 
   *
   * @param newPassword the new password
   * @param writeListener the write listener
   * @see <a href="http://docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon Datasheet - version 2.0</a>
   */
  @Override
  public synchronized void overwritePassword(final String newPassword, final WriteListener writeListener) {
    SDKPreconditions.checkNotNull(writeListener, "WriteListener is null.");
    OperationWrite.PASSWORD.validate(beaconDevice);
    try {
      IBeaconPropertyValidator.validateBeaconPassword(newPassword, beaconDevice.getFirmwareVersion());
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    try {
      overwrite(serviceStore.getSetNewPasswordCharacteristic(), newPassword.getBytes(), writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(WriteListener.Cause.FEATURE_NOT_SUPPORTED);
    }
  }

  /**
   * Changes Beacon model name. The name should be no longer than sequence of 15 bytes
   * (15 ASCII characters). For more information see kontakt.io
   * Datasheet - version 2.0.
   * The result callback is invoked by WriteListener.
   * <br>
   * The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   * 
   *
   * @param newModelName the new model name
   * @param writeListener the write listener
   * @see <a href="http://docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon Datasheet - version 2.0</a>
   */
  @Override
  public synchronized void overwriteModelName(final String newModelName, final WriteListener writeListener) {
    SDKPreconditions.checkNotNull(writeListener, "WriteListener is null.");
    OperationWrite.MODEL_NAME.validate(beaconDevice);
    try {
      IBeaconPropertyValidator.validateModelName(newModelName);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    final byte[] modelNameBytes = newModelName.getBytes();

    Logger.d(String.format("Writing model Name - %s", newModelName));
    try {
      overwrite(serviceStore.getPropagatedDeviceNameCharacteristic(), modelNameBytes, writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(WriteListener.Cause.FEATURE_NOT_SUPPORTED);
    }
  }

  /**
   * Changes Beacon advertising interval. The interval value is accepted in milliseconds and must
   * be within [20 - 10240] ms range.
   * <br>
   * The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   * 
   *
   * @param millis the interval millis
   * @param writeListener the write listener
   */
  @Override
  public synchronized void overwriteAdvertisingInterval(final long millis, final WriteListener writeListener) {
    SDKPreconditions.checkNotNull(writeListener, "Write Listener is null.");
    OperationWrite.ADVERTISING_INTERVAL.validate(beaconDevice);
    try {
      IBeaconPropertyValidator.validateAdvertisingInterval((int) millis);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    Logger.d(String.format("Writing advertising interval - %s", String.valueOf(millis)));

    final int value = Double.valueOf(Math.ceil(millis / 0.625)).intValue();

    try {
      overwrite(serviceStore.getAdvertisingIntervalCharacteristic(), ConversionUtils.invert(
          ConversionUtils.to2ByteArray(value)), writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(WriteListener.Cause.FEATURE_NOT_SUPPORTED);
    }
  }

  /**
   * Changes Beacon power level. The Power level is described by 8 levels where 0 is the
   * lowest (default) and 7 is the highest. For more information see
   * kontakt.io Datasheet - version 2.0.
   * <br>
   * The result callback is invoked by WriteListener.
   * <br>
   * The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   * 
   *
   * @param powerLevel the power level
   * @param writeListener the write listener
   * @see <a href="http://docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon Datasheet - version 2.0.</a>
   */
  @Override
  public synchronized void overwritePowerLevel(final int powerLevel, final WriteListener writeListener) {
    SDKPreconditions.checkNotNull(writeListener, "WriteListener is null.");
    OperationWrite.POWER_LEVEL.validate(beaconDevice);
    try {
      IBeaconPropertyValidator.validatePowerLevel(powerLevel);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    try {
      overwrite(serviceStore.getPowerLevelCharacteristic(), ConversionUtils.convertPowerLevel(powerLevel), writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(WriteListener.Cause.FEATURE_NOT_SUPPORTED);
    }
  }

  /**
   * Switch to device profile.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   * 
   *
   * @param deviceProfile the device profile
   * @param writeListener the write listener
   */
  @Override
  public synchronized void switchToDeviceProfile(final DeviceProfile deviceProfile, final WriteListener writeListener) {
    SDKPreconditions.checkNotNull(deviceProfile, "DeviceProfile cannot be null");
    SDKPreconditions.checkNotNull(writeListener, "WriteListener is null.");
    OperationWrite.DEVICE_PROFILE.validate(beaconDevice);
    final byte[] deviceProfileData = new byte[] { (byte) deviceProfile.getActiveProfileValue() };

    try {
      overwrite(serviceStore.getActiveProfileCharacteristic(), deviceProfileData, writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(WriteListener.Cause.FEATURE_NOT_SUPPORTED);
    }
  }

  /**
   * Resets device so that connection is established once again.
   * <br>
   * For more information see kontakt.io Datasheet - version 2.0.
   * <br>
   * The result callback is invoked by WriteListener.
   * <br>
   * The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   * 
   *
   * @param writeListener the write listener
   * @see <a href="http://docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon Datasheet - version 2.0.</a>
   */
  @Override
  public synchronized void resetDevice(final WriteListener writeListener) {
    SDKPreconditions.checkNotNull(writeListener, "WriteListener is null.");
    OperationWrite.RESET.validate(beaconDevice);
    Logger.d("Resetting Beacon device...");

    try {
      overwrite(serviceStore.getResetCharacteristic(), new byte[] { 0x01 }, writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(WriteListener.Cause.FEATURE_NOT_SUPPORTED);
    }
  }

  /**
   * Enables dfu mode. The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   * 
   *
   * @param masterPassword the master password
   * @param writeListener the write listener
   */
  @Override
  public synchronized void enableDfuMode(final String masterPassword, final WriteListener writeListener) {
    SDKPreconditions.checkNotNull(writeListener, "WriteListener is null.");
    OperationWrite.DFU.validate(beaconDevice);
    try {
      IBeaconPropertyValidator.validateBeaconMasterPassword(masterPassword);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
    }

    try {
      overwrite(serviceStore.getBootloaderCharacteristic(), masterPassword.getBytes(), writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(WriteListener.Cause.FEATURE_NOT_SUPPORTED);
    }
  }

  /**
   * Overwrites url. The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   * 
   *
   * @param newUrl the new url
   * @param writeListener the write listener
   */
  @Override
  public synchronized void overwriteUrl(final String newUrl, final WriteListener writeListener) {
    SDKPreconditions.checkNotNull(writeListener, "WriteListener is null");
    OperationWrite.URL.validate(beaconDevice);
    try {
      EddystonePropertyValidator.validateUrl(newUrl);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    byte[] url = EddystoneUtils.serializeUrl(newUrl);
    try {
      overwrite(serviceStore.getUrlCharacteristic(), url, writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(FEATURE_NOT_SUPPORTED);
    }
  }

  /**
   * Overwrites namespace id. The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   * 
   *
   * @param namespaceId the namespace id
   * @param writeListener the write listener
   */
  @Override
  public synchronized void overwriteNamespaceId(final String namespaceId, final WriteListener writeListener) {
    SDKPreconditions.checkNotNull(writeListener, "WriteListener is null");
    SDKPreconditions.checkNotNullOrEmpty(namespaceId, "Namespace is null or empty");
    OperationWrite.NAMESPACE.validate(beaconDevice);
    byte[] bytes;
    try {
      EddystonePropertyValidator.validateHexString(namespaceId);
      bytes = EddystoneUtils.hexStringToByteArray(namespaceId);
      EddystonePropertyValidator.validateNamespace(bytes);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    try {
      overwrite(serviceStore.getNamespaceIdCharacteristic(), bytes, writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(FEATURE_NOT_SUPPORTED);
    }
  }

  /**
   * Overwrite instance id. The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   * 
   *
   * @param instanceId the instance id
   * @param writeListener the write listener
   */
  @Override
  public synchronized void overwriteInstanceId(final String instanceId, final WriteListener writeListener) {
    SDKPreconditions.checkNotNull(writeListener, "WriteListener is null");
    SDKPreconditions.checkNotNullOrEmpty(instanceId, "Instance id is null or empty");
    OperationWrite.INSTANCE_ID.validate(beaconDevice);
    byte[] bytes;
    try {
      EddystonePropertyValidator.validateHexString(instanceId);
      bytes = EddystoneUtils.hexStringToByteArray(instanceId);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    try {
      EddystonePropertyValidator.validateHexString(instanceId);
      EddystonePropertyValidator.validateInstanceId(bytes);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    try {
      overwrite(serviceStore.getInstanceIdCharacteristic(), bytes, writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(FEATURE_NOT_SUPPORTED);
    }
  }

  /**
   * Restores default Beacon settings. As an argument, 6-byte long Beacon master password must be
   * provided. The master password can be obtained via API Client.
   * <br>
   * The method is synchronized.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided {@link RemoteBluetoothDevice} is in version 4.0 or
   * higher
   * 
   *
   * @param masterPassword the master password
   * @param writeListener the write listener
   * @see <a href="http://docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon Datasheet - version 2.0.</a>
   */
  @Override
  public synchronized void restoreDefaultSettings(final String masterPassword, final WriteListener writeListener) {
    OperationWrite.RESTORE.validate(beaconDevice);
    Logger.d("Restoring default settings to Beacon device...");

    try {
      IBeaconPropertyValidator.validateBeaconMasterPassword(masterPassword);
    } catch (Exception e) {
      writeListener.onWriteFailure(INCORRECT_VALUE);
      return;
    }

    try {
      overwrite(serviceStore.getDefaultSettingsCharacteristic(), masterPassword.getBytes(), writeListener);
    } catch (Exception e) {
      writeListener.onWriteFailure(WriteListener.Cause.FEATURE_NOT_SUPPORTED);
    }
  }

  /**
   * Method for writing secure config. Accessible for device with firmware 4.0 and above
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided device is  below 4.0 firmware version
   * *
   */
  @Override
  public synchronized void applySecureConfig(String secureConfig, WriteListener writeListener) {
    SDKPreconditions.checkNotNull(writeListener, "WriteListener is null");
    SDKPreconditions.checkNotNullOrEmpty(secureConfig, "Secure config is null or empty");

    OperationWrite.SECURE_CONFIG.validate(beaconDevice);

    byte[] decodedConfig = Base64.decode(secureConfig, Base64.DEFAULT);

    try {
      overwriteSecure(serviceStore.getSecureWriteCharacteristic(), decodedConfig, writeListener, true);
    } catch (Exception e) {
      writeListener.onWriteFailure(FEATURE_NOT_SUPPORTED);
    }
  }

  @Override
  public void executeSecureCommand(String secureCommand, WriteListener writeListener) {
    SDKPreconditions.checkNotNull(writeListener, "WriteListener is null");
    SDKPreconditions.checkNotNullOrEmpty(secureCommand, "Secure command is null or empty");

    OperationWrite.SECURE_COMMAND.validate(beaconDevice);

    byte[] decodedCommand = Base64.decode(secureCommand, Base64.DEFAULT);

    try {
      overwriteSecure(serviceStore.getSecureWriteCharacteristic(), decodedCommand, writeListener, false);
    } catch (Exception e) {
      writeListener.onWriteFailure(FEATURE_NOT_SUPPORTED);
    }
  }

  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  protected void overwrite(final BluetoothGattCharacteristic characteristic, final byte[] newValue, final WriteListener listener) {
    SDKPreconditions.checkState(!isClosed(), "The connection is closed");
    SDKPreconditions.checkState(isAuthenticated(), "The device is not authenticated");

    registerWriteListener(listener);
    final byte[] currentValue = characteristic.getValue();
    characteristic.setValue(newValue);
    Logger.v(String.format("Writing value %s to characteristic %s", new String(newValue), characteristic.getUuid().toString()));
    if (!gattController.writeCharacteristic(characteristic)) {
      characteristic.setValue(currentValue);
      writeListener.onWriteFailure(WriteListener.Cause.GATT_FAILURE);
      unregisterWriteListener();
    }
  }

  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  protected void overwriteSecure(final BluetoothGattCharacteristic characteristic, final byte[] newValue, final WriteListener listener,
      final boolean readResponse) {
    SDKPreconditions.checkState(!isClosed(), "The connection is closed");
    SDKPreconditions.checkState(isAuthenticated(), "The device is not authenticated");

    registerWriteListener(listener);
    final byte[] currentValue = characteristic.getValue();
    characteristic.setValue(newValue);
    Logger.v(String.format("Writing value %s to characteristic %s", new String(newValue), characteristic.getUuid().toString()));
    if (!gattController.writeCharacteristic(characteristic, readResponse)) {
      characteristic.setValue(currentValue);
      writeListener.onWriteFailure(WriteListener.Cause.GATT_FAILURE);
      unregisterWriteListener();
    }
  }

  State getState() {
    return state;
  }

  /**
   * Finish write batch.
   */
  void finishWriteBatch() {
    unregisterWriteListener();
  }

  void onDisconnected() {
    serviceStore = null;
    connectionListener.onDisconnected();
  }

  void onDfuModeEnabled() {
    //Not implemented yet
  }

  void onServicesDiscovered(KontaktDeviceServiceStore serviceStore) {
    this.serviceStore = serviceStore;
  }

  void onAuthenticationSuccess(RemoteBluetoothDevice.Characteristics deviceCharacteristics) {
    connectionListener.onAuthenticationSuccess(deviceCharacteristics);
  }

  void onError(final int errorCode) {
    connectionListener.onErrorOccured(errorCode);
  }

  void onFailure(int failureCode) {
    connectionListener.onAuthenticationFailure(failureCode);
  }

  void onConnectionStateChange(final State state) {
    this.state = state;
  }

  void onCharacteristicWritten(final boolean isSuccess, WriteListener.WriteResponse response) {
    if (isSuccess) {
      writeListener.onWriteSuccess(response);
    } else {
      writeListener.onWriteFailure(GATT_FAILURE);
    }
  }

  void onConnected() {
    connectionListener.onConnected();
  }

  void notifyDataSetChanged() {
    connectionListener.onCharacteristicsUpdated(new KontaktDeviceCharacteristics(serviceStore));
  }

  private void registerWriteListener(final WriteListener writeListener) {
    this.writeListener = writeListener;
  }

  private void unregisterWriteListener() {
    this.writeListener = WriteListener.NULL_LISTENER;
  }

    private void validateBeaconPassword(RemoteBluetoothDevice remoteBluetoothDevice) {
        if (FirmwareRevisions.NORMAL_MODE_FIRMWARE_VERSIONS.contains(remoteBluetoothDevice.getFirmwareVersion())) {
            SDKPreconditions.checkNotNull(remoteBluetoothDevice.getPassword(), "Beacon password is null or empty.");
            IBeaconPropertyValidator.validateBeaconPassword(new String(remoteBluetoothDevice.getPassword()), remoteBluetoothDevice.getFirmwareVersion());
        }
    }

  private static <T> KontaktDeviceBatchProcessor.ProcessingListener createProcessingListener(final T batchHolder,
      final WriteBatchListener<T> writeBatchListener) {
    return new KontaktDeviceBatchProcessor.ProcessingListener() {
      @Override
      public void onStart() {
        writeBatchListener.onWriteBatchStart(batchHolder);
      }

      @Override
      public void onError(int errorCode) {
        writeBatchListener.onErrorOccured(errorCode);
      }

      @Override
      public void onFinish() {
        writeBatchListener.onWriteBatchFinish(batchHolder);
      }

      @Override
      public void onRollbackStart() {

      }

      @Override
      public void onRollbackFinish() {

      }

      @Override
      public void onRollbackError(int errorCode) {

      }
    };
  }

  /**
   * State enum describes all possible state that BeaconConnection may be in.
   */
  public enum State {
    /**
     * Reached if the connection with Beacon was
     * established.
     */
    CONNECTED,
    /**
     * Reached if Android device sent authentication password to Beacon.
     */
    AUTHENTICATING,
    /**
     * Reached if Android device is awaiting for all characteristics that has
     * requested.
     */
    CHARACTERISTICS_REQUESTING,
    /**
     * Reached if Authentication ended successfully and all requested
     * characteristics are obtained from Beacon.
     */
    AUTHENTICATED,
    /**
     * Reached if Android device disconnected from Beacon.
     */
    DISCONNECTED
  }
}
