package com.kontakt.sdk.android.ble.configuration.scan;

import com.kontakt.sdk.android.ble.configuration.ActivityCheckConfiguration;
import com.kontakt.sdk.android.ble.configuration.ForceScanConfiguration;
import com.kontakt.sdk.android.ble.configuration.ScanPeriod;
import com.kontakt.sdk.android.ble.rssi.RssiCalculator;
import com.kontakt.sdk.android.ble.rssi.RssiCalculators;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

/**
 * Provides scan-specific parameters influencing scan performance
 * profiles of BLE devices to be found.
 */
public final class ScanContext implements GlobalScanContext {

  /**
   * The constant DEFAULT_DEVICES_UPDATE_CALLBACK_INTERVAL.
   */
  public static final long DEFAULT_DEVICES_UPDATE_CALLBACK_INTERVAL = 3000;

  /**
   * Default settings which works well on most modern devices
   */
  public static final ScanContext DEFAULT = new ScanContext.Builder()
      .setForceScanConfiguration(ForceScanConfiguration.DISABLED)
      .setScanMode(ScanMode.BALANCED)
      .setActivityCheckConfiguration(ActivityCheckConfiguration.MINIMAL)
      .setEddystoneScanContext(EddystoneScanContext.DEFAULT)
      .setIBeaconScanContext(IBeaconScanContext.DEFAULT)
      .setDevicesUpdateCallbackInterval(DEFAULT_DEVICES_UPDATE_CALLBACK_INTERVAL)
      .setScanPeriod(ScanPeriod.RANGING)
      .setRssiCalculator(RssiCalculators.DEFAULT)
      .build();

  private final ForceScanConfiguration forceScanConfiguration;
  private final ScanPeriod scanPeriod;
  private final ScanMode scanMode;
  private final ActivityCheckConfiguration activityCheckConfiguration;
  private final long devicesUpdateCallbackInterval;
  private final RssiCalculator rssiCalculator;

  private final IBeaconScanContext iBeaconScanContext;
  private final EddystoneScanContext eddystoneScanContext;

  private ScanContext(Builder builder) {
    this.forceScanConfiguration = builder.forceScanConfiguration;
    this.scanPeriod = builder.scanPeriod;
    this.scanMode = builder.scanMode;
    this.activityCheckConfiguration = builder.activityCheckConfiguration;
    this.iBeaconScanContext = builder.iBeaconScanContext;
    this.eddystoneScanContext = builder.eddystoneScanContext;
    this.devicesUpdateCallbackInterval = builder.devicesUpdateCallbackInterval;
    this.rssiCalculator = builder.rssiCalculator;
  }

  @Override
  public ActivityCheckConfiguration getActivityCheckConfiguration() {
    return activityCheckConfiguration;
  }

  @Override
  public IBeaconScanContext getIBeaconScanContext() {
    return iBeaconScanContext;
  }

  @Override
  public EddystoneScanContext getEddystoneScanContext() {
    return eddystoneScanContext;
  }

  @Override
  public ForceScanConfiguration getForceScanConfiguration() {
    return forceScanConfiguration;
  }

  @Override
  public ScanPeriod getScanPeriod() {
    return scanPeriod;
  }

  @Override
  public ScanMode getScanMode() {
    return scanMode;
  }

  @Override
  public long getDevicesUpdateCallbackInterval() {
    return devicesUpdateCallbackInterval;
  }

  @Override
  public RssiCalculator getRssiCalculator() {
    return rssiCalculator;
  }

  public static final class Builder {
    private ScanMode scanMode = ScanMode.BALANCED;
    private ForceScanConfiguration forceScanConfiguration = ForceScanConfiguration.DISABLED;
    private ScanPeriod scanPeriod = ScanPeriod.RANGING;
    private ActivityCheckConfiguration activityCheckConfiguration = ActivityCheckConfiguration.MINIMAL;
    private long devicesUpdateCallbackInterval = DEFAULT_DEVICES_UPDATE_CALLBACK_INTERVAL;
    private RssiCalculator rssiCalculator = RssiCalculators.DEFAULT;
    private IBeaconScanContext iBeaconScanContext;
    private EddystoneScanContext eddystoneScanContext;

    public Builder() {

    }

    public Builder setIBeaconScanContext(IBeaconScanContext iBeaconScanContext) {
      this.iBeaconScanContext = iBeaconScanContext;
      return this;
    }

    public Builder setEddystoneScanContext(EddystoneScanContext eddystoneScanContext) {
      this.eddystoneScanContext = eddystoneScanContext;
      return this;
    }

    public Builder setActivityCheckConfiguration(ActivityCheckConfiguration activityCheckConfiguration) {
      SDKPreconditions.checkNotNull(activityCheckConfiguration, "Beacon activity check is null.");

      this.activityCheckConfiguration = activityCheckConfiguration;
      return this;
    }

    /**
     * Sets Scan Mode. Android L comes with functionality
     * controlling scan performance. there are three predefined
     * variables according to which device scan may be handled
     * depending on user's choice.
     * Our SDK takes advantage fo this feature for devices running
     * on Android Lollipop and higher.
     * For devices running on Android OS with version lower than
     * Lollipop this parameter does not affect scan performance.
     *
     * You can set one of three values:
     *
     * {@link ScanMode#BALANCED} - perform Bluetooth LE scan in balanced power mode.
     * {@link ScanMode#LOW_LATENCY} - scan using highest duty cycle.
     * {@link ScanMode#LOW_POWER} - perform Bluetooth LE scan in low power mode.
     *
     * @param scanMode the scan mode
     * @return the builder instance
     */
    public Builder setScanMode(ScanMode scanMode) {
      this.scanMode = scanMode;
      return this;
    }

    /**
     * Sets ForceScanConfiguration ({@link ForceScanConfiguration}).
     *
     * @param forceScanConfiguration the force scan configuration
     * @return the builder instance
     */
    public Builder setForceScanConfiguration(ForceScanConfiguration forceScanConfiguration) {
      SDKPreconditions.checkNotNull(forceScanConfiguration, "By default ForceScanConfiguration is disabled");

      this.forceScanConfiguration = forceScanConfiguration;
      return this;
    }

    /**
     * Sets {@link ScanPeriod}.
     *
     * @param scanPeriod the monitor period
     * @return the builder instance
     */
    public Builder setScanPeriod(ScanPeriod scanPeriod) {
      SDKPreconditions.checkNotNull(scanPeriod, "Monitor period cannot be null");
      ScanContextValidator.validate(scanPeriod);
      this.scanPeriod = scanPeriod;
      return this;
    }

    /**
     * Sets device update callback interval.
     * @param intervalInMillis interval in milliseconds.
     * @return builder instance.
     */
    public Builder setDevicesUpdateCallbackInterval(long intervalInMillis) {
      SDKPreconditions.checkArgument(intervalInMillis > 0, "Interval must be greater than 0");
      this.devicesUpdateCallbackInterval = intervalInMillis;
      return this;
    }

    /**
     * Sets {@link RssiCalculator}.
     * @param rssiCalculator RSSI calculator instance.
     * @return builder instance.
     */
    public Builder setRssiCalculator(RssiCalculator rssiCalculator) {
      SDKPreconditions.checkNotNull(rssiCalculator, "RssiCalculator can't be null");
      this.rssiCalculator = rssiCalculator;
      return this;
    }

    /**
     * Builds global scan context instance.
     *
     * @return new immutable configuration instance
     */
    public ScanContext build() {

      if (activityCheckConfiguration != ActivityCheckConfiguration.DISABLED) {
        ScanContextValidator.validate(activityCheckConfiguration);
      }

      if (forceScanConfiguration != ForceScanConfiguration.DISABLED) {
        ScanContextValidator.validate(forceScanConfiguration);
      }
      String message = "At least one of: [IBeaconScanContext, EddystoneScanContext] must be set";
      SDKPreconditions.checkNotAllNull(new IllegalStateException(message), iBeaconScanContext, eddystoneScanContext);

      return new ScanContext(this);
    }
  }

}
