package com.kontakt.sdk.android.ble.configuration.scan;

import com.kontakt.sdk.android.ble.device.BeaconRegion;
import com.kontakt.sdk.android.ble.filter.ibeacon.IBeaconFilter;
import com.kontakt.sdk.android.common.profile.IBeaconRegion;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

/**
 * Provides description of conditions accordin to which Android device searches IBeacon device and notifies
 * about their presence. The context provides plenty of customizations
 * including:
 * searching by regions,
 * filters for IBeacon advertising packets,
 * selection of desired event types (for more information see {@link com.kontakt.sdk.android.ble.discovery.EventType}),
 * rssi signal manipulation,
 * sorting devices by their distance from Android device,
 * notifying events with updated device with specific interval
 */
public final class IBeaconScanContext extends AbstractProfileSpecificScanContext<IBeaconRegion> {

    /**
     * The DEFAULT IBeacon Scan Context includes all event types to be notified and no filters.
     * Scan with the default predefined {@link IBeaconScanContext} notifies about every iBeacon
     * presence regardless of its belonging to any specific Region.
     */
    public static final IBeaconScanContext DEFAULT = new IBeaconScanContext.Builder().build();

    private final List<IBeaconFilter> iBeaconFilters;
    private final Set<IBeaconRegion> iBeaconRegions;

    private IBeaconScanContext(Builder builder) {
        super(builder);

        ScanContextValidator.validateIBeaconRegionsCount(builder.iBeaconRegions);
        ScanContextValidator.validateIBeaconFiltersCount(builder.iBeaconFilterList);

        this.iBeaconRegions = Collections.unmodifiableSet(new HashSet<IBeaconRegion>(builder.iBeaconRegions));
        this.iBeaconFilters = Collections.unmodifiableList(new LinkedList<IBeaconFilter>(builder.iBeaconFilterList));
    }

    @Override
    public Collection<IBeaconRegion> getSpaces() {
        return iBeaconRegions;
    }

    /**
     * Provides IBeacon filters.
     *
     * @return the filters
     */
    public Collection<IBeaconFilter> getFilters() {
        return iBeaconFilters;
    }

    /**
     * Specifies the context within which the Android device should scan iBeacon devices.
     */
    public static final class Builder extends AbstractProfileSpecificScanContext.Builder<IBeaconScanContext, Builder> {

        private List<IBeaconFilter> iBeaconFilterList = new LinkedList<IBeaconFilter>();
        private Set<IBeaconRegion> iBeaconRegions = new HashSet<IBeaconRegion>();

        /**
         * Sets filters accepting IBeacons only with specific properties.
         * <p>
         * Note: all filters will be executed on the received data from beacon device.
         * <br>
         * So if you have shuffled devices some filters will not work anymore
         * <br>
         * e.g {@link com.kontakt.sdk.android.ble.filter.ibeacon.IBeaconUniqueIdFilter}
         * <br>
         *
         *
         * @param filters the filters
         * @return the builder
         */
        public Builder setIBeaconFilters(Collection<? extends IBeaconFilter> filters) {
            SDKPreconditions.checkNotNull(filters, "Filters are null.");
            this.iBeaconFilterList.addAll(filters);
            return this;
        }

        /**
         * Sets accepted regions
         * <p>
         * Note: regions are checked on data received from beacon device
         * <br>
         * So if you have shuffled devices this regions must be the same as real values received from beacon device
         * <br>
         * <p>
         *
         *
         * @param iBeaconRegions the regions
         * @return the builder
         */
        public Builder setIBeaconRegions(final Collection<IBeaconRegion> iBeaconRegions) {
            SDKPreconditions.checkNotNull(iBeaconRegions, "Regions collection is null");
            this.iBeaconRegions.addAll(iBeaconRegions);
            return this;
        }

        /**
         * Builds {@link IBeaconScanContext}
         * @return IBeaconScanContext
         */
        @Override
        public IBeaconScanContext build() {
            if (iBeaconRegions.isEmpty()) {
                iBeaconRegions.add(BeaconRegion.EVERYWHERE);
            }

            return new IBeaconScanContext(this);
        }

        @Override
        protected Builder getExtension() {
            return this;
        }
    }
}
