package com.kontakt.sdk.android.ble.cache;

import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceEvent;
import com.kontakt.sdk.android.ble.discovery.EventType;
import com.kontakt.sdk.android.ble.discovery.eddystone.EddystoneDeviceEvent;
import com.kontakt.sdk.android.common.profile.IEddystoneDevice;
import com.kontakt.sdk.android.common.profile.IEddystoneNamespace;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.List;

class EddystoneResolveRequest {
    private final IEddystoneDevice fakeDevice;
    private final Integer sourceProximityManagerId;
    private final IEddystoneNamespace namespace;
    private final long timestamp;
    private ResolverType resolvedBy = ResolverType.NONE;
    private ResolveRequestStatus status;

    public EddystoneResolveRequest(Integer sourceProximityManagerId, IEddystoneDevice eddystoneDevice, IEddystoneNamespace namespace, long timestamp) {
        this.sourceProximityManagerId = sourceProximityManagerId;
        this.fakeDevice = eddystoneDevice;
        this.namespace = namespace;
        this.timestamp = timestamp;
    }

    public IEddystoneDevice getFakeDevice() {
        return fakeDevice;
    }

    public Integer getSourceProximityManagerId() {
        return sourceProximityManagerId;
    }

    public IEddystoneNamespace getNamespace() {
        return namespace;
    }

    public long getTimestamp() {
        return timestamp;
    }

    public ResolverType getResolvedBy() {
        return resolvedBy;
    }

    public void setResolvedBy(ResolverType resolvedBy) {
        this.resolvedBy = resolvedBy;
    }

    public void setStatus(ResolveRequestStatus status) {
        this.status = status;
    }

    public ResolveRequestStatus getStatus() {
        return status;
    }

    @Override
    public boolean equals(Object o) {
        if (o == null || !(o instanceof EddystoneResolveRequest)) {
            return false;
        }

        if (o == this) {
            return true;
        }

        EddystoneResolveRequest resolveRequest = (EddystoneResolveRequest) o;
        return SDKEqualsBuilder.start()
                .equals(fakeDevice, resolveRequest.fakeDevice)
                .equals(sourceProximityManagerId, resolveRequest.sourceProximityManagerId)
                .equals(namespace, resolveRequest.namespace)
                .equals(timestamp, resolveRequest.timestamp)
                .equals(resolvedBy, resolveRequest.resolvedBy)
                .result();
    }

    @Override
    public int hashCode() {
        return HashCodeBuilder.init()
                .append(fakeDevice)
                .append(sourceProximityManagerId)
                .append(namespace)
                .append(timestamp)
                .append(resolvedBy)
                .build();
    }

    static EddystoneResolveRequest of(int sourceProximityManagerId, BluetoothDeviceEvent bluetoothDeviceEvent) {
        EddystoneDeviceEvent iBeaconDeviceEvent = (EddystoneDeviceEvent) bluetoothDeviceEvent;
        EventType eventType = iBeaconDeviceEvent.getEventType();
        SDKPreconditions.checkArgument(EventType.DEVICE_DISCOVERED == eventType, "Cannot create resolve request for other event type than DEVICE_DISCOVERED!");
        List<IEddystoneDevice> discoveredDeviceList = iBeaconDeviceEvent.getDeviceList();
        SDKPreconditions.checkState(discoveredDeviceList.size() == 1, "Resolve request must be created for each one discovered device!");

        int proximityManagerId = sourceProximityManagerId;
        IEddystoneDevice beaconDevice = discoveredDeviceList.get(0);
        IEddystoneNamespace namespace = iBeaconDeviceEvent.getNamespace();
        long timestamp = bluetoothDeviceEvent.getTimestamp();

        return new EddystoneResolveRequest(proximityManagerId, beaconDevice, namespace, timestamp);
    }
}
