package com.kontakt.sdk.android.ble.util;

import android.annotation.TargetApi;
import android.os.Build;
import android.util.SparseLongArray;

import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

@TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
public class SafeSparseLongArray extends SparseLongArray {

    private final Lock writeLock;

    private final Lock readLock;

    {
        final ReadWriteLock readWriteLock = new ReentrantReadWriteLock();
        writeLock = readWriteLock.writeLock();
        readLock = readWriteLock.readLock();
    }

    public SafeSparseLongArray() {
        super();
    }

    public SafeSparseLongArray(int initialCapacity) {
        super(initialCapacity);
    }

    @Override
    public SparseLongArray clone() {
        try {
            readLock.lock();
            return super.clone();
        } finally {
            unlockSafely(readLock);
        }
    }

    @Override
    public long get(int key, long valueIfKeyNotFound) {
        try {
            readLock.lock();
            return super.get(key, valueIfKeyNotFound);
        } finally {
            unlockSafely(readLock);
        }
    }

    @Override
    public void delete(int key) {
        try {
            writeLock.lock();
            super.delete(key);
        } finally {
            unlockSafely(writeLock);
        }
    }

    @Override
    public void removeAt(int index) {
        try {
            writeLock.lock();
            super.removeAt(index);
        } finally {
            unlockSafely(writeLock);
        }
    }

    @Override
    public void put(int key, long value) {
        try {
            writeLock.lock();
            super.put(key, value);
        } finally {
            unlockSafely(writeLock);
        }
    }

    @Override
    public int size() {
        try {
            readLock.lock();
            return super.size();
        } finally {
            unlockSafely(readLock);
        }
    }

    @Override
    public int keyAt(int index) {
        try {
            readLock.lock();
            return super.keyAt(index);
        } finally {
            unlockSafely(readLock);
        }
    }

    @Override
    public long valueAt(int index) {
        try {
            readLock.lock();
            return super.valueAt(index);
        } finally {
            unlockSafely(readLock);
        }
    }

    @Override
    public int indexOfKey(int key) {
        try {
            readLock.lock();
            return super.indexOfKey(key);
        } finally {
            unlockSafely(readLock);
        }
    }

    @Override
    public int indexOfValue(long value) {
        try {
            readLock.lock();
            return super.indexOfValue(value);
        } finally {
            unlockSafely(readLock);
        }
    }

    @Override
    public void clear() {
        try {
            writeLock.lock();
            super.clear();
        } finally {
            unlockSafely(writeLock);
        }
    }

    @Override
    public void append(int key, long value) {
        try {
            writeLock.lock();
            super.append(key, value);
        } finally {
            unlockSafely(writeLock);
        }
    }

    private static void unlockSafely(final Lock lock) {
        try {
            lock.unlock();
        } catch (Throwable ignored) {
        }
    }
}
