package com.kontakt.sdk.android.ble.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

/**
 * Replacing list behaves as normal ArrayList. Additionally it allows to replace
 * value provided that it already exists in the list.
 *
 * The list provides thread-safe implementation for most of the methods including:
 * {@link #add(Object)},{@link #addOrReplace(Object)}, {@link #add(Object)},
 * {@link #contains(Object)}, {@link #containsAll(Collection)}, {@link #remove(int)},
 * {@link #removeAll(Collection)}, {@link #indexOf(Object)}, {@link #retainAll(Collection)}.
 *
 *
 * @param <T>  generic parameter type
 */
public class ReplacingArrayList<T> extends ArrayList<T> {

    private final Lock readLock;

    private final Lock writeLock;

    {
        final ReadWriteLock readWriteLock = new ReentrantReadWriteLock();
        readLock = readWriteLock.readLock();
        writeLock = readWriteLock.writeLock();
    }

    /**
     * Add or replace new value.
     *
     * @param object the object
     * @return the boolean indicating whether new value was added (true) or
     * replaced (false).
     * Method is thread-safe.
     */
    public boolean addOrReplace(final T object) {
        try {
            writeLock.lock();
            final int index = super.indexOf(object);
            if (index == -1) {
                return super.add(object);
            } else {
                super.set(index, object);
                return false;
            }
        } finally {
            unlockSafely(writeLock);
        }
    }

    @Override
    public int indexOf(Object object) {
        try {
            readLock.lock();
            return super.indexOf(object);
        } finally {
            unlockSafely(readLock);
        }
    }

    @Override
    public boolean add(T object) {
        try {
            writeLock.lock();
            return super.add(object);
        } finally {
            unlockSafely(writeLock);
        }
    }

    @Override
    public T set(int index, T object) {
        try {
            writeLock.lock();
            return super.set(index, object);
        } finally {
            unlockSafely(writeLock);
        }
    }

    @Override
    public boolean contains(Object object) {
        try {
            readLock.lock();
            return super.contains(object);
        } finally {
            unlockSafely(readLock);
        }
    }

    @Override
    public boolean containsAll(Collection<?> collection) {
        try {
            readLock.lock();
            Iterator<?> it = collection.iterator();
            while (it.hasNext()) {
                if (!super.contains(it.next())) {
                    return false;
                }
            }
            return true;
        } finally {
            unlockSafely(readLock);
        }
    }

    @Override
    public T remove(int index) {
        try {
            writeLock.lock();
            return super.remove(index);
        } finally {
            unlockSafely(writeLock);
        }
    }

    @Override
    public boolean retainAll(Collection<?> collection) {
        try {
            writeLock.lock();
            return super.retainAll(collection);
        } finally {
           unlockSafely(writeLock);
        }
    }

    @Override
    public boolean removeAll(Collection<?> collection) {
        try {
            writeLock.lock();
            boolean result = false;
            Iterator<?> it = iterator();
            while (it.hasNext()) {
                if (collection.contains(it.next())) {
                    it.remove();
                    result = true;
                }
            }
            return result;
        } finally {
            unlockSafely(writeLock);
        }
    }

    private static void unlockSafely(final Lock lock) {
        try {
            lock.unlock();
        } catch(Throwable ignored) { }
    }
}
