package com.kontakt.sdk.android.ble.service;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.le.BluetoothLeScanner;
import android.bluetooth.le.ScanCallback;
import android.bluetooth.le.ScanFilter;
import android.bluetooth.le.ScanSettings;
import android.os.Build;
import android.os.Messenger;

import com.kontakt.sdk.android.ble.configuration.ForceScanConfiguration;
import com.kontakt.sdk.android.ble.configuration.scan.ScanContext;
import com.kontakt.sdk.android.ble.manager.ProximityManager;
import com.kontakt.sdk.android.common.log.Logger;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

final class ScanCompat {

    private static final List<ScanFilter> EMPTY_SCAN_FILTERS = new ArrayList<ScanFilter>();

    private ScanCompat() {
    }


    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    static ScanConfiguration createScanConfiguration(ListenerAccessor listenerAccessor,
                                                     ScanContext scanContext,
                                                     Messenger messenger) {

        final BleScanCallback scanCallback = Callbacks.newCallback(
                new Callbacks.Configuration.Builder()
                        .addMonitoringListeners(listenerAccessor.getMonitoringListeners())
                        .setScanContext(scanContext)
                        .setServiceMessenger(messenger)
                        .build());

        if(isLollipopOrHigher()) {
            final ScanSettings scanSettings = new ScanSettings.Builder()
                    .setScanMode(scanContext.getScanMode())
                    .build();

            final BleScanCallback callbackL = CallbacksL.newCallback(scanCallback);

            return new AbstractScanConfigurationL(scanContext, callbackL) {
                @Override
                public ScanSettings getScanSettings() {
                    return scanSettings;
                }
            };
        } else {
            return new AbstractScanConfiguration(scanContext, scanCallback) { };
        }
    }

    static ScanController createScanController(ScanConfiguration configuration, ForceScanScheduler forceScanScheduler) {

        return isLollipopOrHigher() ?
                new ScanController.Builder()
                        .setScanPeriod(configuration.getScanContext().getScanPeriod())
                        .setScanActiveRunner(RunnersL.newRunner(Runners.RunnerType.MONITOR_ACTIVE_RUNNER, configuration))
                        .setScanPassiveRunner(RunnersL.newRunner(Runners.RunnerType.MONITOR_PASSIVE_RUNNER, configuration))
                        .setForceScanScheduler(forceScanScheduler)
                        .build()
                :
                new ScanController.Builder()
                        .setScanActiveRunner(Runners.newRunner(Runners.RunnerType.MONITOR_ACTIVE_RUNNER, configuration))
                        .setScanPassiveRunner(Runners.newRunner(Runners.RunnerType.MONITOR_PASSIVE_RUNNER, configuration))
                        .setScanPeriod(configuration.getScanContext().getScanPeriod())
                        .setForceScanScheduler(forceScanScheduler)
                        .build();
    }

    static ForceScanScheduler createForceScanScheduler(final ScanConfiguration configuration) {
        final ScanContext scanContext = configuration.getScanContext();

        if (scanContext.getForceScanConfiguration() == ForceScanConfiguration.DISABLED) {
            return ForceScanScheduler.DISABLED;
        }

        if(isLollipopOrHigher()) {
            return new ForceScanScheduler(RunnersL.newRunner(Runners.RunnerType.FORCE_SCAN_RUNNER, configuration));
        } else {
            return new ForceScanScheduler(Runners.newRunner(Runners.RunnerType.FORCE_SCAN_RUNNER, configuration));
        }
    }

    @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
    static void onScanStart(final ScanConfiguration configuration) {
        if(isLollipopOrHigher()) {
            final ScanConfigurationL scanConfiguration = (ScanConfigurationL) configuration;

            final ScanSettings scanSettings = scanConfiguration.getScanSettings();
            final ScanCallback scanCallback = (ScanCallback) scanConfiguration.getScanCallback();

            final BluetoothLeScanner scanner = getScanner();

            if (scanner != null) {
                scanner.startScan(EMPTY_SCAN_FILTERS, scanSettings, scanCallback);
            } else {
                Logger.e("onRangingStart(): BluetoothLeScanner is null.");
            }
        } else {
            final BluetoothAdapter.LeScanCallback scanCallback = configuration.getScanCallback();
            final BluetoothAdapter adapter = BluetoothAdapter.getDefaultAdapter();
            if (adapter != null) {
                adapter.startLeScan(scanCallback);
            } else {
                Logger.d("Scan start requested but Bluetooth Adapter is null.");
            }
        }
    }

    static void onScanStop(BleScanCallback callback) {
        if(isLollipopOrHigher()) {
            if (callback != null) {
                final ScanCallback scanCallback = (ScanCallback) callback;
                final BluetoothLeScanner scanner = getScanner();
                if (scanner != null) {
                    scanner.flushPendingScanResults(scanCallback);
                    scanner.stopScan(scanCallback);
                }
            }
        } else {
            final BluetoothAdapter adapter = BluetoothAdapter.getDefaultAdapter();
            if (adapter != null) {
                adapter.stopLeScan(callback);
            }
        }
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    private static BluetoothLeScanner getScanner() {
        BluetoothAdapter adapter = BluetoothAdapter.getDefaultAdapter();

        if (adapter != null) {
            return adapter.getBluetoothLeScanner();
        }

        return null;
    }

    private static boolean isLollipopOrHigher() {
        return Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP;
    }

    private static abstract class AbstractScanConfiguration implements ScanConfiguration {

        private final ScanContext scanContext;

        private final BleScanCallback scanCallback;

        protected AbstractScanConfiguration(final ScanContext scanContext,
                                            final BleScanCallback scanCallback) {
            this.scanCallback = scanCallback;
            this.scanContext = scanContext;
        }

        @Override
        public BleScanCallback getScanCallback() {
            return scanCallback;
        }

        @Override
        public void addListener(ProximityManager.ProximityListener proximityListener) {
            scanCallback.addListener(proximityListener);
        }

        @Override
        public void removeListener(ProximityManager.ProximityListener proximityListener) {
            scanCallback.removeListener(proximityListener);
        }

        @Override
        public void close() throws IOException {
            scanCallback.close();
        }

        @Override
        public ScanContext getScanContext() {
            return scanContext;
        }
    }

    private static abstract class AbstractScanConfigurationL extends AbstractScanConfiguration implements ScanConfigurationL {

        protected AbstractScanConfigurationL(ScanContext scanContext, BleScanCallback scanCallback) {
            super(scanContext, scanCallback);
        }
    }
}
