package com.kontakt.sdk.android.ble.service;

import android.bluetooth.BluetoothDevice;
import android.os.Message;
import android.os.Messenger;
import android.os.RemoteException;

import com.kontakt.sdk.android.ble.configuration.ActivityCheckConfiguration;
import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceDiscoverer;
import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceEvent;
import com.kontakt.sdk.android.ble.discovery.DiscoveryContract;
import com.kontakt.sdk.android.ble.manager.ProximityManager;
import com.kontakt.sdk.android.common.KontaktSDK;
import com.kontakt.sdk.android.common.log.Logger;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

abstract class MonitorCallback implements BleScanCallback, DiscoveryContract {

    private static final String TAG = MonitorCallback.class.getSimpleName();

    protected final BluetoothDeviceDiscoverer bluetoothDeviceDiscoverer;

    private final Messenger serviceMessenger;

    private ScheduledExecutorService periodicActivityCheckExecutor;

    private final Map<Integer, ProximityManager.ProximityListener> monitoringListenerMap;

    private final ActivityCheckConfiguration activityCheckConfiguration;

    MonitorCallback(Callbacks.Configuration configuration) {
        bluetoothDeviceDiscoverer = new DefaultBluetoothDeviceDiscoverer(configuration.scanContext, this);
        serviceMessenger = configuration.messenger;
        monitoringListenerMap = new ConcurrentHashMap<Integer, ProximityManager.ProximityListener>();
        activityCheckConfiguration = configuration.scanContext.getActivityCheckConfiguration();

        addRegisteredListeners(configuration);
    }

    @Override
    public void onLeScan(BluetoothDevice device, int rssi, byte[] scanRecord) {
        if (!isAnyListenerRegistered() && KontaktSDK.isInitialized()) {
            return;
        }

        if (device == null) {
            Logger.d("Remote device discovered but is null");
            return;
        }

        Logger.d(String.format("Device discovered %s", device.toString()));

        final boolean isDeviceDiscovered = bluetoothDeviceDiscoverer.performDiscovery(device, rssi, scanRecord);

        if (!isDeviceDiscovered) {
            Logger.d(String.format("Device found but not recognized as Beacon [name: %s, address: %s]",
                    device.getName(),
                    device.getAddress()));
        }
    }

    @Override
    public void close() {
        bluetoothDeviceDiscoverer.disable();
        monitoringListenerMap.clear();
    }

    @Override
    public Collection<ProximityManager.ProximityListener> getMonitoringListeners() {
        return Collections.unmodifiableCollection(new ArrayList<ProximityManager.ProximityListener>(monitoringListenerMap.values()));
    }

    @Override
    public void addListener(ProximityManager.ProximityListener proximityListener) {
        final int hashCode = System.identityHashCode(proximityListener);
        final ProximityManager.ProximityListener insertion = monitoringListenerMap.put(hashCode, proximityListener);

        if (insertion == null) {
            Logger.d("MonitoringListener registered: " + hashCode);
        }
    }

    @Override
    public void removeListener(ProximityManager.ProximityListener proximityListener) {
        final int hashCode = System.identityHashCode(proximityListener);
        final ProximityManager.ProximityListener deletion = monitoringListenerMap.remove(hashCode);

        if (deletion != null) {
            Logger.d("MonitoringListener unregistered: " + hashCode);
        }
    }

    @Override
    public void onScanStarted() {
        for (final ProximityManager.ProximityListener proximityListener : monitoringListenerMap.values()) {
            proximityListener.onScanStart();
        }

        startPeriodicInactivityCheck();
    }

    @Override
    public void onEvent(BluetoothDeviceEvent event) {
        for (final ProximityManager.ProximityListener proximityListener : monitoringListenerMap.values()) {
            proximityListener.onEvent(event);
        }
    }

    @Override
    public void onScanStopped() {
        for (final ProximityManager.ProximityListener proximityListener : monitoringListenerMap.values()) {
            proximityListener.onScanStop();
        }
        stopPeriodicInactivityCheck();
    }

    void notifyScanStarted() {
        sendMessage(ProximityService.MESSAGE_SCAN_STARTED);
    }

    void notifyScanStopped() {
        sendMessage(ProximityService.MESSAGE_SCAN_STOPPED);
    }

    protected boolean sendMessage(final int messageCode) {
        try {
            serviceMessenger.send(Message.obtain(null, messageCode));
            return true;
        } catch (RemoteException e) {
            Logger.e(TAG + " Error occures when sending message", e);
            return false;
        }
    }

    private boolean isAnyListenerRegistered() {
        return !monitoringListenerMap.isEmpty();
    }

    private void addRegisteredListeners(final Callbacks.Configuration configuration) {
        for (ProximityManager.ProximityListener proximityListener : configuration.proximityListeners) {
            addListener(proximityListener);
        }
    }

    private void startPeriodicInactivityCheck() {

        if (activityCheckConfiguration == ActivityCheckConfiguration.DISABLED) {
            return;
        }

        final long checkPeriod = activityCheckConfiguration.getCheckPeriod();

        periodicActivityCheckExecutor = Executors.newSingleThreadScheduledExecutor();
        periodicActivityCheckExecutor.scheduleAtFixedRate(new Runnable() {
            @Override
            public void run() {
                try {
                    bluetoothDeviceDiscoverer.evictInactiveDevices();
                } catch (Exception e) {
                    Logger.e(TAG + "Error during evictiong", e);
                }
            }
        }, checkPeriod, checkPeriod, TimeUnit.MILLISECONDS);
    }

    private void stopPeriodicInactivityCheck() {
        if (activityCheckConfiguration == ActivityCheckConfiguration.DISABLED) {
            return;
        }

        periodicActivityCheckExecutor.shutdown();
    }
}
