package com.kontakt.sdk.android.ble.rssi;

import com.kontakt.sdk.android.ble.math.Mean;
import com.kontakt.sdk.android.common.util.ConversionUtils;

import java.util.LinkedList;
import java.util.Map;
import java.util.Queue;
import java.util.WeakHashMap;

/**
 * Limited Mean Rssi Calculator provides rssi calculation based on the
 * arithmethic mean of n last rssi records where the n is the parameter
 * applied during calculator construction.
 */
public final class LimitedMeanRssiCalculator implements RssiCalculator {

    private static final Mean MEAN = new Mean();

    private final Map<Integer, Queue<Double>> rssiMapLog = new WeakHashMap<Integer, Queue<Double>>();
    private final int logSize;

    LimitedMeanRssiCalculator(final int logSize) {
        this.logSize = logSize;
    }

    @Override
    public double calculateRssi(final int uniqueKey, int rssi) {
        Queue<Double> rssiQueue = rssiMapLog.get(uniqueKey);
        if(rssiQueue == null) {
            rssiQueue = new LinkedList<Double>();
            rssiMapLog.put(uniqueKey, rssiQueue);
        }

        if(rssiQueue.size() >= logSize) {
            rssiQueue.poll();
        }

        rssiQueue.add((double) rssi);

        return MEAN.evaluate(ConversionUtils.toPrimitive(rssiQueue.toArray(new Double[rssiQueue.size()])));
    }

    @Override
    public void clear() {
        rssiMapLog.clear();
    }

    @Override
    public void clear(int uniqueKey) {
        rssiMapLog.remove(uniqueKey);
    }
}