package com.kontakt.sdk.android.ble.manager;

import com.kontakt.sdk.android.ble.configuration.scan.ScanContext;
import com.kontakt.sdk.android.ble.connection.OnServiceReadyListener;

/**
 * Contract interface representing Proximity Manager programming interface.
 */
public interface ProximityManagerContract {

    /**
     * Gets proximity manager id.
     *
     * @return the id
     */
    int getId();

    /**
     * Connects to backing Service.
     *
     * @param listener the listener
     */
    void connect(OnServiceReadyListener listener);

    /**
     * Provides information about connection state with backing service.
     *
     * @return the true if connection with backing service is established.
     */
    boolean isConnected();

    /**
     * Disconnects from backing service.
     */
    void disconnect();

    /**
     * Provides information whether the ProximityManager is currently scanning.
     *
     * @return true if the scan is active
     */
    boolean isScanning();

    /**
     * Initializes scan
     *
     * @param scanContext the scan context
     * @return the true if scan initialization request is sent. False if either the connection with
     * backing service is not established or the applied scan context is same as the one according
     * to which scan has been initialized.
     */
    boolean initializeScan(ScanContext scanContext);

    /**
     * Initializes scan and connects to backing service if the connection is not established.
     *
     * @param scanContext            the scan context
     * @param onServiceReadyListener the on service ready listener
     */
    void initializeScan(ScanContext scanContext, OnServiceReadyListener onServiceReadyListener);

    /**
     * Restarts scan.
     *
     * @param scanContext the scan context
     * @return true if scan restart request has been sent to the backing service. False
     * if connection to the backing service is not established.
     */
    boolean restartScan(ScanContext scanContext);

    /**
     * Restart scan.
     *
     * @param scanContext            the scan context
     * @param onServiceReadyListener the on service ready listener
     */
    void restartScan(ScanContext scanContext, OnServiceReadyListener onServiceReadyListener);

    /**
     * Attaches listener.
     *
     * @param proximityListener the proximity listener
     * @return true if attach request is sent. False if connection to the backing service is not
     * established.
     */
    boolean attachListener(ProximityManager.ProximityListener proximityListener);

    /**
     * Detaches listener.
     *
     * @param proximityListener the proximity listener
     * @return true if detach request is sent. False is connection to the backing service is not
     * established.
     */
    boolean detachListener(ProximityManager.ProximityListener proximityListener);

    /**
     * Finishes scan.
     *
     * @return true if finish scan request is sent. False if connection to the backing service
     * is not established.
     */
    boolean finishScan();

    /**
     * Clears resolved beacon cache
     * <br/>
     * Available only in KontaktProximityManager
     */
    void clearCache();

    /**
     * Clears buffers resolved beacon cache
     * <br/>
     * Available only in KontaktProximityManager
     */
    void clearBuffers();
}
