package com.kontakt.sdk.android.ble.manager;

import android.Manifest;
import android.annotation.TargetApi;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.Build;
import android.os.Handler;
import android.os.IBinder;
import android.os.Message;
import android.os.Messenger;
import android.os.RemoteException;

import com.kontakt.sdk.android.ble.configuration.scan.ScanContext;
import com.kontakt.sdk.android.ble.connection.OnServiceReadyListener;
import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceEvent;
import com.kontakt.sdk.android.ble.service.ProximityService;
import com.kontakt.sdk.android.common.interfaces.SDKSupplier;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.util.SDKPreconditions;


/**
 * Provides scan management API according to which the Android device may scan remote bluetooth devices.
 */
public class ProximityManager extends AbstractServiceConnector implements ProximityManagerContract {

    /**
     * Perform Bluetooth LE scan in low power mode. This is the default scan mode as it consumes the
     * least power.
     */
    public static final int SCAN_MODE_LOW_POWER = 0;

    /**
     * Perform Bluetooth LE scan in balanced power mode. Scan results are returned at a rate that
     * provides a good trade-off between scan frequency and power consumption.
     */
    public static final int SCAN_MODE_BALANCED = 1;

    /**
     * Scan using highest duty cycle. It's recommended to only use this mode when the application is
     * running in the foreground.
     */
    public static final int SCAN_MODE_LOW_LATENCY = 2;

    /**
     * The constant DEFAULT_DEVICES_UPDATE_CALLBACK_INTERVAL.
     */
    public static final long DEFAULT_DEVICES_UPDATE_CALLBACK_INTERVAL = 100;

    private static final String TAG = ProximityManager.class.getSimpleName();

    private final int id;

    private Context context;

    private ServiceConnection serviceConnection;

    private Messenger serviceMessenger;

    private Messenger managerMessenger;

    private boolean isScanning;

    /**
     * Instantiates a new Proximity manager.
     *
     * @param ctx the ctx
     */
    public ProximityManager(final Context ctx) {
        super(ctx, new String[]{
                Manifest.permission.BLUETOOTH,
                Manifest.permission.BLUETOOTH_ADMIN
        });
        this.context = ctx.getApplicationContext();
        this.managerMessenger = new Messenger(new ManagerHandler(this));
        this.id = System.identityHashCode(this);
    }

    @Override
    public int getId() {
        return id;
    }

    @TargetApi(Build.VERSION_CODES.ICE_CREAM_SANDWICH_MR1)
    @Override
    /**
     * {@inheritDoc}
     * The method is synchronized.
     */
    public synchronized void connect(final OnServiceReadyListener onServiceReadyistener) {
        if (isConnected()) {
            Logger.d("ProximityManager already connected to BeaconService.");
            return;
        }

        SDKPreconditions.checkNotNull(onServiceReadyistener, "OnServiceBoundListener is null.");

        checkPermissions();


        final Intent serviceIntent = new Intent(context, ProximityService.class);

        serviceConnection = new ServiceConnection() {
            @Override
            public void onServiceConnected(ComponentName name, IBinder serviceBinder) {
                Logger.d(TAG + ": Beacon Service connected.");

                final SDKSupplier<Messenger> messengerSupplier = (SDKSupplier<Messenger>) serviceBinder;

                serviceMessenger = messengerSupplier.get();

                onServiceReadyistener.onServiceReady();
            }

            @Override
            public void onServiceDisconnected(ComponentName name) {
                Logger.e(TAG + ": disconnected from Beacon Service");
            }
        };

        final boolean isBindRequestSent = context.bindService(serviceIntent,
                serviceConnection,
                Context.BIND_AUTO_CREATE);

        if (!isBindRequestSent) {
            final String serviceName = getClass().getSimpleName();
            Logger.i(String.format("Could not connect to %s. Please check if the %s is registered in AndroidManifest.xml",
                    serviceName,
                    serviceName));

            onServiceReadyistener.onConnectionFailure();
        }

    }

    @Override
    /**
     * {@inheritDoc}
     * The method is synchronized.
     */
    public synchronized boolean isConnected() {
        return serviceConnection != null && serviceMessenger != null;
    }

    @Override
    /**
     * Start ranging for the desired set of regions ({@link BeaconRegion})
     * Invoking this method more than once does not restart ranging but continues with
     * existing configuration.
     * To restart restart ranging you need to call #restartRanging();
     *
     * @param scanContext - wrapper to region and namespace set
     * @throws RemoteException - the remote exception is thrown when ranging
     *                         start failes.
     */
    public synchronized boolean initializeScan(final ScanContext scanContext) {
        SDKPreconditions.checkNotNull(scanContext, "ScanContext cannot be null");
        return startScanIfConnected(scanContext, ProximityService.MESSAGE_INITIALIZE_SCAN);
    }

    @Override
    public void initializeScan(final ScanContext scanContext, final OnServiceReadyListener onServiceReadyListener) {
        SDKPreconditions.checkNotNull(scanContext, "ScanContext cannot be null");
        SDKPreconditions.checkNotNull(onServiceReadyListener, "onServiceReadyListener is null");
        connectIfNeededAndStartScan(scanContext, onServiceReadyListener, ProximityService.MESSAGE_INITIALIZE_SCAN);
    }

    @Override
    public synchronized boolean restartScan(final ScanContext scanContext) {
        SDKPreconditions.checkNotNull(scanContext, "ScanContext cannot be null");
        return startScanIfConnected(scanContext, ProximityService.MESSAGE_RESTART_SCAN);
    }

    private void connectIfNeededAndStartScan(final ScanContext scanContext,
                                             final OnServiceReadyListener onServiceReadyListener,
                                             final int messageCode) {
        if (isConnected()) {
            final boolean isScanStartScheduled = startScanIfConnected(scanContext, messageCode);
            if (isScanStartScheduled) {
                onServiceReadyListener.onServiceReady();
            } else {
                onServiceReadyListener.onConnectionFailure();
            }
        } else {
            connect(new OnServiceReadyListener() {
                @Override
                public void onServiceReady() {
                    initializeScan(scanContext);
                    onServiceReadyListener.onServiceReady();
                }

                @Override
                public void onConnectionFailure() {
                    onServiceReadyListener.onConnectionFailure();
                }
            });
        }
    }

    @Override
    public void restartScan(final ScanContext scanContext, final OnServiceReadyListener onServiceReadyListener) {
        SDKPreconditions.checkNotNull(scanContext, "ScanContext cannot be null");
        SDKPreconditions.checkNotNull(onServiceReadyListener, "onServiceReadyListener is null");
        connectIfNeededAndStartScan(scanContext, onServiceReadyListener, ProximityService.MESSAGE_RESTART_SCAN);
    }

    @Override
    public synchronized boolean attachListener(final ProximityListener proximityListener) {
        SDKPreconditions.checkNotNull(proximityListener, "Proximity listener is null");
        final Message message = createMessage(ProximityService.MESSAGE_ATTACH_MONITORING_LISTENER, proximityListener);
        return sendMessage(message);
    }

    @Override
    public synchronized boolean detachListener(final ProximityListener proximityListener) {
        SDKPreconditions.checkNotNull(proximityListener, "Proximity listener is null");
        final Message message = createMessage(ProximityService.MESSAGE_DETACH_MONITORING_LISTENER, proximityListener);
        return sendMessage(message);
    }

    private boolean sendMessage(final Message message) {
        if (isConnected()) {
            try {
                serviceMessenger.send(message);
                return true;
            } catch (RemoteException e) {
                return false;
            }
        } else {
            Logger.i("BeaconManager already disconnected");
        }

        return false;
    }

    @Override
    /**
     * Stop ranging beacons. The method is thread safe.
     * <p/>
     * The method is synchronized.
     */
    public synchronized boolean finishScan() {
        if (!isConnected()) {
            Logger.d("BeaconManger not connected");
            return true;
        }

        if (!isScanning()) {
            Logger.d("BeaconManager is not scanning");
            return true;
        }

        final Message message = createMessage(ProximityService.MESSAGE_FINISH_SCAN, null);
        return sendMessage(message);
    }

    @Override
    /**
     * {@inheritDoc}
     */
    public synchronized void disconnect() {

        if (!isConnected()) {
            Logger.d(TAG + ": BeaconManager already disconnected.");
            return;
        } else if (isScanning()) {
            finishScan();
        }

        try {
            if (serviceConnection != null) {
                final Message message = createMessage(ProximityService.MESSAGE_WORK_FINISHED, null);
                serviceMessenger.send(message);
                context.unbindService(serviceConnection);
                serviceConnection = null;
                serviceMessenger = null;
            }
            super.disconnect();
        } catch (RemoteException e) {
            Logger.e(TAG + ": unexpected exception thrown while disconnecting from Beacon Service ", e);
            throw new IllegalStateException(e);
        }
    }

    @Override
    public synchronized boolean isScanning() {
        return isScanning;
    }

    @Override
    public void clearCache() {
        // Nothing to do
    }

    @Override
    public void clearBuffers() {
        // Nothing to do
    }

    private boolean startScanIfConnected(final ScanContext scanContext, final int messageCode) {
        final boolean isInitializationRequest = messageCode == ProximityService.MESSAGE_INITIALIZE_SCAN;
        if (isInitializationRequest && isScanning()) {
            Logger.d(TAG + ": BeaconManager is already scanning");
            return false;
        }

        final Message message = createMessage(messageCode, scanContext);
        return sendMessage(message);
    }

    private Message createMessage(final int messageCode, final Object obj) {
        final Message message = Message.obtain(null, messageCode, id, -1, obj);
        message.replyTo = managerMessenger;

        return message;
    }

    private static class ManagerHandler extends Handler {

        private final ProximityManager manager;

        private ManagerHandler(ProximityManager manager) {
            this.manager = manager;
        }

        @Override
        public void handleMessage(Message msg) {

            switch (msg.what) {
                case ProximityService.MESSAGE_INITIALIZE_SCAN:
                case ProximityService.MESSAGE_RESTART_SCAN:
                    manager.isScanning = (msg.arg1 == ProximityService.MESSAGE_SERVICE_RESPONSE_OK);
                    break;

                case ProximityService.MESSAGE_FINISH_SCAN:
                case ProximityService.MESSAGE_WORK_FINISHED:
                    manager.isScanning = false;
                    break;

                default:
                    throw new IllegalArgumentException("Unsupported response code: " + msg.what);
            }
        }
    }

    /**
     * Proximity listener provides callbacks for scan process performed by
     * ({@link ProximityManager}).
     * <p/>
     * Please note that callback methods are invoked in a different thread than
     * the UI thread.
     */
    public interface ProximityListener {
        /**
         * Called when scan starts.
         */
        void onScanStart();

        /**
         * Called when scan stops.
         */
        void onScanStop();

        /**
         * Called whenever specific event occurs.
         *
         * @param event the event
         */
        void onEvent(BluetoothDeviceEvent event);
    }
}
