package com.kontakt.sdk.android.ble.discovery.ibeacon;

import android.util.SparseArray;

import com.kontakt.sdk.android.ble.discovery.ScanResponse;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;

import java.util.UUID;

class AdvertisingPacketImpl implements IBeaconAdvertisingPacket {

    private final SparseArray<byte[]> advertisingData;
    private final UUID proximityUUID;
    private final int major;
    private final int minor;
    private final String beaconUniqueId;
    private final String firmwareVersion;
    private final int batteryPercentagePower;
    private final int txPower;
    private final int deviceHashCode;
    private final String name;
    private final String address;

    private long timestamp;
    private double distance;
    private double rssi;
    private Proximity proximity;
    private boolean shuffled;

    private AdvertisingPacketImpl(Builder builder) {
        this.advertisingData = builder.advertisingData;
        this.rssi = builder.rssi;
        this.proximityUUID = builder.proximityUUID;
        this.major = builder.major;
        this.minor = builder.minor;
        this.beaconUniqueId = builder.beaconUniqueId;
        this.firmwareVersion = builder.firmwareVersion;
        this.batteryPercentagePower = builder.batteryPercentagePower;
        this.txPower = builder.txPower;
        this.distance = builder.distance;
        this.proximity = builder.proximity;
        this.timestamp = builder.timestamp;
        this.name = builder.name;
        this.address = builder.address;
        this.shuffled = builder.shuffled;

        this.deviceHashCode = HashCodeBuilder.init()
                .append(builder.name)
                .append(advertisingData.get(IBeaconAdvertisingDataController.TYPE_MANUFACTURER_SPECIFIC_DATA))
                .build();
    }

    @Override
    public byte[] getManufacturerData() {
        return advertisingData.get(IBeaconAdvertisingDataController.TYPE_MANUFACTURER_SPECIFIC_DATA);
    }

    @Override
    public byte[] getServiceData() {
        return advertisingData.get(ScanResponse.TYPE_SERVICE_DATA);
    }

    @Override
    public double getRssi() {
        return rssi;
    }

    @Override
    public UUID getProximityUUID() {
        return proximityUUID;
    }

    @Override
    public int getMajor() {
        return major;
    }

    @Override
    public int getMinor() {
        return minor;
    }

    @Override
    public String getBeaconUniqueId() {
        return beaconUniqueId;
    }

    @Override
    public String getFirmwareVersion() {
        return firmwareVersion;
    }

    @Override
    public int getBatteryPercentagePower() {
        return batteryPercentagePower;
    }

    @Override
    public int getTxPower() {
        return txPower;
    }

    @Override
    public double getDistance() {
        return distance;
    }

    @Override
    public Proximity getProximity() {
        return proximity;
    }

    @Override
    public long getTimestamp() {
        return timestamp;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getAddress() {
        return address;
    }

    @Override
    public int getDeviceHashCode() {
        return deviceHashCode;
    }

    @Override
    public boolean isShuffled() {
        return shuffled;
    }

    /**
     * Sets distance.
     *
     * @param distance the distance
     */
    public void setDistance(final double distance) {
        this.distance = distance;
    }

    /**
     * Sets proximity.
     *
     * @param proximity the proximity
     */
    public void setProximity(final Proximity proximity) {
        this.proximity = proximity;
    }

    /**
     * Sets rssi.
     *
     * @param rssi the rssi
     */
    public void setRssi(final double rssi) {
        this.rssi = rssi;
    }

    /**
     * Sets timespamp
     *
     * @param timestamp the timestamp
     */
    public void setTimestamp(long timestamp) {
        this.timestamp = timestamp;
    }

    /**
     * The type Builder.
     */
    static class Builder {
        private SparseArray<byte[]> advertisingData;
        private double rssi;
        private UUID proximityUUID;
        private int major;
        private int minor;
        private String beaconUniqueId;
        private String firmwareVersion;
        private int batteryPercentagePower;
        private int txPower;
        private double distance;
        private Proximity proximity;
        private long timestamp;
        private String address;
        private String name;
        private boolean shuffled;

        /**
         * Sets rssi.
         *
         * @param rssi the rssi
         * @return the rssi
         */
        public Builder setRssi(double rssi) {
            this.rssi = rssi;
            return this;
        }

        /**
         * Sets proximity uUID.
         *
         * @param proximityUUID the proximity uUID
         * @return the proximity uUID
         */
        public Builder setProximityUUID(UUID proximityUUID) {
            this.proximityUUID = proximityUUID;
            return this;
        }

        /**
         * Sets major.
         *
         * @param major the major
         * @return the major
         */
        public Builder setMajor(int major) {
            this.major = major;
            return this;
        }

        /**
         * Sets minor.
         *
         * @param minor the minor
         * @return the minor
         */
        public Builder setMinor(int minor) {
            this.minor = minor;
            return this;
        }

        /**
         * Sets beacon unique id.
         *
         * @param beaconUniqueId the beacon unique id
         * @return the beacon unique id
         */
        public Builder setBeaconUniqueId(String beaconUniqueId) {
            this.beaconUniqueId = beaconUniqueId;
            return this;
        }

        /**
         * Sets firmware version.
         *
         * @param firmwareVersion the firmware version
         * @return the firmware version
         */
        public Builder setFirmwareVersion(String firmwareVersion) {
            this.firmwareVersion = firmwareVersion;
            return this;
        }

        /**
         * Sets battery percentage power.
         *
         * @param batteryPercentagePower the battery percentage power
         * @return the battery percentage power
         */
        public Builder setBatteryPercentagePower(int batteryPercentagePower) {
            this.batteryPercentagePower = batteryPercentagePower;
            return this;
        }

        /**
         * Sets address.
         *
         * @param address the address
         * @return the address
         */
        public Builder setAddress(final String address) {
            this.address = address;
            return this;
        }

        /**
         * Sets name.
         *
         * @param name the name
         * @return the name
         */
        public Builder setName(final String name) {
            this.name = name;
            return this;
        }

        /**
         * Sets tx power.
         *
         * @param txPower the tx power
         * @return the tx power
         */
        public Builder setTxPower(int txPower) {
            this.txPower = txPower;
            return this;
        }

        /**
         * Sets distance.
         *
         * @param distance the distance
         * @return the distance
         */
        public Builder setDistance(double distance) {
            this.distance = distance;
            return this;
        }

        /**
         * Sets proximity.
         *
         * @param proximity the proximity
         * @return the proximity
         */
        public Builder setProximity(Proximity proximity) {
            this.proximity = proximity;
            return this;
        }

        /**
         * Sets timestamp.
         *
         * @param timestamp the timestamp
         * @return the timestamp
         */
        public Builder setTimestamp(long timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        /**
         * Sets advertising data.
         *
         * @param advertisingData the advertising data
         * @return the advertising data
         */
        public Builder setAdvertisingData(SparseArray<byte[]> advertisingData) {
            this.advertisingData = advertisingData;
            return this;
        }

        public Builder setShuffled(boolean shuffled) {
            this.shuffled = shuffled;
            return this;
        }

        /**
         * Build advertising packet impl.
         *
         * @return the advertising packet impl
         */
        public AdvertisingPacketImpl build() {
            return new AdvertisingPacketImpl(this);
        }
    }
}
