package com.kontakt.sdk.android.ble.discovery.eddystone;

import android.bluetooth.BluetoothDevice;
import android.text.TextUtils;

import com.kontakt.sdk.android.ble.configuration.ActivityCheckConfiguration;
import com.kontakt.sdk.android.ble.configuration.scan.EddystoneScanContext;
import com.kontakt.sdk.android.ble.device.EddystoneNamespace;
import com.kontakt.sdk.android.ble.discovery.AbstractBluetoothDeviceDiscoverer;
import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceEvent;
import com.kontakt.sdk.android.ble.discovery.DiscoveryContract;
import com.kontakt.sdk.android.ble.discovery.EventType;
import com.kontakt.sdk.android.ble.spec.EddystoneFrameType;
import com.kontakt.sdk.android.ble.util.ReplacingArrayList;
import com.kontakt.sdk.android.common.profile.IEddystoneDevice;
import com.kontakt.sdk.android.common.profile.IEddystoneNamespace;

import java.util.ArrayList;

/**
 * Performs Eddystone devices discovery and sends specific events via
 * {@link DiscoveryContract}. The discoverer adjusts its behaviour according to
 * the specified {@link EddystoneScanContext} and {@link ActivityCheckConfiguration}.
 */
public class EddystoneDiscoverer extends AbstractBluetoothDeviceDiscoverer<IEddystoneNamespace, IEddystoneDevice> {

    private final EddystoneAdvertisingDataController controller;

    /**
     * Instantiates a new Eddystone discoverer.
     *
     * @param scanContext                the scan context
     * @param activityCheckConfiguration the activity check configuration
     * @param discoveryContract          the discovery contract
     */
    public EddystoneDiscoverer(EddystoneScanContext scanContext,
                               ActivityCheckConfiguration activityCheckConfiguration,
                               DiscoveryContract discoveryContract) {
        super(discoveryContract,
                scanContext.getDistanceSort(),
                scanContext.getEventTypes(),
                activityCheckConfiguration,
                scanContext.getSpaces(),
                scanContext.getDevicesUpdateCallbackInterval());

        controller = new EddystoneAdvertisingDataController(scanContext);
    }

    @Override
    protected BluetoothDeviceEvent createEvent(EventType eventType,
                                               IEddystoneNamespace eddystoneNamespace,
                                               ArrayList<IEddystoneDevice> deviceList) {
        return new EddystoneDeviceEvent(eventType, eddystoneNamespace, deviceList);
    }

    @Override
    protected void onBeforeDeviceLost(IEddystoneDevice device) {
        controller.clearRssiCalculation(device.getAddress().hashCode());
    }

    @Override
    public boolean performDiscovery(BluetoothDevice bluetoothDevice, int rssi, byte[] scanResult) {

        if (controller.isEnabled() && !EddystoneAdvertisingDataController.isEddystoneSpecificFrame(scanResult)) {
            return PROFILE_UNRECOGNIZED;
        }

        String deviceAddress = bluetoothDevice.getAddress();

        EddystoneFrameType frameType = EddystoneFrameType.fromScanRecord(scanResult);

        if (frameType == null) {
            return PROFILE_UNRECOGNIZED;
        }

        controller.cacheProperty(frameType, deviceAddress, scanResult);

        if (!controller.filter(deviceAddress)) {
            return PROFILE_RECOGNIZED_FILTERING_NOT_PASSED;
        }

        IEddystoneNamespace namespace = extractNamespace(controller.getNamespaceIfAllTriggersAreSatisfied(deviceAddress));

        if (namespace == null) {
            return PROFILE_RECOGNIZED_NO_BELONGING_SPACE_FOUND;
        }

        notifySpacePresent(namespace.hashCode(), System.currentTimeMillis());

        ReplacingArrayList<IEddystoneDevice> deviceList = getDevicesInSpace(namespace);

        if (deviceList == null) {
            deviceList = new ReplacingArrayList<IEddystoneDevice>();
            insertDevicesIntoSpace(namespace, deviceList);

            onSpaceEnteredEvent(namespace);

            return true;
        }

        IEddystoneDevice device = controller.getOrCreateDevice(bluetoothDevice, rssi);

        if (device == null) {
            return PROFILE_RECOGNIZED_DEVICE_NOT_READY;
        }

        if (deviceList.addOrReplace(device)) {
            onDeviceDiscoveredEvent(namespace, device);
        } else {
            sortIfEnabled(deviceList);
            onDevicesUpdatedEvent(namespace, deviceList);
        }

        return true;
    }

    @Override
    public void disable() {
        controller.clearResources();
        controller.disable();
    }

    IEddystoneNamespace extractNamespace(final EddystoneUIDAdvertisingPacket advertisingPacket) {

        if (advertisingPacket == null) {
            return null;
        }

        String namespaceId = advertisingPacket.getNamespaceId();

        for (IEddystoneNamespace namespace : getSpaceSet()) {
            if (namespace == EddystoneNamespace.EVERYWHERE) {
                return namespace;
            }

            if (TextUtils.equals(namespace.getId(), namespaceId)) {
                return namespace;
            }
        }

        return null;
    }
}
