package com.kontakt.sdk.android.ble.connection;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothGatt;
import android.bluetooth.BluetoothGattCallback;
import android.bluetooth.BluetoothGattCharacteristic;
import android.bluetooth.BluetoothProfile;
import android.content.Context;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.os.RemoteException;
import com.kontakt.sdk.android.ble.device.KontaktDeviceCharacteristics;
import com.kontakt.sdk.android.ble.exception.ServiceAbsentException;
import com.kontakt.sdk.android.ble.spec.BluetoothDeviceCharacteristic;
import com.kontakt.sdk.android.ble.spec.KontaktDeviceCharacteristic;
import com.kontakt.sdk.android.ble.spec.KontaktDeviceService;
import com.kontakt.sdk.android.ble.util.BluetoothUtils;
import com.kontakt.sdk.android.common.TimestampUtil;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayDeque;
import java.util.Collection;
import java.util.Queue;
import java.util.concurrent.TimeUnit;

@TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
final class NormalGattController extends BluetoothGattCallback implements Closeable, GattController {

    private KontaktDeviceConnection contract;

    private BluetoothGatt gattServer;

    private KontaktDeviceServiceStore iBeaconServiceStore;

    private Handler uiHandler;

    private final byte[] beaconPassword;

    private Queue<BluetoothDeviceCharacteristic> preAuthCharacteristicsQueue;

    private Context context;
    private BluetoothDevice bluetoothDevice;

    NormalGattController(KontaktDeviceConnection contract,
                         Context context,
                         RemoteBluetoothDevice device) throws RemoteException {
        this.contract = contract;
        this.context = context;
        this.beaconPassword = device.getPassword();
        this.uiHandler = new Handler(Looper.getMainLooper());

        final BluetoothDevice bluetoothDevice = BluetoothUtils.getBluetoothDevice(device.getAddress());

        if (bluetoothDevice == null) {
            throw new RemoteException("Bluetooth device is null");
        }

        this.bluetoothDevice = bluetoothDevice;
    }


    @Override
    public boolean connect() {
        gattServer = bluetoothDevice.connectGatt(context, false, this);
        return gattServer != null;
    }

    @Override
    public void onConnectionStateChange(BluetoothGatt gatt, int status, int newState) {
        switch (newState) {
            case BluetoothProfile.STATE_CONNECTED:
                if (BluetoothGatt.GATT_SUCCESS == status) {
                    contract.onConnectionStateChange(KontaktDeviceConnection.State.CONNECTED);
                    contract.onConnected();
                    if (!gattServer.discoverServices()) {
                        contract.onError(KontaktDeviceConnection.ERROR_SERVICES_DISCOVERY);
                    }
                } else {
                    contract.onError(KontaktDeviceConnection.toGattError(status));
                }
                break;

            case BluetoothProfile.STATE_DISCONNECTED:
                if (contract != null) {
                    if (BluetoothGatt.GATT_SUCCESS != status) {
                        contract.onError(KontaktDeviceConnection.toGattError(status));
                    }
                    if (contract != null) {
                        if (contract.getState() == KontaktDeviceConnection.State.AUTHENTICATING) {
                            contract.onFailure(KontaktDeviceConnection.FAILURE_WRONG_PASSWORD);
                        }
                    }
                    if (contract != null) {
                        contract.onConnectionStateChange(KontaktDeviceConnection.State.DISCONNECTED);
                    }
                    if (contract != null) {
                        contract.onDisconnected();
                    }
                }
                break;

            default:
                throw new IllegalArgumentException("Unsupported connection state change code: " + newState);
        }
    }

    @Override
    public void onServicesDiscovered(BluetoothGatt gatt, int status) {
        Logger.d(String.format("On Services Discovered: %s ", gatt.toString()));

        if (status != BluetoothGatt.GATT_SUCCESS) {
            Logger.v("Services discovered but with no success: " + status);
            return;
        }

        try {
            iBeaconServiceStore = new KontaktDeviceServiceStore(gatt.getServices(), KontaktDeviceServiceStore.StoreType.NORMAL);
            contract.onServicesDiscovered(iBeaconServiceStore);
            if (iBeaconServiceStore.contains(KontaktDeviceService.DFU_SERVICE)) {
                contract.onDfuModeEnabled();
            } else {
                authorize(gatt);
            }
        } catch (RemoteException e) {
            contract.onFailure(KontaktDeviceConnection.FAILURE_UNKNOWN_BEACON);
        }
    }

    @Override
    public void onCharacteristicRead(BluetoothGatt gatt, BluetoothGattCharacteristic characteristic, int status) {
        if (status == BluetoothGatt.GATT_SUCCESS) {
            final BluetoothDeviceCharacteristic wrapper = new BluetoothDeviceCharacteristic(characteristic);
            try {
                iBeaconServiceStore.replace(wrapper);
            } catch (ServiceAbsentException ignored) {
            }
            if (contract.getState() == KontaktDeviceConnection.State.CHARACTERISTICS_REQUESTING) {
                requestOrSetAuthenticated();
            }
        } else if (status == BluetoothGatt.GATT_READ_NOT_PERMITTED) {
            if (contract.getState() == KontaktDeviceConnection.State.CHARACTERISTICS_REQUESTING) {
                requestOrSetAuthenticated();
            }
        }
    }

    @Override
    public void onCharacteristicWrite(final BluetoothGatt gatt, final BluetoothGattCharacteristic characteristic, final int status) {
        if (contract.isConnected()) {
            try {
                iBeaconServiceStore.replace(new BluetoothDeviceCharacteristic(characteristic));
            } catch (ServiceAbsentException ignored) {
            }

            final boolean isSuccess = (status == BluetoothGatt.GATT_SUCCESS);
            KontaktDeviceCharacteristic beaconCharacteristic = KontaktDeviceCharacteristic.valueOf(characteristic.getUuid());
            if (beaconCharacteristic == null) {
                return;
            }

            if (beaconCharacteristic == KontaktDeviceCharacteristic.PASSWORD) {
                if (isSuccess) {
                    uiHandler.postDelayed(characteristicRequestRunnable, TimeUnit.SECONDS.toMillis(1));
                    contract.onConnectionStateChange(KontaktDeviceConnection.State.AUTHENTICATING);
                } else {
                    contract.onConnectionStateChange(KontaktDeviceConnection.State.CONNECTED);
                    contract.onError(KontaktDeviceConnection.ERROR_AUTHENTICATION);
                }
            } else {
                contract.onCharacteristicWritten(isSuccess, new WriteListener.WriteResponse(TimestampUtil.getTimestamp(), null));
                if (isSuccess) {
                    if (contract != null) {
                        contract.notifyDataSetChanged();
                    }
                }
            }
        }
    }


    @Override
    public void disconnect() {
        if (gattServer != null) {
            refresh();
            gattServer.disconnect();
        }
    }

    @Override
    public boolean writeCharacteristic(BluetoothGattCharacteristic characteristic) {
        return gattServer.writeCharacteristic(characteristic);
    }

    @Override
    public boolean writeCharacteristic(BluetoothGattCharacteristic characteristic, boolean readResponse) {
        throw new UnsupportedOperationException("Not supported in normal mode");
    }

    @Override
    public void refresh() {
        BluetoothUtils.refreshGattServer(gattServer);
    }

    @Override
    public void close() throws IOException {
        if (gattServer != null) {
            disconnect();
            gattServer.close();
        }

        if (iBeaconServiceStore != null) {
            iBeaconServiceStore.clear();
        }

        if (preAuthCharacteristicsQueue != null) {
            preAuthCharacteristicsQueue.clear();
        }

        contract = null;
        uiHandler.removeCallbacks(characteristicRequestRunnable);
    }

    private Runnable characteristicRequestRunnable = new Runnable() {
        @Override
        public void run() {
            if (contract == null) {
                return;
            }
            if (contract.getState() == KontaktDeviceConnection.State.AUTHENTICATING) {
                contract.onConnectionStateChange(KontaktDeviceConnection.State.CHARACTERISTICS_REQUESTING);
                requestCharacteristics();
            }
        }
    };

    private void requestOrSetAuthenticated() {
        if (preAuthCharacteristicsQueue.isEmpty()) {
            contract.onConnectionStateChange(KontaktDeviceConnection.State.AUTHENTICATED);
            contract.onAuthenticationSuccess(new KontaktDeviceCharacteristics(iBeaconServiceStore));
        } else {
            request(preAuthCharacteristicsQueue.poll());
        }
    }

    private void requestCharacteristics() {

        Collection<BluetoothDeviceCharacteristic> readableCharacteristics = iBeaconServiceStore.getReadableCharacteristics();

        if (preAuthCharacteristicsQueue == null) {
            preAuthCharacteristicsQueue = new ArrayDeque<BluetoothDeviceCharacteristic>(readableCharacteristics.size());
        } else {
            preAuthCharacteristicsQueue.clear();
        }

        preAuthCharacteristicsQueue.addAll(readableCharacteristics);

        request(preAuthCharacteristicsQueue.poll());
    }

    @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
    private void request(final BluetoothDeviceCharacteristic wrapper) {
        if (!gattServer.readCharacteristic(wrapper)) {
            if (contract != null && contract.getState() == KontaktDeviceConnection.State.CHARACTERISTICS_REQUESTING) {
                requestOrSetAuthenticated();
            }
        }
    }

    @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
    private void authorize(final BluetoothGatt gatt) {
        try {
            final BluetoothGattCharacteristic passwordCharacteristic = iBeaconServiceStore.getPasswordCharacteristic();
            passwordCharacteristic.setValue(beaconPassword);

            if (!gatt.writeCharacteristic(passwordCharacteristic)) {
                contract.onError(KontaktDeviceConnection.ERROR_AUTHENTICATION);
            }
        } catch (Exception ignored) {
            contract.onError(KontaktDeviceConnection.ERROR_AUTHENTICATION);
        }
    }
}
