package com.kontakt.sdk.android.ble.connection;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothGattCharacteristic;
import android.bluetooth.BluetoothGattService;
import android.os.Build;
import android.os.RemoteException;

import com.kontakt.sdk.android.ble.exception.CharacteristicAbsentException;
import com.kontakt.sdk.android.ble.exception.ServiceAbsentException;
import com.kontakt.sdk.android.ble.spec.BluetoothDeviceCharacteristic;
import com.kontakt.sdk.android.ble.spec.GattServiceStore;
import com.kontakt.sdk.android.ble.spec.KontaktDeviceCharacteristic;
import com.kontakt.sdk.android.ble.spec.KontaktDeviceService;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

/**
 * Acts as a container of characteristics requested before authorization.
 */
public class KontaktDeviceServiceStore implements GattServiceStore {

    private static final KontaktDeviceService REQUIRED_SERVICE_NORMAL_MODE = KontaktDeviceService.CONTROL;
    private static final KontaktDeviceService REQUIRED_SERVICE_SECURE_MODE = KontaktDeviceService.SECURE_CONFUGIRATION_SERVICE;

    private final Map<KontaktDeviceService, Map<KontaktDeviceCharacteristic, BluetoothDeviceCharacteristic>> storeMap;
    private final StoreType storeType;

    /**
     * Type of store
     * <ul>
     * <li>{@link #NORMAL} if beacon firmware revision is lower than 4.0</li>
     * <li>{@link #SECURE} if beacon firmware revision is 4.0 and greater</li>
     * </ul>
     */
    public enum StoreType {
        NORMAL,
        SECURE
    }

    /**
     * Instantiates a new Service store. If services sent are not compatible with
     * the ones propagated by kontakt.io beacons(services list does not contain
     * all required services: Service 1, Service 2, Service 3, Service 4,
     * Service 5, Service 6, Service 7) then the RemoteException (informing that
     * the beacon is not a property of kontakt.io) is thrown.
     *
     * @param gattServices beacon services list
     * @throws RemoteException the remote exception thrown if services list does
     *                         not match the specification of kontakt.io beacon
     */
    @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
    public KontaktDeviceServiceStore(final List<BluetoothGattService> gattServices, StoreType storeType) throws RemoteException {
        SDKPreconditions.checkNotNullOrEmpty(gattServices, "Gatt Service list is null.");
        this.storeMap = new HashMap<KontaktDeviceService, Map<KontaktDeviceCharacteristic, BluetoothDeviceCharacteristic>>(gattServices.size());
        this.storeType = storeType;
        switch (storeType) {
            case NORMAL:
                createNormalMode(gattServices);
                break;
            case SECURE:
                createSecureMode(gattServices);
                break;
        }
        try {
            assertRequiredServiceFound(storeType);
            assertRequiredServicesFoundIfDfuModeEnabled();
        } catch (ServiceAbsentException e) {
            throw new RemoteException("Beacon is not recognized as product of kontakt.io company");
        }
    }

    private void createNormalMode(List<BluetoothGattService> gattServices) {
        for (final BluetoothGattService service : gattServices) {
            KontaktDeviceService serviceModel = getServiceModel(service.getUuid());
            if (serviceModel == null) {
                continue;
            }

            Map<KontaktDeviceCharacteristic, BluetoothDeviceCharacteristic> characteristics = storeMap.get(serviceModel);

            Collection<BluetoothGattCharacteristic> gattCharacteristics = service.getCharacteristics();

            if (characteristics == null) {
                characteristics = new HashMap<KontaktDeviceCharacteristic, BluetoothDeviceCharacteristic>(gattCharacteristics.size());
                storeMap.put(serviceModel, characteristics);
            }

            for (BluetoothGattCharacteristic gattCharacteristic : gattCharacteristics) {
                KontaktDeviceCharacteristic characteristicModel = getCharacteristicModel(gattCharacteristic.getUuid());
                if (characteristicModel == null) {
                    continue;
                }

                characteristics.put(characteristicModel, new BluetoothDeviceCharacteristic(gattCharacteristic));
            }
        }
    }

    private void createSecureMode(List<BluetoothGattService> gattServices) {
        for (BluetoothGattService gattService : gattServices) {
            KontaktDeviceService secureService = KontaktDeviceService.valueOf(gattService.getUuid());
            if (secureService == null) {
                continue;
            }

            Map<KontaktDeviceCharacteristic, BluetoothDeviceCharacteristic> characteristics = storeMap.get(secureService);

            if (characteristics == null) {
                characteristics = new HashMap<KontaktDeviceCharacteristic, BluetoothDeviceCharacteristic>();
                storeMap.put(secureService, characteristics);
            }

            List<BluetoothGattCharacteristic> bluetoothGattCharacteristics = gattService.getCharacteristics();

            for (BluetoothGattCharacteristic gattCharacteristic : bluetoothGattCharacteristics) {
                KontaktDeviceCharacteristic secureCharacteristic = KontaktDeviceCharacteristic.valueOf(gattCharacteristic.getUuid());
                if (secureCharacteristic == null) {
                    continue;
                }
                characteristics.put(secureCharacteristic, new BluetoothDeviceCharacteristic(gattCharacteristic));
            }
        }
    }

    /**
     * Gets firmware revision wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getFirmwareRevisionCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.DEVICE_INFORMATION),
                KontaktDeviceCharacteristic.FIRMWARE_REVISION_STRING
        );
    }

    /**
     * Gets hardware revision wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getHardwareRevisionCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.DEVICE_INFORMATION),
                KontaktDeviceCharacteristic.HARDWARE_REVISION
        );
    }

    /**
     * Gets manufacturer name wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getManufacturerNameCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.DEVICE_INFORMATION),
                KontaktDeviceCharacteristic.MANUFACTURER_NAME
        );
    }

    /**
     * Gets device name wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getDeviceNameCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.GENERIC_ACCESS),
                KontaktDeviceCharacteristic.DEVICE_NAME
        );
    }

    /**
     * Gets propagated device name wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getPropagatedDeviceNameCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.PROXIMITY),
                KontaktDeviceCharacteristic.PROPAGATED_DEVICE_NAME
        );
    }

    /**
     * Gets minor wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getMinorCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.PROXIMITY),
                KontaktDeviceCharacteristic.MINOR
        );
    }

    /**
     * Gets non connectable wrapper.
     * The Non-connectable characteristic was added with "2.6" firmware. If beacon
     * has lower firmware, then it returns null.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getNonConnectableCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.CONTROL),
                KontaktDeviceCharacteristic.NON_CONNECTABLE_MODE
        );
    }

    /**
     * Provides shuffle interval wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getShuffleIntervalCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.CONTROL),
                KontaktDeviceCharacteristic.SHUFFLE_INTERVAL
        );
    }

    /**
     * Provides shuffle key wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getShuffleKeyCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.CONTROL),
                KontaktDeviceCharacteristic.SHUFFLE_KEY
        );
    }

    /**
     * Provides sensor on wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getSensorOnCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.SENSORS_SERVICE),
                KontaktDeviceCharacteristic.SENSOR_ON
        );
    }

    /**
     * Provides accelerometer wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getAccelerometerCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.SENSORS_SERVICE),
                KontaktDeviceCharacteristic.ACCELEROMETER
        );
    }

    /**
     * Gets active profile wrapper.
     * The active profile characteristic wrapper. The wrapper was added in "3.0" firmware version.
     * If beacon has lower firmware, then null is returned.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getActiveProfileCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.CONTROL),
                KontaktDeviceCharacteristic.ACTIVE_PROFILE
        );
    }

    /**
     * Gets major wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getMajorCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.PROXIMITY),
                KontaktDeviceCharacteristic.MAJOR
        );
    }

    /**
     * Gets proximity wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getProximityCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.PROXIMITY),
                KontaktDeviceCharacteristic.PROXIMITY_UUID
        );
    }

    /**
     * Gets power level wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getPowerLevelCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.TX_POWER),
                KontaktDeviceCharacteristic.TX_POWER_LEVEL
        );
    }

    /**
     * Gets battery level wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getBatteryLevelCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.BATTERY_LEVEL),
                KontaktDeviceCharacteristic.BATTERY_LEVEL
        );
    }

    /**
     * Gets password wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getPasswordCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.CONTROL),
                KontaktDeviceCharacteristic.PASSWORD
        );
    }

    /**
     * Provides master password wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getMasterPasswordCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.CONTROL),
                KontaktDeviceCharacteristic.MASTER_PASSWORD
        );
    }

    /**
     * Gets set new password wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getSetNewPasswordCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.CONTROL),
                KontaktDeviceCharacteristic.SET_PASSWORD
        );
    }

    /**
     * Gets advertising interval wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getAdvertisingIntervalCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.TIMERS),
                KontaktDeviceCharacteristic.ADVERTISING_INTERVAL
        );
    }

    /**
     * Gets reset wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getResetCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.CONTROL),
                KontaktDeviceCharacteristic.RESET
        );
    }

    /**
     * Gets bootloader wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getBootloaderCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.CONTROL),
                KontaktDeviceCharacteristic.BOOTLOADER
        );
    }

    /**
     * Provides unique id wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getUniqueIdCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.PROXIMITY),
                KontaktDeviceCharacteristic.BEACON_ID
        );
    }

    /**
     * Provides namespace id wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getNamespaceIdCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.PROXIMITY),
                KontaktDeviceCharacteristic.NAMESPACE_ID
        );
    }

    /**
     * Provides instance id wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getInstanceIdCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.PROXIMITY),
                KontaktDeviceCharacteristic.INSTANCE_ID
        );
    }

    @Override
    /**
     * Provides url wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    public BluetoothDeviceCharacteristic getUrlCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.PROXIMITY),
                KontaktDeviceCharacteristic.URL_ID
        );
    }

    /**
     * Gets default settings wrapper.
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getDefaultSettingsCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.CONTROL),
                KontaktDeviceCharacteristic.DEFAULT_SETTINGS
        );
    }

    /**
     * Gets last processed requet token wrapper
     * <p>
     * Characteristic added in firmware revision 4.0, on lower firmware revision throws exception
     * </p>
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getSecureLastProcessedRequestTokenCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.SECURE_CONFUGIRATION_SERVICE),
                KontaktDeviceCharacteristic.SECURE_LAST_PROCESSED_REQUEST_TOKEN
        );
    }

    /**
     * Gets secure write wrapper
     * <p>
     * Characteristic added in firmware revision 4.0, on lower firmware revision throws exception
     * </p>
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getSecureWriteCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.SECURE_CONFUGIRATION_SERVICE),
                KontaktDeviceCharacteristic.SECURE_WRITE
        );
    }

    /**
     * Gets secure write response wrapper
     * <p>
     * Characteristic added in firmware revision 4.0, on lower firmware revision throws exception
     * </p>
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getSecureResponseCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.SECURE_CONFUGIRATION_SERVICE),
                KontaktDeviceCharacteristic.SECURE_RESPONSE
        );
    }

    /**
     * Gets secure control point wrapper
     * <p>
     * Characteristic added in firmware revision 4.0, on lower firmware revision throws exception
     * </p>
     *
     * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
     * @throws ServiceAbsentException        if service not found
     * @throws CharacteristicAbsentException if characteristic not found
     */
    @Override
    public BluetoothDeviceCharacteristic getSecureControlPointCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
        return getCharacteristic(
                getService(KontaktDeviceService.SECURE_CONFUGIRATION_SERVICE),
                KontaktDeviceCharacteristic.SECURE_CONTROL_POINT
        );
    }

    /**
     * Gets type of store
     *
     * @return {@link com.kontakt.sdk.android.ble.connection.KontaktDeviceServiceStore.StoreType}
     */
    public StoreType getStoreType() {
        return storeType;
    }

    /**
     * Checks if {@link KontaktDeviceService} was found during service discovery
     *
     * @param serviceModel model to find
     * @return true or false
     */
    @Override
    public boolean contains(KontaktDeviceService serviceModel) {
        return storeMap.containsKey(serviceModel);
    }

    /**
     * Replaces provided characteristic in cache
     *
     * @param characteristicWrapper new characteristic
     * @throws ServiceAbsentException if service not found
     */
    @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
    @Override
    public void replace(BluetoothDeviceCharacteristic characteristicWrapper) throws ServiceAbsentException {
        UUID serviceId = characteristicWrapper.getService().getUuid();

        UUID characteristicId = characteristicWrapper.getId();

        KontaktDeviceService serviceModel = getServiceModel(serviceId);
        KontaktDeviceCharacteristic characteristicModel = getCharacteristicModel(characteristicId);

        storeMap.get(serviceModel).put(characteristicModel, characteristicWrapper);
    }

    /**
     * Clears cache
     */
    @Override
    public void clear() {
        storeMap.clear();
    }

    /**
     * Gets all readable characteristic found during service discovery
     *
     * @return Collection of readable characteristic
     */
    public final Collection<BluetoothDeviceCharacteristic> getReadableCharacteristics() {

        Collection<BluetoothDeviceCharacteristic> readableCharacteristics = new ArrayList<BluetoothDeviceCharacteristic>();

        for (Map<KontaktDeviceCharacteristic, BluetoothDeviceCharacteristic> characteristicMap : storeMap.values()) {
            for (BluetoothDeviceCharacteristic characteristic : characteristicMap.values()) {
                if (characteristic.isReadable()) {
                    readableCharacteristics.add(characteristic);
                }
            }
        }

        return Collections.unmodifiableCollection(readableCharacteristics);
    }

    Map<KontaktDeviceCharacteristic, BluetoothDeviceCharacteristic> getService(KontaktDeviceService service) throws ServiceAbsentException {
        SDKPreconditions.checkNotNull(service, "Service is null.");
        SDKPreconditions.checkArgument(contains(service), new ServiceAbsentException(String.format("The Service %s was not found during services discovery", service)));

        return Collections.unmodifiableMap(storeMap.get(service));
    }

    BluetoothDeviceCharacteristic getCharacteristic(Map<KontaktDeviceCharacteristic, BluetoothDeviceCharacteristic> map, KontaktDeviceCharacteristic characteristic) throws CharacteristicAbsentException {
        SDKPreconditions.checkState(map.containsKey(characteristic), new CharacteristicAbsentException(String.format("The characteristic %s is absent", characteristic == null ? "" : characteristic)));
        return map.get(characteristic);
    }

    private KontaktDeviceService getServiceModel(UUID serviceUUID) {
        return KontaktDeviceService.valueOf(serviceUUID);
    }

    private KontaktDeviceCharacteristic getCharacteristicModel(UUID characteristicUUID) {
        return KontaktDeviceCharacteristic.valueOf(characteristicUUID);
    }

    private void assertRequiredServiceFound(StoreType storeType) throws ServiceAbsentException {
        switch (storeType) {
            case NORMAL:
                getService(REQUIRED_SERVICE_NORMAL_MODE);
                break;
            case SECURE:
                getService(REQUIRED_SERVICE_SECURE_MODE);
                break;
        }
    }

    private void assertRequiredServicesFoundIfDfuModeEnabled() throws ServiceAbsentException {
        if (contains(KontaktDeviceService.DFU_SERVICE)) {
            final Set<KontaktDeviceService> requiredServices = EnumSet.of(KontaktDeviceService.GENERIC_ACCESS,
                    KontaktDeviceService.GENERIC_ATTRIBUTE,
                    KontaktDeviceService.DFU_SERVICE);

            for (KontaktDeviceService kontaktDeviceService : requiredServices) {
                getService(kontaktDeviceService);
            }
        }
    }
}
