package com.kontakt.sdk.android.ble.configuration.scan;

import com.kontakt.sdk.android.ble.configuration.ActivityCheckConfiguration;
import com.kontakt.sdk.android.ble.configuration.ForceScanConfiguration;
import com.kontakt.sdk.android.ble.configuration.ScanPeriod;
import com.kontakt.sdk.android.ble.filter.eddystone.TLMFilter;
import com.kontakt.sdk.android.ble.filter.eddystone.UIDFilter;
import com.kontakt.sdk.android.ble.filter.eddystone.URLFilter;
import com.kontakt.sdk.android.ble.filter.ibeacon.IBeaconFilter;
import com.kontakt.sdk.android.common.profile.IBeaconRegion;
import com.kontakt.sdk.android.common.profile.IEddystoneNamespace;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.TimeUnit;

final class ScanContextValidator {

    private ScanContextValidator() {
    }

    private static final int MAX_EDDYSTONE_FILTERS_SIZE = 5;

    private static final int MAX_IBEACON_FILTERS_SIZE = 10;

    private static final int REGIONS_LIMIT = 20;

    private static final long MINIMAL_ACTIVE_SCAN_PERIOD = TimeUnit.SECONDS.toMillis(3);

    private static final long MINIMAL_POSITVE_PASSIVE_SCAN_PERIOD = TimeUnit.SECONDS.toMillis(2);

    static void validate(final ActivityCheckConfiguration activityCheckConfiguration) {
        if (activityCheckConfiguration == ActivityCheckConfiguration.MINIMAL ||
                activityCheckConfiguration == ActivityCheckConfiguration.DISABLED) {
            return;
        }

        long inactivityTimeout = activityCheckConfiguration.getInactivityTimeout();
        long minimalInactivityTimeout = ActivityCheckConfiguration.MINIMAL.getInactivityTimeout();

        SDKPreconditions.checkArgument(inactivityTimeout >= minimalInactivityTimeout,
                "The inactivity timeout must be no shorter than 1 seconds");

        long checkPeriod = activityCheckConfiguration.getCheckPeriod();
        long minimalCheckPeriod = ActivityCheckConfiguration.MINIMAL.getCheckPeriod();

        SDKPreconditions.checkArgument(checkPeriod >= minimalCheckPeriod,
                "The check period must be no shorter than 1 second");
    }

    static void validate(final ForceScanConfiguration forceScanConfiguration) {
        final ForceScanConfiguration defaultConfig = ForceScanConfiguration.MINIMAL;

        final long forceScanActivePeriod = defaultConfig.getForceScanActivePeriod();
        if (forceScanConfiguration.getForceScanActivePeriod() < forceScanActivePeriod) {
            throw new IllegalArgumentException(String.format(
                    "The Force Scan Active period must not be shorter than %d milliseconds",
                    forceScanActivePeriod));
        }

        final long forceScanPassivePeriod = defaultConfig.getForceScanPassivePeriod();
        if (forceScanConfiguration.getForceScanPassivePeriod() < forceScanPassivePeriod) {
            throw new IllegalArgumentException(String.format(
                    "The Force Scan Passive period must not be shorter than %d",
                    forceScanPassivePeriod));
        }
    }

    static void validate(final ScanPeriod scanPeriod) {

        final long activeSubperiod = scanPeriod.getActivePeriod();
        SDKPreconditions.checkArgument(activeSubperiod >= MINIMAL_ACTIVE_SCAN_PERIOD,
                String.format("Active subperiod should last no shorter than %d seconds",
                        MINIMAL_ACTIVE_SCAN_PERIOD / 1000));


        final long passiveSubPeriod = scanPeriod.getPassivePeriod();

        SDKPreconditions.checkArgument((passiveSubPeriod == 0L) || (passiveSubPeriod >= MINIMAL_POSITVE_PASSIVE_SCAN_PERIOD),
                "Unsupported ScanPeriod's passive period");
    }

    static void validateIBeaconRegionsCount(final Collection<IBeaconRegion> iBeaconRegions) {
        SDKPreconditions.checkNotNull(iBeaconRegions, "Regions are null");
        SDKPreconditions.checkArgument(iBeaconRegions.size() <= REGIONS_LIMIT, "You can range beacons within max 20 regions.");
    }

    static void validateIBeaconFiltersCount(final Collection<IBeaconFilter> filters) {
        SDKPreconditions.checkNotNull(filters, "IBeacon filters are null");
        SDKPreconditions.checkState(filters.size() <= MAX_IBEACON_FILTERS_SIZE,
                "Too many iBeacon filters specified. The limit is " + MAX_IBEACON_FILTERS_SIZE);
    }

    static void validateEddystoneUIDFiltersCount(List<UIDFilter> eddystoneFilterList) {
        SDKPreconditions.checkNotNull(eddystoneFilterList, "Eddystone filters are null");
        SDKPreconditions.checkState(eddystoneFilterList.size() <= MAX_EDDYSTONE_FILTERS_SIZE,
                "Too many eddystone filters specified. The limit is " + MAX_EDDYSTONE_FILTERS_SIZE);
    }

    static void validateEddystoneTLMFiltersCount(List<TLMFilter> eddystoneFilterList) {
        SDKPreconditions.checkNotNull(eddystoneFilterList, "Eddystone filters are null");
        SDKPreconditions.checkState(eddystoneFilterList.size() <= MAX_EDDYSTONE_FILTERS_SIZE,
                "Too many eddystone filters specified. The limit is " + MAX_EDDYSTONE_FILTERS_SIZE);
    }

    static void validateEddystoneURLFiltersCount(List<URLFilter> eddystoneFilterList) {
        SDKPreconditions.checkNotNull(eddystoneFilterList, "Eddystone filters are null");
        SDKPreconditions.checkState(eddystoneFilterList.size() <= MAX_EDDYSTONE_FILTERS_SIZE,
                "Too many eddystone filters specified. The limit is " + MAX_EDDYSTONE_FILTERS_SIZE);
    }

    static void validateNamespacesCount(Collection<IEddystoneNamespace> eddystoneNamespaceSet) {
        SDKPreconditions.checkNotNull(eddystoneNamespaceSet, "Eddystone namespaces are null");
        SDKPreconditions.checkArgument(eddystoneNamespaceSet.size() <= REGIONS_LIMIT,
                "You can range eddystones within max 20 namespaces.");
    }

    static void validateDevicesUpdateCallbackInterval(long devicesUpdateCallbackInterval) {
        SDKPreconditions.checkArgument(devicesUpdateCallbackInterval >= 0, "Minimal callback period should be positive");
    }
}