package com.kontakt.sdk.android.ble.configuration.scan;

import com.kontakt.sdk.android.ble.configuration.ActivityCheckConfiguration;
import com.kontakt.sdk.android.ble.configuration.ForceScanConfiguration;
import com.kontakt.sdk.android.ble.configuration.ScanPeriod;
import com.kontakt.sdk.android.ble.manager.ProximityManager;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

/**
 * Provides scan-specific parameters influencing scan performance
 * profiles of BLE devices to be found.
 */
public final class ScanContext implements GlobalScanContext {

    /**
     * Default settings which works well on most modern devices
     */
    public static final ScanContext DEFAULT = new ScanContext.Builder()
            .setForceScanConfiguration(ForceScanConfiguration.DISABLED)
            .setScanMode(ProximityManager.SCAN_MODE_LOW_POWER)
            .setActivityCheckConfiguration(ActivityCheckConfiguration.MINIMAL)
            .setEddystoneScanContext(EddystoneScanContext.DEFAULT)
            .setIBeaconScanContext(IBeaconScanContext.DEFAULT)
            .setScanPeriod(ScanPeriod.RANGING)
            .build();

    private final int id;

    //Common properties
    private final ForceScanConfiguration forceScanConfiguration;
    private final ScanPeriod scanPeriod;
    private final int scanMode;
    private final ActivityCheckConfiguration activityCheckConfiguration;

    private final IBeaconScanContext iBeaconScanContext;

    private final EddystoneScanContext eddystoneScanContext;

    private ScanContext(Builder builder) {
        this.forceScanConfiguration = builder.forceScanConfiguration;
        this.scanPeriod = builder.scanPeriod;
        this.scanMode = builder.scanMode;
        this.activityCheckConfiguration = builder.activityCheckConfiguration;
        this.iBeaconScanContext = builder.iBeaconScanContext;
        this.eddystoneScanContext = builder.eddystoneScanContext;

        this.id = System.identityHashCode(this);
    }

    @Override
    /**
     * Provides id.
     *
     * @return the id
     */
    public int getId() {
        return id;
    }

    @Override
    /**
     * Provides {@link ActivityCheckConfiguration}.
     *
     * @return the beacon activity check configuration
     */
    public ActivityCheckConfiguration getActivityCheckConfiguration() {
        return activityCheckConfiguration;
    }

    @Override
    public IBeaconScanContext getIBeaconScanContext() {
        return iBeaconScanContext;
    }

    @Override
    public EddystoneScanContext getEddystoneScanContext() {
        return eddystoneScanContext;
    }

    @Override
    /**
     * Provides {@link ForceScanConfiguration}.
     *
     * @return the force scan configuration
     */
    public ForceScanConfiguration getForceScanConfiguration() {
        return forceScanConfiguration;
    }

    @Override
    /**
     * Provides Scan period.
     *
     * @return the monitor period
     */
    public ScanPeriod getScanPeriod() {
        return scanPeriod;
    }

    @Override
    /**
     * Provides scan mode.
     *
     * @return the scan mode
     */
    public int getScanMode() {
        return scanMode;
    }

    /**
     * Configuration builder is responsible for creating immutable configuration
     * which acts as a parameters and callbacks supplier and thus describes
     * desired behaviour.
     *
     * By default the Builder is initialized with all default parameters which
     * do not enable any of the functionality.
     */
    public static final class Builder {
        private int scanMode = ProximityManager.SCAN_MODE_BALANCED;
        private ForceScanConfiguration forceScanConfiguration = ForceScanConfiguration.DISABLED;
        private ScanPeriod scanPeriod = ScanPeriod.RANGING;
        private ActivityCheckConfiguration activityCheckConfiguration = ActivityCheckConfiguration.MINIMAL;
        private IBeaconScanContext iBeaconScanContext;
        private EddystoneScanContext eddystoneScanContext;

        /**
         * Sets {@link IBeaconScanContext} to ScanContext. Use null for disabling ibeacon scan
         * @param iBeaconScanContext
         * @return the builder
         */
        public Builder setIBeaconScanContext(IBeaconScanContext iBeaconScanContext) {
            this.iBeaconScanContext = iBeaconScanContext;
            return this;
        }

        /**
         * Sets {@link EddystoneScanContext} to ScanContext. Use null for disabling eddystone scan
         * @param eddystoneScanContext
         * @return the builder
         */
        public Builder setEddystoneScanContext(EddystoneScanContext eddystoneScanContext) {
            this.eddystoneScanContext = eddystoneScanContext;
            return this;
        }

        /**
         * Sets {@link ActivityCheckConfiguration}.
         * By default, the removal configuration is disabled and beacons which
         * signal's strength becomes unidentifiable remain in the delivered list.
         * Filtering active beacons list by applying removal configuration gives
         * more reliable relation to real beacons surrounding in particular moment.
         * This parameter should be applied if desired before invoking
         * startMonitoring/startRanging methods.
         * If not set, the RemovalConfiguration with unacceptable values is applied
         * and thus the functionality is disabled.
         *
         * @param actvitivyCheckConfiguration the activity check configuration
         */
        public Builder setActivityCheckConfiguration(ActivityCheckConfiguration actvitivyCheckConfiguration) {
            SDKPreconditions.checkNotNull(actvitivyCheckConfiguration, "Beacon activity check is null.");

            this.activityCheckConfiguration = actvitivyCheckConfiguration;
            return this;
        }

        /**
         * Sets Scan Mode. Android L comes with functionality
         * controlling scan performance. there are three predefined
         * variables according to which device scan may be handled
         * depending on user's choice.
         * Our SDK takes advantage fo this feature for devices running
         * on Android Lollipop and higher.
         * For devices running on Android OS with version lower than
         * Lollipop this parameter does not affect scan performance.
         *
         * You can set one of three values:
         *
         * {@link ProximityManager#SCAN_MODE_BALANCED} - perform Bluetooth LE scan in balanced power mode.
         * {@link ProximityManager#SCAN_MODE_LOW_LATENCY} - scan using highest duty cycle.
         * {@link ProximityManager#SCAN_MODE_LOW_POWER} - perform Bluetooth LE scan in low power mode.
         *
         * @param scanMode the scan mode
         * @return the builder instance
         */
        public Builder setScanMode(final int scanMode) {
            SDKPreconditions.checkArgument(
                    (scanMode == ProximityManager.SCAN_MODE_BALANCED) ||
                            (scanMode == ProximityManager.SCAN_MODE_LOW_LATENCY) ||
                            (scanMode == ProximityManager.SCAN_MODE_LOW_POWER),
                    "Unknown scan mode specified: " + scanMode);
            this.scanMode = scanMode;
            return this;
        }

        /**
         * Sets ForceScanConfiguration ({@link ForceScanConfiguration}).
         *
         * @param forceScanConfiguration the force scan configuration
         * @return the builder instance
         */
        public Builder setForceScanConfiguration(ForceScanConfiguration forceScanConfiguration) {
            SDKPreconditions.checkNotNull(forceScanConfiguration, "By default ForceScanConfiguration is disabled");

            this.forceScanConfiguration = forceScanConfiguration;
            return this;
        }

        /**
         * Sets {@link ScanPeriod}.
         *
         *
         * @param scanPeriod the monitor period
         * @return the builder instance
         */
        public Builder setScanPeriod(ScanPeriod scanPeriod) {
            SDKPreconditions.checkNotNull(scanPeriod, "Monitor period cannot be null");
            ScanContextValidator.validate(scanPeriod);
            this.scanPeriod = scanPeriod;
            return this;
        }

        /**
         * Builds global scan context instance.
         *
         * @return new immutable configuration instance
         */
        public ScanContext build() {

            if (activityCheckConfiguration != ActivityCheckConfiguration.DISABLED) {
                ScanContextValidator.validate(activityCheckConfiguration);
            }

            if (forceScanConfiguration != ForceScanConfiguration.DISABLED) {
                ScanContextValidator.validate(forceScanConfiguration);
            }
            String message = "At least one of: [IBeaconScanContext, EddystoneScanContext] must be set";
            SDKPreconditions.checkNotAllNull(new IllegalStateException(message),
                    iBeaconScanContext,
                    eddystoneScanContext);

            return new ScanContext(this);
        }
    }
}
