package com.kidoz.sdk.api.ui_views.video_unit;

import android.app.Activity;
import android.content.Context;
import android.content.SharedPreferences;
import android.content.res.Configuration;
import android.graphics.Color;
import android.graphics.Point;
import android.os.AsyncTask;
import android.os.Build;
import android.os.Bundle;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.RelativeLayout;

import com.kidoz.events.EventBulk;
import com.kidoz.events.EventManager;
import com.kidoz.events.EventParameters;
import com.kidoz.sdk.api.fullscreen_video_layer.FullScreenVideoEnabledWebChromeClient;
import com.kidoz.sdk.api.fullscreen_video_layer.FullScreenVideoEnabledWebView;
import com.kidoz.sdk.api.general.ContentExecutionHandler;
import com.kidoz.sdk.api.general.EventMessage;
import com.kidoz.sdk.api.general.assets_handling.AssetUtil;
import com.kidoz.sdk.api.general.database.DatabaseManager;
import com.kidoz.sdk.api.general.enums.ContentType;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.general.utils.ConstantDef;
import com.kidoz.sdk.api.general.utils.PropertiesObj;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.structure.ContentItem;
import com.kidoz.sdk.api.ui_views.html_view.HtmlViewWrapper;
import com.kidoz.sdk.api.ui_views.loading_progress_view.LoadingProgressView;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;
import org.json.JSONObject;

import java.lang.ref.WeakReference;

public class VideoUnitActivity extends Activity
{
    public static final String VIDEO_UNIT_ACTION_BUTTON_TAG = "VIDEO_UNIT_ACTION_BUTTON_TAG";
    public static final String VIDEO_UNIT_URL_TAG = "VIDEO_UNIT_URL_TAG";

    private final String TAG = VideoUnitActivity.class.getSimpleName();
    public static FullScreenVideoEnabledWebView mWebView;
    public static VideoUnit.VideoUnitListener sVideoUnitListener;
    private static RelativeLayout sLoadingViewContainer;
    private static LoadingProgressView sLoadingProgressView;
    private boolean mIsAlreadyLoaded = false;
    private int mDeviceOrientation;
    private RedirectManager mRedirectManager;

    @Override
    public void onConfigurationChanged(Configuration newConfig)
    {
        super.onConfigurationChanged(newConfig);
    }

    @Override
    protected void onCreate(Bundle savedInstanceState)
    {
        super.onCreate(savedInstanceState);

        if (EventBus.getDefault().isRegistered(this) == false)
        {
            EventBus.getDefault().register(this);
        }

        try
        {
            mDeviceOrientation = getRequestedOrientation();
            initActivity();
            if (mWebView != null)
            {
                invokeJavaScript("javascript:onWidgetOpen();");
                if (sVideoUnitListener != null && sVideoUnitListener != null)
                {
                    sVideoUnitListener.onOpen();
                }
            } else
            {
                finish();
            }
        } catch (Exception ex)
        {
            SDKLogger.printErrorLog(TAG, "Exception = " + ex);
        }
    }

    @Override
    protected void onDestroy()
    {
        super.onDestroy();
        try
        {
            invokeJavaScript("javascript:onWidgetClose();");
            if (sVideoUnitListener != null && sVideoUnitListener != null)
            {
                sVideoUnitListener.onClose();
            }
            mWebView = null;
            sVideoUnitListener = null;
        } catch (Exception ex)
        {
            SDKLogger.printErrorLog(TAG, "Exception = " + ex);
        }

        if (EventBus.getDefault().isRegistered(this) == true)
        {
            EventBus.getDefault().unregister(this);
        }
    }

    @Override
    protected void onStop()
    {
        super.onStop();
        setRequestedOrientation(mDeviceOrientation);
        mRedirectManager.stopRedirectHandler();
    }

    @Override
    protected void onResume()
    {
        super.onResume();
        try
        {
            initFullScreenFlags();

            if (mWebView.getParent() != null)
            {
                mWebView.getParent().requestLayout();
            }
            invokeJavaScript("javascript:onWidgetResume();");
        } catch (Exception ex)
        {
            SDKLogger.printErrorLog(TAG, "Exception = " + ex);
        }
    }

    @Override
    protected void onPause()
    {
        super.onPause();
        try
        {
            invokeJavaScript("javascript:onWidgetPause();");
        } catch (Exception ex)
        {
            SDKLogger.printErrorLog(TAG, "Exception = " + ex);
        }
    }

    private void initActivity()
    {
        if (getIntent() != null)
        {
            mIsAlreadyLoaded = getIntent().getBooleanExtra("isLoaded", false);
        }
        initFullScreenFlags();
        initWebView();
        initRedirectManager();
        initLoadingProgressView();
    }

    private void initLoadingProgressView()
    {
        sLoadingViewContainer = new RelativeLayout(this);
        sLoadingViewContainer.setOnClickListener(new View.OnClickListener()
        {
            @Override
            public void onClick(View v)
            {

            }
        });
        sLoadingViewContainer.setBackgroundColor(Color.parseColor("#80000000"));

        sLoadingProgressView = new LoadingProgressView(this);
        sLoadingProgressView.setCircleColor(Color.parseColor("#289E9E"));
        Point point = Utils.getScreenSize(this);
        int size = (int) (Math.min(point.x, point.y) * 0.35);
        sLoadingProgressView.setCircleWidthRelativeToSize(size);
        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(size, size);
        params.addRule(RelativeLayout.CENTER_IN_PARENT);
        sLoadingViewContainer.addView(sLoadingProgressView, params);

        ((ViewGroup) mWebView.getParent()).addView(sLoadingViewContainer, new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT));
        toggleLoadingView(false);
    }

    private void initRedirectManager()
    {
        mRedirectManager = new RedirectManager(this);
        mRedirectManager.setRedirectManagerListener(new RedirectManager.RedirectManagerListener()
        {
            @Override
            public void onRedirectStarted()
            {
                toggleLoadingView(true);
            }

            @Override
            public void onRedirectEnd(boolean isError)
            {
                toggleLoadingView(false);
            }
        });
    }

    private void forwardToGooglePlay(final String advertiserID, final String appID, final String googlePlayLink, final String widgetType, final String styleID, final String itemIndex)
    {
        runOnUiThread(new Runnable()
        {
            @Override
            public void run()
            {
                int itemIndexAsInteger = 0;
                try
                {
                    itemIndexAsInteger = Integer.parseInt(itemIndex);
                } catch (Exception ex)
                {
                    SDKLogger.printErrorLog(TAG, "Error when trying to parse item index: " + ex.getMessage());
                }
                ContentItem contentItem = new ContentItem();
                contentItem.setId(appID);
                contentItem.setName("");
                contentItem.setData(googlePlayLink);
                contentItem.setContentType(ContentType.PROMOTED_PLAY_APPLICATION);
                contentItem.setAdvertiserID(advertiserID);
                contentItem.setIsPromoted(true);

                ContentExecutionHandler.handleContentItemClick(VideoUnitActivity.this, contentItem, widgetType, styleID, itemIndexAsInteger, false, null);
            }
        });
    }

    private void initWebView()
    {
        if (mWebView != null)
        {
            if (mWebView.getParent() != null)
            {
                ((ViewGroup) mWebView.getParent()).removeAllViews();
            }
            if (Build.VERSION.SDK_INT >= 19)
            {
                mWebView.setWebContentsDebuggingEnabled(ConstantDef.DGM);
            }
            setContentView(mWebView, new ViewGroup.LayoutParams(ViewGroup.MarginLayoutParams.MATCH_PARENT, ViewGroup.MarginLayoutParams.MATCH_PARENT));
        }
        if (mIsAlreadyLoaded == false)
        {
            initWebViewSettings();
            VideoUnit.misLoaded = true;
        }
    }

    private void initWebViewSettings()
    {
        if (mWebView != null)
        {
            mWebView.initWebViewSettings();
            mWebView.addJavascriptInterface(new VideoUnitJavascriptInterface(), "VideoUnit");
        }

        PropertiesObj propertiesObj = DatabaseManager.getInstance(VideoUnitActivity.this).getConfigTable().loadAppProperties();
        if (propertiesObj != null)
        {
            String videoUnitURL = propertiesObj.getVideoUnitURL();
            StringBuilder stringBuilder = HtmlViewWrapper.populateMustHaveParams(VideoUnitActivity.this, videoUnitURL);

            stringBuilder.append("&style_id=" + 0);
            stringBuilder.append("&widget_type=").append(WidgetType.WIDGET_TYPE_VIDEO_UNIT.getStringValue());

            SharedPreferences sharedPreferences = VideoUnitActivity.this.getSharedPreferences(EventManager.EVENTS_SHARED_PREFERENCES_KEY, 0);
            long appSessionID = sharedPreferences.getLong(EventManager.EVENTS_LAST_SESSION_ID_KEY, 0);
            stringBuilder.append("&appSessionID=").append(appSessionID);

            if (mWebView != null)
            {
                mWebView.setWebViewClient(new WebViewClient()
                {
                    @Override
                    public boolean shouldOverrideUrlLoading(WebView view, String url)
                    {
                        SDKLogger.printDebbugLog(TAG, "shouldOverrideUrlLoading: URL = " + url);
                        mRedirectManager.loadURL(url);
                        return true;
                    }

                    @Override
                    public void onReceivedError(WebView view, int errorCode, String description, String failingUrl)
                    {
                        toggleLoadingView(false);
                        EventManager.getInstance(VideoUnitActivity.this).logEvent(VideoUnitActivity.this, WidgetType.WIDGET_TYPE_VIDEO_UNIT.getStringValue(), "0", EventManager.LOG_CRITICAL_LEVEL, EventParameters.CATEGORY_WEB_VIEW_ERROR, String.valueOf(errorCode).concat(": ").concat(description), failingUrl);
                    }
                });

                FullScreenVideoEnabledWebChromeClient webChromeClient = new FullScreenVideoEnabledWebChromeClient(null, null, null, mWebView)
                {
                };

                webChromeClient.setOnToggledFullscreen(new FullScreenVideoEnabledWebChromeClient.ToggledFullscreenCallback()
                {
                    @Override
                    public void toggledFullscreen(boolean fullscreen)
                    {

                    }
                });
                mWebView.setWebChromeClient(webChromeClient);

                mWebView.loadUrl(stringBuilder.toString());
            }
        }
    }

    private void initFullScreenFlags()
    {
        if (Build.VERSION.SDK_INT < 16)
        {
            getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN, WindowManager.LayoutParams.FLAG_FULLSCREEN);
        } else
        {
            if (Build.VERSION.SDK_INT < 18)
            {
                View decorView = getWindow().getDecorView();
                int uiOptions = View.SYSTEM_UI_FLAG_FULLSCREEN;
                decorView.setSystemUiVisibility(uiOptions);
            } else
            {
                final View decorView = getWindow().getDecorView();
                decorView.setSystemUiVisibility(View.SYSTEM_UI_FLAG_LAYOUT_STABLE | View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN | View.SYSTEM_UI_FLAG_HIDE_NAVIGATION | View.SYSTEM_UI_FLAG_FULLSCREEN | View.SYSTEM_UI_FLAG_IMMERSIVE_STICKY);
                decorView.setOnSystemUiVisibilityChangeListener(new View.OnSystemUiVisibilityChangeListener()
                {
                    @Override
                    public void onSystemUiVisibilityChange(int visibility)
                    {
                        if ((visibility & View.SYSTEM_UI_FLAG_FULLSCREEN) == 0)
                        {
                            decorView.setSystemUiVisibility(View.SYSTEM_UI_FLAG_LAYOUT_STABLE | View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN | View.SYSTEM_UI_FLAG_HIDE_NAVIGATION | View.SYSTEM_UI_FLAG_FULLSCREEN | View.SYSTEM_UI_FLAG_IMMERSIVE_STICKY);
                        }
                    }
                });
            }
        }
    }

    private void invokeJavaScript(final String javascript)
    {
        runOnUiThread(new Runnable()
        {
            @Override
            public void run()
            {
                if (mWebView != null)
                {
                    try
                    {

                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT)
                        {
                            mWebView.evaluateJavascript(javascript, null);
                        } else
                        {
                            mWebView.loadUrl(javascript);
                        }
                    } catch (Exception ex)
                    {
                        mWebView.loadUrl(javascript);
                    }
                }
            }
        });
    }

    public class VideoUnitJavascriptInterface
    {
        @android.webkit.JavascriptInterface
        public void onVideoUnitReady()
        {
        }

        @android.webkit.JavascriptInterface
        public void onCloseVideoUnit()
        {
            runOnUiThread(new Runnable()
            {
                @Override
                public void run()
                {
                    EventMessage widgetEventMessage = new EventMessage(EventMessage.MessageType.CLOSE_VIDEO_UNIT);
                    EventBus.getDefault().post(widgetEventMessage);
                }
            });
        }

        @android.webkit.JavascriptInterface
        public void forwardToGooglePlay(final String advertiserID, final String appID, final String googlePlayLink, final String widgetType, final String styleID, final String itemIndex)
        {
            runOnUiThread(new Runnable()
            {
                @Override
                public void run()
                {
                    VideoUnitActivity.this.forwardToGooglePlay(advertiserID, appID, googlePlayLink, widgetType, styleID, itemIndex);
                }
            });
        }

        @android.webkit.JavascriptInterface
        public void setOrientation(final int orientation)
        {
            runOnUiThread(new Runnable()
            {
                @Override
                public void run()
                {
                    setRequestedOrientation(orientation);
                }
            });
        }

        @android.webkit.JavascriptInterface
        public void sendItemDetailsToClient(final String itemDetailsJson)
        {
            runOnUiThread(new Runnable()
            {
                @Override
                public void run()
                {
                    mRedirectManager.setItemDetails(itemDetailsJson);
                }
            });
        }
    }

    public static void parseVideoUnitStyle(final Context context, final JSONObject json)
    {
        new Thread(new Runnable()
        {
            @Override
            public void run()
            {
                VideoUnitParserAsyncTask asyncTask = new VideoUnitParserAsyncTask(context,json);
                if (Build.VERSION.SDK_INT < 11)
                {
                    asyncTask.execute();
                } else
                {
                    asyncTask.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR);
                }
            }
        }).start();
    }



    private void toggleLoadingView(boolean isNeedToShow)
    {
        if (isNeedToShow == true)
        {
            sLoadingProgressView.startLoadingAnimation();
            sLoadingViewContainer.setVisibility(View.VISIBLE);
        } else
        {
            sLoadingProgressView.stopLoadingAnimation();
            sLoadingViewContainer.setVisibility(View.INVISIBLE);
        }
    }

    @Subscribe
    public void onHandleEvent(EventMessage event)
    {
        if (event.getMessageType() == EventMessage.MessageType.WEB_PLAYER_DIALOG_CLOSED)
        {
            if (mWebView.getParent() != null)
            {
                mWebView.getParent().requestLayout();
            }

            invokeJavaScript("javascript:onWidgetResume();");
        } else if (event.getMessageType() == EventMessage.MessageType.CLOSE_VIDEO_UNIT)
        {
            finish();
        }
    }

    public static class VideoUnitParserAsyncTask extends AsyncTask<EventBulk, Void, Boolean>
    {
        WeakReference<Context> weakActivity = null;
        JSONObject mJsonObject;

        public VideoUnitParserAsyncTask(Context context, JSONObject json)
        {
            weakActivity = new WeakReference<>(context);
            mJsonObject = json;
        }

        @Override
        protected Boolean doInBackground(EventBulk... params)
        {
            boolean isNeedRefresh = false;
            String actionButtonURL = null;
            if (mJsonObject != null)
            {
                try
                {
                    JSONObject videoUnitStyle = mJsonObject.getJSONObject("videoUnitStyle");
                    JSONObject videoUnit = videoUnitStyle.getJSONObject("videoUnit");
                    actionButtonURL = videoUnit.getString("action_button_url");
                } catch (Exception ex)
                {

                }

                if (actionButtonURL != null)
                {
                    boolean isNeedUpdate = false;

                    JSONObject jsonObject = DatabaseManager.getInstance(weakActivity.get()).getConfigTable().loadProperties(VIDEO_UNIT_ACTION_BUTTON_TAG);
                    if (jsonObject != null)
                    {
                        try
                        {
                            String oldActionButtonURL = jsonObject.getString(VIDEO_UNIT_ACTION_BUTTON_TAG);
                            if (oldActionButtonURL != null && oldActionButtonURL.equals(actionButtonURL) == false)
                            {
                                // Need to update the action button.
                                isNeedUpdate = true;
                                AssetUtil.deleteAssetFile2(weakActivity.get(), oldActionButtonURL);
                            }
                        } catch (Exception ex)
                        {
                        }
                    } else
                    {
                        // This is the first action button update.
                        isNeedUpdate = true;
                        isNeedRefresh = true;
                    }

                    if (isNeedUpdate == true)
                    {
                        AssetUtil.createAssetFile2(weakActivity.get(), actionButtonURL);
                        jsonObject = new JSONObject();
                        try
                        {
                            jsonObject.put(VIDEO_UNIT_ACTION_BUTTON_TAG, actionButtonURL);
                            DatabaseManager.getInstance(weakActivity.get()).getConfigTable().insertProperties(VIDEO_UNIT_ACTION_BUTTON_TAG, jsonObject);
                        } catch (Exception ex)
                        {
                        }
                    }
                }
            }

            if (weakActivity != null)
            {
                weakActivity.clear();
            }
            return isNeedRefresh;
        }

        @Override
        protected void onPostExecute(Boolean needTofresh)
        {
            super.onPostExecute(needTofresh);

            if (needTofresh)
            {
                EventMessage widgetEventMessage = new EventMessage(EventMessage.MessageType.VIDEO_UNIT_BUTTON_READY);
                EventBus.getDefault().post(widgetEventMessage);
            }
        }
    }
}
