package com.kidoz.sdk.api.ui_views.video_unit;

import android.content.Context;
import android.content.Intent;
import android.content.pm.ResolveInfo;
import android.net.Uri;
import android.os.Handler;
import android.util.Log;
import android.webkit.WebView;
import android.webkit.WebViewClient;

import com.kidoz.events.EventManager;
import com.kidoz.events.EventParameters;
import com.kidoz.sdk.api.fullscreen_video_layer.FullScreenVideoEnabledWebView;
import com.kidoz.sdk.api.general.database.DatabaseManager;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.structure.IsEventRecord;

import org.json.JSONObject;

import java.util.List;

/**
 * Created by RonyBrosh on 1/10/2017.
 */

public class RedirectManager
{
    private final String TAG = RedirectManager.class.getSimpleName();
    private final long MAX_REDIRECT_TIME = 5000;

    private String[] mAppStoreArray;
    private FullScreenVideoEnabledWebView mWebView;
    private String mGooglePlayAppPrefix = "market://details?id=";
    private String mGooglePlayWebsitePrefix = "https://play.google.com/store/apps/details?id=";
    private String mYoutubePrefix = "youtube.com";
    private Handler mHandler;
    private Runnable mHandlerRunnable;
    private String mLastLoadedURL;
    private IsEventRecord mIsEventRecord;
    private RedirectManagerListener mRedirectManagerListener;

    public RedirectManager(Context context)
    {
        init(context);
    }

    public void setRedirectManagerListener(RedirectManagerListener redirectManagerListener)
    {
        mRedirectManagerListener = redirectManagerListener;
    }

    /*
    *   Setting the ad item details, so in case of click on app item the item details will be saved.
    */
    public void setItemDetails(String itemDetailsJson)
    {
        mIsEventRecord = null;
        try
        {
            if (itemDetailsJson != null)
            {
                JSONObject json = new JSONObject(itemDetailsJson);
                mIsEventRecord = new IsEventRecord();
                mIsEventRecord.setPackageName(json.optString("packageName"));
                mIsEventRecord.setContentType(json.optString("contentType"));
                mIsEventRecord.setName(json.optString("name"));
                mIsEventRecord.setPositionIndex(json.optInt("position"));
                mIsEventRecord.setAdvertiserId(json.optString("advertiserId"));
                mIsEventRecord.setTimeStamp(System.currentTimeMillis() + "");
                mIsEventRecord.setWidgetType(json.optString("widgetType"));
                mIsEventRecord.setStyleId(json.optString("styleId"));
            }
        }
        catch (Exception ex)
        {
        }
    }

    public void stopRedirectHandler()
    {
        mHandler.removeCallbacks(mHandlerRunnable);
    }

    private void init(Context context)
    {
        mAppStoreArray = new String[]{mGooglePlayAppPrefix, mGooglePlayWebsitePrefix};
        mHandler = new Handler();
        mHandlerRunnable = new Runnable()
        {
            @Override
            public void run()
            {
                launchURLIntent(mLastLoadedURL);
            }
        };
        initWebView(context);
    }

    private boolean getIsAppStoreLink(String url)
    {
        boolean result = false;
        if (mIsEventRecord != null && mIsEventRecord.getPackageName() != null)
        {
            for (String nextStorePrefix : mAppStoreArray)
            {
                if (url.contains(nextStorePrefix) == true)
                {
                    result = true;
                    break;
                }
            }
        }
        return result;
    }

    private void initWebView(Context context)
    {
        mWebView = new FullScreenVideoEnabledWebView(context);
        mWebView.initWebViewSettings();
        mWebView.setWebViewClient(new WebViewClient()
        {
            @Override
            public boolean shouldOverrideUrlLoading(WebView view, String url)
            {
                boolean result = false;
                mLastLoadedURL = url;
                stopRedirectHandler();
                if (getIsAppStoreLink(url) == true)
                {
                    launchURLIntent(url);
                    result = true;
                }
                else
                {
                    restartRedirectHandler();
                }
                return result;
            }

            @Override
            public void onReceivedError(WebView view, int errorCode, String description, String failingUrl)
            {
                stopRedirectHandler();
                notifyRedirectFinished(true);
            }
        });
    }

    private void restartRedirectHandler()
    {
        stopRedirectHandler();
        mHandler.postDelayed(mHandlerRunnable, MAX_REDIRECT_TIME);
    }

    private void notifyRedirectFinished(boolean isError)
    {
        if (mRedirectManagerListener != null)
        {
            mRedirectManagerListener.onRedirectEnd(isError);
        }
    }

    private void launchURLIntent(String url)
    {
        boolean isCanLaunchIntent = false;
        if (mIsEventRecord == null || mIsEventRecord.getPackageName() == null)
        {
            // Open the url in browser.
            isCanLaunchIntent = true;
        }
        else
        {
            // It's an app so verify it's the expected app.
            if (url.contains(mIsEventRecord.getPackageName()) == true)
            {
                isCanLaunchIntent = true;
                if (url.contains(mGooglePlayWebsitePrefix) == true)
                {
                    String googlePlayAppURL = url.replace(mGooglePlayWebsitePrefix, mGooglePlayAppPrefix);
                    Intent googlePlayAppIntent = new Intent(Intent.ACTION_VIEW);
                    googlePlayAppIntent.setData(Uri.parse(googlePlayAppURL));
                    List<ResolveInfo> intentArray = mWebView.getContext().getPackageManager().queryIntentActivities(googlePlayAppIntent, 0);
                    if (intentArray != null || intentArray.isEmpty() == false)
                    {
                        url = googlePlayAppURL;
                    }
                }
            }
        }
        if (isCanLaunchIntent == true)
        {
            saveItemDetailsToDataBase();

            Intent intent = new Intent(Intent.ACTION_VIEW);
            intent.setData(Uri.parse(url));
            mWebView.getContext().startActivity(intent);
            notifyRedirectFinished(false);
        }
        else
        {
            notifyRedirectFinished(true);
        }
    }

    private void saveItemDetailsToDataBase()
    {
        try
        {
            if (mIsEventRecord != null)
            {
                DatabaseManager.getInstance(mWebView.getContext()).getIsEventTable().insertRecord(mIsEventRecord);
            }
        }
        catch (Exception ex)
        {
        }
    }

    public void loadURL(String url)
    {
        mLastLoadedURL = url;
        stopRedirectHandler();
        // Don't even try to load youtube links :)
        if(url.contains(mYoutubePrefix) == false)
        {
            if(mRedirectManagerListener != null)
            {
                mRedirectManagerListener.onRedirectStarted();
            }

            if (getIsAppStoreLink(url) == true)
            {
                launchURLIntent(url);
            }
            else
            {
                restartRedirectHandler();
                mWebView.loadUrl(url);
            }
        }
    }

    public interface RedirectManagerListener
    {
        void onRedirectStarted();

        void onRedirectEnd(boolean isError);
    }
}
