package com.kidoz.sdk.api.ui_views.custom_drawables;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.ColorFilter;
import android.graphics.CornerPathEffect;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.PixelFormat;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.drawable.Drawable;

import com.kidoz.sdk.api.general.utils.ScreenUtils;
import com.kidoz.sdk.api.ui_views.panel_view.PANEL_TYPE;

public class DefaultPanelDrawable extends Drawable
{
    private static final float SIZE_RATIO = 0.54f;
    private static final float LINE_WIDTH = 0.258f;

    private static final float CLOSE_BUTTON_SIZE_RATIO_RELATIVE_TO_OPEN = 0.60f;

    public enum PanelDrawableType
    {
        OPEN_VIEW, CLOSE_VIEW
    }

    private Paint mXPaint;
    private Paint mBgPaint;
    private int mWidth;
    private int mHeight;

    private RectF mOpenBackgroundRoundedRectF;
    private RectF mOpenBackgroundRectF;

    private RectF mCloseBackgroundRoundedRectF;
    private RectF mCloseBackgroundRectF;

    private int mBaseColor;
    private int mXcolor;
    private PANEL_TYPE mPanelType;
    private PanelDrawableType mDrawableType;

    private Path mPlus = new Path();
    private Path mMinus = new Path();
    private boolean mDrawSign = true;

    public DefaultPanelDrawable(Context context, PANEL_TYPE panel_type, PanelDrawableType panelDrawableType)
    {
        float scaleFactor = ScreenUtils.getScaleFactor(context);

        mWidth *= scaleFactor;
        mHeight *= scaleFactor;

        mBaseColor = Color.parseColor("#ffffff");
        mXcolor = Color.parseColor("#cdffffff");

        mXPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mXPaint.setStyle(Paint.Style.FILL_AND_STROKE);
        mXPaint.setStrokeJoin(Paint.Join.ROUND);    // set the join to round you want
        mXPaint.setStrokeCap(Paint.Cap.ROUND);      // set the paint cap to round too
        mXPaint.setPathEffect(new CornerPathEffect(15));
        mXPaint.setColor(mXcolor);
        mXPaint.setShadowLayer(4.5f, 0, 1.5f, 0x45000000);
        mXPaint.setDither(true);

        mBgPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mBgPaint.setStyle(Paint.Style.FILL_AND_STROKE);
        mBgPaint.setColor(mBaseColor);
        mBgPaint.setDither(true);

        mPanelType = panel_type;

        mDrawableType = panelDrawableType;
        initRectF();
    }

    @Override
    public void setAlpha(int alpha)
    {
        mBgPaint.setAlpha(alpha);
        mXPaint.setAlpha(alpha);
    }

    @Override
    public void setColorFilter(ColorFilter cf)
    {
        mBgPaint.setColorFilter(cf);
        mXPaint.setColorFilter(cf);
    }

    @Override
    public int getOpacity()
    {
        return PixelFormat.TRANSLUCENT;
    }

    @Override
    protected void onBoundsChange(Rect bounds)
    {
        super.onBoundsChange(bounds);

        mWidth = bounds.width();
        mHeight = bounds.height();

        initRectF();
    }

    private void initRectF()
    {
        switch (mPanelType)
        {
            case TOP:
            {
                if (mDrawableType == PanelDrawableType.OPEN_VIEW)
                {
                    mOpenBackgroundRectF = new RectF(0, 0, mWidth, mHeight / 2f);
                    mOpenBackgroundRoundedRectF = new RectF(0, 0, mWidth, mHeight);
                    generatePlusButtonPath(mWidth / 2f, mHeight / 2f);
                } else
                {
                    mCloseBackgroundRectF = new RectF(0, 0, mWidth, (mHeight - mHeight * CLOSE_BUTTON_SIZE_RATIO_RELATIVE_TO_OPEN) / 1.3f);
                    mCloseBackgroundRoundedRectF = new RectF(0, 0, mWidth, mHeight - mHeight * CLOSE_BUTTON_SIZE_RATIO_RELATIVE_TO_OPEN);

                    float centerY = (mHeight - mHeight * CLOSE_BUTTON_SIZE_RATIO_RELATIVE_TO_OPEN) / 2;

                    generateMinusButtonPath(mWidth / 2f, centerY);
                }

                break;
            }
            case BOTTOM:
            {
                if (mDrawableType == PanelDrawableType.OPEN_VIEW)
                {
                    mOpenBackgroundRectF = new RectF(0, mHeight / 2f, mWidth, mHeight);
                    mOpenBackgroundRoundedRectF = new RectF(0, 0, mWidth, mHeight);

                    generatePlusButtonPath(mWidth / 2f, mHeight / 2f);
                } else
                {
                    mCloseBackgroundRectF = new RectF(0, mHeight * CLOSE_BUTTON_SIZE_RATIO_RELATIVE_TO_OPEN * 1.3f, mWidth, mHeight);
                    mCloseBackgroundRoundedRectF = new RectF(0, mHeight * CLOSE_BUTTON_SIZE_RATIO_RELATIVE_TO_OPEN, mWidth, mHeight);

                    float start = mHeight * CLOSE_BUTTON_SIZE_RATIO_RELATIVE_TO_OPEN;
                    float centerY = start + (mHeight - start) / 2;

                    generateMinusButtonPath(mWidth / 2f, centerY);
                }

                break;
            }
        }
    }

    private void generatePlusButtonPath(float centerX, float centerY)
    {
        float size = mWidth * SIZE_RATIO;
        float width = mWidth * LINE_WIDTH;

        float offsetSize = size / 2f - width / 3.5f;
        float offsetWidth = width / 2f;

        mPlus.reset();

        // Create Plus draw path
        mPlus.moveTo(centerX - offsetWidth / 2f, centerY + offsetWidth / 2f);
        mPlus.lineTo(centerX - offsetSize - offsetWidth / 2f, centerY + offsetWidth / 2f);
        mPlus.lineTo(centerX - offsetSize - offsetWidth / 2f, centerY - offsetWidth / 2f);

        mPlus.lineTo(centerX - offsetWidth / 2f, centerY - offsetWidth / 2f);
        mPlus.lineTo(centerX - offsetWidth / 2f, centerY - offsetWidth / 2f - offsetSize);
        mPlus.lineTo(centerX + offsetWidth / 2f, centerY - offsetWidth / 2f - offsetSize);
        mPlus.lineTo(centerX + offsetWidth / 2f, centerY - offsetWidth / 2f);

        mPlus.lineTo(centerX + offsetWidth / 2f + offsetSize, centerY - offsetWidth / 2f);
        mPlus.lineTo(centerX + offsetWidth / 2f + offsetSize, centerY + offsetWidth / 2f);
        mPlus.lineTo(centerX + offsetWidth / 2f, centerY + offsetWidth / 2f);

        mPlus.lineTo(centerX + offsetWidth / 2f, centerY + offsetWidth / 2f + offsetSize);
        mPlus.lineTo(centerX - offsetWidth / 2f, centerY + offsetWidth / 2f + offsetSize);
        mPlus.lineTo(centerX - offsetWidth / 2f, centerY + offsetWidth / 2f);
        mPlus.lineTo(centerX - offsetSize - offsetWidth / 2f, centerY + offsetWidth / 2f);
    }

    private void generateMinusButtonPath(float centerX, float centerY)
    {
        float size = mWidth * SIZE_RATIO;
        float width = mWidth * LINE_WIDTH;

        float offsetSize = size / 2f - width / 2.6f;
        float offsetWidth = width / 2f;

        mMinus.reset();

        mMinus.moveTo(centerX - offsetSize - offsetWidth / 2f, centerY + offsetWidth / 2f);
        mMinus.lineTo(centerX - offsetSize - offsetWidth / 2f, centerY - offsetWidth / 2f);
        mMinus.lineTo(centerX + offsetSize + offsetWidth / 2f, centerY - offsetWidth / 2f);
        mMinus.lineTo(centerX + offsetSize + offsetWidth / 2f, centerY + offsetWidth / 2f);
        mMinus.lineTo(centerX - offsetSize - offsetWidth / 2f, centerY + offsetWidth / 2f);
        mMinus.lineTo(centerX - offsetSize - offsetWidth / 2f, centerY - offsetWidth / 2f);
    }

    @Override
    public void draw(Canvas canvas)
    {
        mBgPaint.setColor(mBaseColor);
        mXPaint.setColor(mXcolor);

        float radius = (20f / 142) * mWidth;

        switch (mDrawableType)
        {
            case OPEN_VIEW:
            {
                canvas.drawRect(mOpenBackgroundRectF, mBgPaint);
                canvas.drawRoundRect(mOpenBackgroundRoundedRectF, radius, radius, mBgPaint);
                if (mDrawSign)
                {
                    canvas.drawPath(mPlus, mXPaint);
                }
                break;
            }

            case CLOSE_VIEW:
            {
                canvas.drawRect(mCloseBackgroundRectF, mBgPaint);
                canvas.drawRoundRect(mCloseBackgroundRoundedRectF, radius, radius, mBgPaint);

                if (mDrawSign)
                {
                    canvas.drawPath(mMinus, mXPaint);
                }
                break;
            }
        }
    }

    public void setPanelType(PANEL_TYPE panelType)
    {
        mPanelType = panelType;
        initRectF();
        invalidateSelf();
    }

    public void setBaseColor(int baseColor)
    {
        mBaseColor = baseColor;
        mBgPaint.setColor(baseColor);
    }

    public void setDrawSign(boolean draw)
    {
        mDrawSign = draw;
        invalidateSelf();
    }
}