package com.kidoz.sdk.api.server_connect;

import android.content.ContentValues;
import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.graphics.Point;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Build;
import android.telephony.TelephonyManager;
import android.util.Log;

import com.kidoz.events.EventBulk;
import com.kidoz.sdk.api.general.utils.ConstantDef;
import com.kidoz.sdk.api.general.utils.EncoderUtils;
import com.kidoz.sdk.api.general.utils.PropertiesObj;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.ScreenUtils;
import com.kidoz.sdk.api.general.utils.SdkDeviceUtils;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.structure.ContentData;

import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Locale;
import java.util.Set;

public class SdkAPIManager extends BaseAPIManager
{
    private static final String TAG = SdkAPIManager.class.getName();

    private String mPackageName;
    private boolean mDeveloperLogging;
    private String mCalculatedDeviceHash = null;

    private static SdkAPIManager mSdkInstnace;

    public static SdkAPIManager getSdkApiInstance(Context context)
    {
        if (mSdkInstnace == null)
        {
            mSdkInstnace = new SdkAPIManager(context);
        }
        return mSdkInstnace;
    }

    public static void init(Context context, String pId, String sKey, boolean developerLogging)
    {
        getSdkApiInstance(context).setProperties(pId, sKey, developerLogging);
    }

    public SdkAPIManager(Context context)
    {
        initConnectionClient(context);
        mPackageName = context.getPackageName();
    }

    private void setProperties(String pId, String sKey, boolean developerLogging)
    {
        mPublicherId = pId;
        mSecurityToken = sKey;
        mDeveloperLogging = developerLogging;
    }

    //SDK ApiApi Functions
    private static final String JSON = "JSON";
    private static final String REQUEST_FORMAT = "resFormat";

    private static final String GET_CONTENT = "getContent";
    private static final String WATERFALL = "waterfall";
    private static final String INIT_SDK_CONFIGURATION = "initSDK";
    private static final String SEND_KIDOZ_EVENTS_TO_SERVER = "SendKidozEventsToServer";

    public static final String METHOD_NAME = "method_name";

    /**
     * Load content
     *
     * @param styleId        content view style
     * @param resultCallback result callback
     */
    public void loadContent(Context context, String styleId, String widgetType, final ApiResultCallback<ContentData> resultCallback)
    {
        ContentValues contentValues = new ContentValues();
        contentValues.put(METHOD_NAME, GET_CONTENT);

        setGeneralParams(context, contentValues);

        contentValues.put("style_id", styleId);
        contentValues.put("widget_type", widgetType);

        long ts = System.currentTimeMillis() / 1000;
        contentValues.put("ts", ts);
        contentValues.put("hash", generateRequestHash(contentValues, ts, mSecurityToken));

        startServerConnection(context, HTTP_SCHEME + DOMAIN + MAIN_API, CONNECTION_TYPE.GET, SdkRequestType.LOAD_SDK_CONTENT, contentValues, 1, resultCallback, mDeveloperLogging, false);
    }

    /**
     * loadWaterfall
     *
     * @param context           the current context
     * @param styleId           content view style
     * @param widgetType        the widget type
     * @param rewarded          select if the request is for rewarded
     * @param resultCallback    result callback
     */
    public void loadWaterfall(Context context, String styleId, String widgetType, String rewarded, final ApiResultCallback<ContentData> resultCallback)
    {
        ContentValues contentValues = new ContentValues();
        contentValues.put(METHOD_NAME, WATERFALL);

        setGeneralParams(context, contentValues);

        contentValues.put("style_id", styleId);
        contentValues.put("widget_type", widgetType);
        contentValues.put("is_rewarded",rewarded);

        long ts = System.currentTimeMillis() / 1000;
        contentValues.put("ts", ts);
        contentValues.put("hash", generateRequestHash(contentValues, ts, mSecurityToken));

        startServerConnection(context, HTTP_SCHEME + WATERFALL_DOMAIN + WATERFALL_URL, CONNECTION_TYPE.GET, SdkRequestType.LOAD_WATERFALL, contentValues, 1, resultCallback, mDeveloperLogging, false);
    }

    public void initSdk(Context context, ApiResultCallback<PropertiesObj> resultCallback)
    {
        ContentValues contentValues = new ContentValues();
        contentValues.put(METHOD_NAME, INIT_SDK_CONFIGURATION);

        setGeneralParams(context, contentValues);

        try
        {
            startServerConnection(context, HTTP_SCHEME + SDK_API_URL + INIT_SDK_CONFIGURATION, CONNECTION_TYPE.GET, SdkRequestType.VALIDATE_SDK, contentValues, 1, resultCallback, mDeveloperLogging, false);
        } catch (Exception ex)
        {
            SDKLogger.printErrorLog(TAG, "Error when trying to validateSDK: " + ex.getMessage());
            resultCallback.onFailed();
        }
    }

    public void setGeneralParams(Context context, ContentValues contentValues)
    {
        contentValues.put("publisher_id", mPublicherId);
        contentValues.put("package_id", mPackageName);
        contentValues.put("device_lang", Locale.getDefault().getLanguage());
        contentValues.put("device_hash", getDeviceHash());
        String addId = SdkDeviceUtils.getGoogleAdvertisingID(context);
        if (addId == null)
        {
            addId = "";
        }
        contentValues.put("google_id", addId);
        contentValues.put("device_type", ScreenUtils.getIsTablet(context) == true ? "2" : "1");
        contentValues.put("version", ConstantDef.SDK_VERSION);
        contentValues.put("sdk_version", ConstantDef.SDK_VERSION);
        contentValues.put("actual_sdk_version", ConstantDef.ACTUAL_SDK_VERSION);
        contentValues.put("os_type", "android");
        contentValues.put("os_version", String.valueOf(Build.VERSION.SDK_INT));
        contentValues.put("extension_type", ConstantDef.SDK_EXTENSION_TYPE);
        contentValues.put("access_token", mSecurityToken);


        String webviewVersion = "w";
        String useragent = Utils.getDefaultUserAgent();
        if (useragent != null && useragent.contains("Chrome"))
        {
            int index = useragent.indexOf("Chrome");
            String sub = useragent.substring(index, useragent.length());
            String[] res = sub.split(" ");
            if (res != null && res.length > 0)
            {
                String[] tempVal = res[0].split("/");
                if (tempVal != null && tempVal.length > 1)
                {
                    webviewVersion = "c_" + tempVal[1];
                }
            }
        }

        contentValues.put("webview_version", webviewVersion);

        try
        {
            Point point = ScreenUtils.getScreenSize(context);
            contentValues.put("resolution_height", point.y);
            contentValues.put("resolution_width", point.x);
            contentValues.put("screen_size", ScreenUtils.getDeviceScreenSizeInInches(context));
            contentValues.put("dpi", context.getResources().getDisplayMetrics().density);
            contentValues.put("screen_category", context.getResources().getConfiguration().screenLayout & Configuration.SCREENLAYOUT_SIZE_MASK);
        } catch (Exception e)
        {

        }

        String appVersionCode = "";
        String appVersionName = "";
        try
        {
            PackageManager packageManager = context.getPackageManager();
            PackageInfo packageInfo = packageManager.getPackageInfo(context.getPackageName(), 0);
            appVersionCode = String.valueOf(packageInfo.versionCode);
            appVersionName = packageInfo.versionName;
        } catch (Exception ex)
        {
            SDKLogger.printErrorLog(TAG, "Error when trying to get app version name and code: " + ex.getMessage());
        }

        contentValues.put("app_version_code", appVersionCode);
        contentValues.put("app_version_name", appVersionName);

        try
        {
            contentValues.put("manufacturer", Build.MANUFACTURER);
            contentValues.put("model", Build.MODEL);
        } catch (Exception e)
        {

        }

        boolean isWiFi = true;
        try
        {

            ConnectivityManager cm =
                    (ConnectivityManager)context.getSystemService(Context.CONNECTIVITY_SERVICE);

            NetworkInfo activeNetwork = cm.getActiveNetworkInfo();

            isWiFi = activeNetwork.getType() == ConnectivityManager.TYPE_WIFI;



        } catch (Exception e)
        {
            Log.e("KIDOZ"," "+ e.getMessage());
        }
        if(isWiFi)
        {
            contentValues.put("wifi_mode", "1");
        }
        else
        {
            contentValues.put("wifi_mode", "0");
        }

        String carrierName = "unknown";

        try
        {
            TelephonyManager manager = (TelephonyManager)context.getSystemService(Context.TELEPHONY_SERVICE);
            carrierName = manager.getNetworkOperatorName();

        } catch (Exception e)
        {
            Log.e("KIDOZ","" + e.getMessage());
        }
        contentValues.put("carrier_name", carrierName);


        String networkString = "Unknown";
        try
        {
            TelephonyManager manager = (TelephonyManager)context.getSystemService(Context.TELEPHONY_SERVICE);
            int networkType = manager.getNetworkType();

            switch (networkType) {
                case TelephonyManager.NETWORK_TYPE_GPRS:
                case TelephonyManager.NETWORK_TYPE_EDGE:
                case TelephonyManager.NETWORK_TYPE_CDMA:
                case TelephonyManager.NETWORK_TYPE_1xRTT:
                case TelephonyManager.NETWORK_TYPE_IDEN:
                    networkString =  "2G";
                case TelephonyManager.NETWORK_TYPE_UMTS:
                case TelephonyManager.NETWORK_TYPE_EVDO_0:
                case TelephonyManager.NETWORK_TYPE_EVDO_A:
                case TelephonyManager.NETWORK_TYPE_HSDPA:
                case TelephonyManager.NETWORK_TYPE_HSUPA:
                case TelephonyManager.NETWORK_TYPE_HSPA:
                case TelephonyManager.NETWORK_TYPE_EVDO_B:
                case TelephonyManager.NETWORK_TYPE_EHRPD:
                case TelephonyManager.NETWORK_TYPE_HSPAP:
                    networkString =  "3G";
                case TelephonyManager.NETWORK_TYPE_LTE:
                    networkString = "4G";
                default:
                    networkString = "Unknown";
            }

        } catch (Exception e)
        {
            Log.e("KIDOZ"," " + e.getMessage());
        }
        contentValues.put("network_type", networkString);

    }

    /**
     * Send kidoz events to server
     *
     * @param eventsJsonBulk
     */
    public ResultData<Boolean> sendKidozEventsToServerNotAsync(String eventsJsonBulk)
    {
        ResultData<Boolean> resultCallback = null;

        final ContentValues contentValues = new ContentValues();
        contentValues.put("methodName", SEND_KIDOZ_EVENTS_TO_SERVER);

        try
        {

            contentValues.put("KidozEventsLogAsJson", URLEncoder.encode(eventsJsonBulk, "UTF-8"));
            contentValues.put(REQUEST_FORMAT, JSON);

            String responce = makePostConnection(HTTP_SCHEME + KIDOZ_ANALITYCS_SERVER_URL, contentValues);
            if (responce != null)
            {
                resultCallback = EventBulk.parseResultSuccsessStatus(responce);
            }
        } catch (Exception e)
        {
            SDKLogger.printErrorLog(" \n IO Exception On Event send request! \n" + e.getMessage());
        }

        return resultCallback;
    }

    /**
     * Generate security hash for a server request
     */
    private String generateRequestHash(ContentValues contentValues, long ts, String token)
    {
        Set<String> set = contentValues.keySet();
        ArrayList<String> list = new ArrayList<>();
        list.addAll(set);
        Collections.sort(list);

        StringBuilder stringBuilder = new StringBuilder();
        for (int i = 0; i < list.size(); i++)
        {
            stringBuilder.append(contentValues.getAsString(list.get(i)));
        }
        stringBuilder.append(ts);
        stringBuilder.append(token);

        return EncoderUtils.SHA1(stringBuilder.toString());
    }

    private String getDeviceHash()
    {
        if (mCalculatedDeviceHash == null)
        {
            mCalculatedDeviceHash = Utils.generateUniqeDeviceID(mPackageName, mPublicherId);
        }
        return mCalculatedDeviceHash;
    }

    public void getCountryCode(Context context, ApiResultCallback<String> apiResultCallback)
    {
        ContentValues contentValues = new ContentValues();
        contentValues.put("key", "Xg4VFb!7P$_p87Eq+7-t&X=VPHw3P&yL");
        startServerConnection(context, "http://accounts.kidoz.net/api/cc", CONNECTION_TYPE.POST, SdkRequestType.GET_COUNTRY_CODE, contentValues, 0, apiResultCallback, mDeveloperLogging, false);
    }

    public void callGetURL(Context context, String url, ApiResultCallback<String> apiResultCallback)
    {
        startServerConnection(context,url, CONNECTION_TYPE.GET, SdkRequestType.SEND_EVENT, null, 0, apiResultCallback, mDeveloperLogging, false);

    }
}

