package com.kidoz.sdk.api.players.web_player;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.graphics.Bitmap;
import android.graphics.Color;
import android.net.Uri;
import android.os.Build;
import android.os.Looper;
import android.util.AttributeSet;
import android.view.View;
import android.webkit.WebChromeClient;
import android.webkit.WebSettings;
import android.webkit.WebSettings.LayoutAlgorithm;
import android.webkit.WebSettings.PluginState;
import android.webkit.WebView;
import android.webkit.WebViewClient;

import com.kidoz.events.Event;
import com.kidoz.events.EventManager;
import com.kidoz.events.EventParameters;
import com.kidoz.sdk.api.general.ContentExecutionHandler;
import com.kidoz.sdk.api.general.enums.ContentType;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.general.utils.ConstantDef;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.structure.ContentItem;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;

public class KidozWebView extends WebView
{
    protected final static String HTTP = "http://";
    protected final static String HTTPS = "https://";
    protected final static String WWW = "www.";
    protected final static String WWW2 = "www2.";
    protected final static String KIDOZ_DOMAIN = "kidoz";
    protected final static List<String> PREFIXES_TO_REMOVE = new LinkedList<String>();
    private Utils.StaticHandler staticHandler;
    private ContentItem mContentItem;

    private String mLastOverloadUrl = "";

    static
    {
        PREFIXES_TO_REMOVE.add(HTTP);
        PREFIXES_TO_REMOVE.add(HTTPS);
        PREFIXES_TO_REMOVE.add(WWW);
        PREFIXES_TO_REMOVE.add(WWW2);
    }

    private final String TAG = KidozWebView.class.getSimpleName();
    private static int API = Build.VERSION.SDK_INT;
    private Context mContext;
    private ArrayList<String> mAllowedSitesKeyWords = new ArrayList<String>();

    private IOnLounchExternalAppFromRedirectListener onLounchExternalAppFromRedirectListener;

    private WebPlayerType mWebPlayerType = WebPlayerType.WEB_BROWSER;
    private String mGooglePlayAppId = null;

    public KidozWebView(Context context, AttributeSet attrs)
    {
        super(context, attrs);
        mContext = context;

        initWebView();

        initForTarget(mWebPlayerType);
    }

    public KidozWebView(Context context)
    {
        super(context);
        mContext = context;

        initWebView();

        initForTarget(mWebPlayerType);
    }

    public KidozWebView(Context context, WebPlayerType webPlayerType)
    {
        super(context);
        mContext = context;

        mWebPlayerType = webPlayerType;
        initWebView();

        initForTarget(webPlayerType);
    }

    public void setContentItem(ContentItem contentItem)
    {
        mContentItem = contentItem;
    }

    public void initForTarget(WebPlayerType webPlayerType)
    {
        if (webPlayerType != null)
        {
            switch (webPlayerType)
            {
                case ONLINE_GAME:
                {
                    //                    initOnlineGameSettings(getContext());
                    initAWorkingSettingsForVideoInWebView();
                    break;
                }

                case WEB_BROWSER:
                {
                    //                    initializeGeneralSettingsState(getContext());
                    initAWorkingSettingsForVideoInWebView();
                    break;
                }

                case EXTERNAL_LINK:
                {
                    initializeGeneralSettingsState(getContext());
                    break;
                }
                case ROVIO:
                {
                    //                    initializeGeneralSettingsState(getContext());
                    initAWorkingSettingsForVideoInWebView();
                    break;
                }
            }
        }
    }

    @SuppressWarnings("deprecation")
    @SuppressLint({"SetJavaScriptEnabled", "NewApi"})
    public void initializeGeneralSettingsState(Context context)
    {
        if (API < 18)
        {
            getSettings().setAppCacheMaxSize(Long.MAX_VALUE);
        }
        if (API < 17)
        {
            getSettings().setEnableSmoothTransition(true);
        }
        if (API >= 17)
        {
            getSettings().setMediaPlaybackRequiresUserGesture(false);
        }
        if (API >= 19)
        {
            setWebContentsDebuggingEnabled(ConstantDef.DGM);
        }
        if (API < 19)
        {
            getSettings().setDatabasePath(context.getCacheDir() + "/databases");
        }
        getSettings().setDomStorageEnabled(true);
        getSettings().setAppCacheEnabled(true);
        getSettings().setAppCachePath(context.getCacheDir().toString());
        getSettings().setCacheMode(WebSettings.LOAD_DEFAULT);
        getSettings().setGeolocationDatabasePath(context.getFilesDir().toString());
        getSettings().setAllowFileAccess(true);
        getSettings().setDatabaseEnabled(true);
        getSettings().setSupportZoom(true);
        getSettings().setBuiltInZoomControls(true);
        getSettings().setDisplayZoomControls(false);
        getSettings().setAllowContentAccess(true);
        getSettings().setDefaultTextEncodingName("utf-8");
        if (API >= 16)
        {
            getSettings().setAllowFileAccessFromFileURLs(false);
            getSettings().setAllowUniversalAccessFromFileURLs(false);
        }

        SharedPreferences mPreferences = context.getSharedPreferences(WebPreferenceConstants.PREFERENCES, 0);

        getSettings().setGeolocationEnabled(mPreferences.getBoolean(WebPreferenceConstants.LOCATION, false));
        if (API < 19)
        {

            switch (mPreferences.getInt(WebPreferenceConstants.ADOBE_FLASH_SUPPORT, 0))
            {
                case 0:
                    getSettings().setPluginState(PluginState.OFF);
                    break;
                case 1:
                    getSettings().setPluginState(PluginState.ON_DEMAND);
                    break;
                case 2:
                    getSettings().setPluginState(PluginState.ON);
                    break;
                default:
                    break;
            }
        }

        if (mPreferences.getBoolean(WebPreferenceConstants.SAVE_PASSWORDS, false))
        {
            if (API < 18)
            {
                getSettings().setSavePassword(true);
            }
            getSettings().setSaveFormData(true);
        }

        if (mPreferences.getBoolean(WebPreferenceConstants.JAVASCRIPT, true))
        {
            getSettings().setJavaScriptEnabled(true);
            getSettings().setJavaScriptCanOpenWindowsAutomatically(true);
        }

        if (mPreferences.getBoolean(WebPreferenceConstants.TEXT_REFLOW, false))
        {
            getSettings().setLayoutAlgorithm(LayoutAlgorithm.NARROW_COLUMNS);
        }
        else if (API >= Build.VERSION_CODES.KITKAT)
        {
            try
            {
                getSettings().setLayoutAlgorithm(WebSettings.LayoutAlgorithm.TEXT_AUTOSIZING);
            }
            catch (Exception ex)
            {
                getSettings().setLayoutAlgorithm(WebSettings.LayoutAlgorithm.NORMAL);
                SDKLogger.printErrorLog(TAG, "Error when trying to set layout algorithm: " + ex.getMessage());
            }
        }
        else
        {
            getSettings().setLayoutAlgorithm(LayoutAlgorithm.NORMAL);
        }

        getSettings().setBlockNetworkImage(mPreferences.getBoolean(WebPreferenceConstants.BLOCK_IMAGES, false));
        getSettings().setSupportMultipleWindows(mPreferences.getBoolean(WebPreferenceConstants.POPUPS, true));
        getSettings().setUseWideViewPort(mPreferences.getBoolean(WebPreferenceConstants.USE_WIDE_VIEWPORT, true));
        getSettings().setLoadWithOverviewMode(mPreferences.getBoolean(WebPreferenceConstants.OVERVIEW_MODE, true));
        switch (mPreferences.getInt(WebPreferenceConstants.TEXT_SIZE, 3))
        {
            case 1:
                getSettings().setTextZoom(200);
                break;
            case 2:
                getSettings().setTextZoom(150);
                break;
            case 3:
                getSettings().setTextZoom(100);
                break;
            case 4:
                getSettings().setTextZoom(75);
                break;
            case 5:
                getSettings().setTextZoom(50);
                break;
        }

        if (Build.VERSION.SDK_INT > Build.VERSION_CODES.LOLLIPOP_MR1)
        {
            setLayerType(View.LAYER_TYPE_HARDWARE, null);
        }
        else
        {
            setLayerType(View.LAYER_TYPE_SOFTWARE, null);
        }

        if (Build.VERSION.SDK_INT >= 21)
        {
            getSettings().setMixedContentMode(WebSettings.MIXED_CONTENT_ALWAYS_ALLOW);
        }
    }

    @TargetApi(Build.VERSION_CODES.KITKAT)
    public synchronized void initOnlineGameSettings(Context context)
    {

        if (API < 18)
        {
            getSettings().setAppCacheMaxSize(Long.MAX_VALUE);
        }
        if (API < 17)
        {
            getSettings().setEnableSmoothTransition(true);
        }
        if (API >= 17)
        {
            getSettings().setMediaPlaybackRequiresUserGesture(true);
        }
        if (API >= 19)
        {
            setWebContentsDebuggingEnabled(ConstantDef.DGM);
        }
        if (API < 19)
        {
            getSettings().setDatabasePath(context.getCacheDir() + "/databases");
        }

        getSettings().setJavaScriptEnabled(true);
        getSettings().setLoadWithOverviewMode(true);
        getSettings().setAllowContentAccess(true);
        getSettings().setUserAgentString(ConstantDef.USER_AGENT);
        getSettings().setUseWideViewPort(false);
        //ooOri todo: complete this function no PreferenceConstants
        if (Build.VERSION.SDK_INT < 18)
        {
            getSettings().setPluginState(PluginState.ON);
        }

        getSettings().setDomStorageEnabled(true);
        getSettings().setAllowFileAccess(true);

        clearHistory();
        clearFormData();

        if (API >= Build.VERSION_CODES.KITKAT)
        {
            try
            {
                getSettings().setLayoutAlgorithm(WebSettings.LayoutAlgorithm.TEXT_AUTOSIZING);
            }
            catch (Exception ex)
            {
                getSettings().setLayoutAlgorithm(WebSettings.LayoutAlgorithm.NORMAL);
                SDKLogger.printErrorLog(TAG, "Error when trying to set layout algorithm: " + ex.getMessage());
            }
        }
        else
        {
            getSettings().setLayoutAlgorithm(LayoutAlgorithm.NORMAL);
        }

        if (Build.VERSION.SDK_INT > Build.VERSION_CODES.LOLLIPOP_MR1)
        {
            setLayerType(View.LAYER_TYPE_HARDWARE, null);
        }
        else
        {
            setLayerType(View.LAYER_TYPE_SOFTWARE, null);
        }
    }

    @SuppressLint("NewApi")
    private void initWebView()
    {
        staticHandler = new Utils.StaticHandler(Looper.getMainLooper());
        staticHandler.removeCallbacksAndMessages(null);

        setDrawingCacheBackgroundColor(0x00000000);
        setFocusableInTouchMode(true);
        setFocusable(true);

        setAnimationCacheEnabled(true);
        setDrawingCacheEnabled(true);

        //setBackgroundColor(mContext.getResources().getColor(android.R.color.white));

        if (API > 15)
        {
            setBackground(null);
            getRootView().setBackground(null);
        }
        else
        {
            getRootView().setBackgroundDrawable(null);
        }

        setWillNotCacheDrawing(false);
        setAlwaysDrawnWithCacheEnabled(true);
        setScrollbarFadingEnabled(true);
        setSaveEnabled(true);

        setWebChromeClient(new WebChromeClient()
        {

        });

        initWebChromeClient();

        if (Build.VERSION.SDK_INT >= 21)
        {
            getSettings().setMixedContentMode(WebSettings.MIXED_CONTENT_ALWAYS_ALLOW);
        }
    }

    // Initiate web chrome client
    private void initWebChromeClient()
    {
        setWebViewClient(new WebViewClient()
        {
            @Override
            public void onPageStarted(WebView view, String url, Bitmap favicon)
            {
                super.onPageStarted(view, url, favicon);
            }

            @Override
            public void onReceivedError(WebView view, int errorCode, String description, String failingUrl)
            {
                Event event = new Event();
                if (mContentItem != null)
                {
                    event.addParameterToJsonObject(EventParameters.ITEM_ID, mContentItem.getId());
                    event.addParameterToJsonObject(EventParameters.ADVERTISER_ID, mContentItem.getAdvertiserID());
                }

                EventManager.getInstance(getContext()).logEvent(getContext(), "0", "", EventManager.LOG_CRITICAL_LEVEL, event, EventParameters.CATEGORY_WEB_VIEW_ERROR, String.valueOf(errorCode).concat(": ").concat(description), failingUrl);
            }

            @Override
            public void onPageFinished(WebView view, String url)
            {
                //ooOri todo: complete this function
                super.onPageFinished(view, url);
            }

            @Override
            public boolean shouldOverrideUrlLoading(WebView view, String url)
            {
                mLastOverloadUrl = url;
                SDKLogger.printDebbugLog(TAG, "shouldOverrideUrlLoading: url= " + url);
                boolean result = true;
                if (mWebPlayerType == WebPlayerType.ROVIO)
                {
                    return super.shouldOverrideUrlLoading(view, url);
                }
                else if (mWebPlayerType == WebPlayerType.ONLINE_GAME || mWebPlayerType == WebPlayerType.WEB_BROWSER)
                {
                    result = !checkIfSafeUrl(url);
                    if (result == true)
                    {
                        ContentItem contentItem = new ContentItem();
                        contentItem.setContentType(ContentType.EXTERNAL_BROWSER_URL);
                        contentItem.setIsPromoted(true);
                        contentItem.setData(url);
                        ContentExecutionHandler.handleContentItemClick(getContext(), contentItem, WidgetType.WIDGET_TYPE_NONE.getStringValue(), "0", 0, false, null);
                    }
                }
                else if (mWebPlayerType == WebPlayerType.EXTERNAL_LINK)
                {
                    if (url.contains("market://") || url.contains("play.google.com") && mGooglePlayAppId != null)
                    {
                        try
                        {
                            String googlePlayMarketURL = "market://details?id=" + mGooglePlayAppId;
                            String[] urlPazzel = url.split(mGooglePlayAppId);
                            if (urlPazzel != null && urlPazzel.length > 1)
                            {
                                googlePlayMarketURL += urlPazzel[1];
                            }

                            Intent intent = new Intent(Intent.ACTION_VIEW);
                            intent.setData(Uri.parse(googlePlayMarketURL));
                            // Used for notifing the the webview holder that the webiew is lounching and external application die to redirect link
                            // there fore it need to be closed
                            if (onLounchExternalAppFromRedirectListener != null)
                            {
                                onLounchExternalAppFromRedirectListener.onLounch();
                            }
                            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                            mContext.startActivity(intent);
                        }
                        catch (Exception ex)
                        {
                            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(TAG, "Error when trying to open google play for promoted app: \n" + ex.getMessage());
                        }
                        result = true;
                    }
                    else
                    {
                        view.loadUrl(url);
                        result = true;
                    }
                }
                else
                {
                    view.loadUrl(url);
                    result = true;
                }
                return result;
            }
        });
    }

    public String getLastOverloadUrl() {
        return mLastOverloadUrl;
    }

    private void initAWorkingSettingsForVideoInWebView()
    {
        getSettings().setJavaScriptEnabled(true);

        if (Build.VERSION.SDK_INT >= 21)
        {
            getSettings().setMixedContentMode(WebSettings.MIXED_CONTENT_ALWAYS_ALLOW);
        }
        setDrawingCacheBackgroundColor(0x00000000);
        setBackgroundColor(Color.argb(1, 0, 0, 0));
        setFocusableInTouchMode(true);
        setFocusable(true);
        setAnimationCacheEnabled(true);
        setDrawingCacheEnabled(true);
        if (Build.VERSION.SDK_INT < 19)
        {
            getSettings().setDatabasePath(getContext().getCacheDir() + "/databases");
        }
        getSettings().setAppCacheEnabled(true);
        getSettings().setDatabaseEnabled(true);

        if (Build.VERSION.SDK_INT < 18)
        {
            getSettings().setAppCacheMaxSize(Long.MAX_VALUE);
        }
        if (Build.VERSION.SDK_INT < 17)
        {
            getSettings().setEnableSmoothTransition(true);
        }

        if (Build.VERSION.SDK_INT < 18)
        {
            getSettings().setRenderPriority(WebSettings.RenderPriority.HIGH);
        }
        getSettings().setBuiltInZoomControls(false);
        getSettings().setDisplayZoomControls(false);
        getSettings().setGeolocationDatabasePath(getContext().getFilesDir().toString());
        getSettings().setAppCachePath(getContext().getCacheDir().toString());
        getSettings().setAllowContentAccess(false);
        getSettings().setDefaultTextEncodingName("utf-8");
        getSettings().setUserAgentString(ConstantDef.USER_AGENT);
        getSettings().setGeolocationEnabled(true);
        getSettings().setSupportMultipleWindows(false);
        getSettings().setJavaScriptEnabled(true);
        getSettings().setJavaScriptCanOpenWindowsAutomatically(true);
        getSettings().setBlockNetworkImage(false);
        getSettings().setPluginState(WebSettings.PluginState.ON);
        if (Build.VERSION.SDK_INT < 18)
        {
            getSettings().setSavePassword(false);
        }
        if (Build.VERSION.SDK_INT >= 17)
        {
            getSettings().setMediaPlaybackRequiresUserGesture(false);
        }
        getSettings().setDomStorageEnabled(true);
        getSettings().setAllowFileAccess(true);
        getSettings().setSupportZoom(false);
        if (Build.VERSION.SDK_INT >= 16)
        {
            getSettings().setAllowFileAccessFromFileURLs(true);
            getSettings().setAllowUniversalAccessFromFileURLs(true);
        }
        getSettings().setSaveFormData(false);
        getSettings().setLoadWithOverviewMode(false);
        getSettings().setUseWideViewPort(false);
        getSettings().setCacheMode(WebSettings.LOAD_NO_CACHE);
        if (Build.VERSION.SDK_INT >= 16)
        {
            setLayerType(View.LAYER_TYPE_HARDWARE, null);
        }
        if (Build.VERSION.SDK_INT >= 21)
        {
            getSettings().setMixedContentMode(WebSettings.MIXED_CONTENT_ALWAYS_ALLOW);
        }
        getSettings().setLayoutAlgorithm(WebSettings.LayoutAlgorithm.NORMAL);
        if (Build.VERSION.SDK_INT >= 19)
        {
            setWebContentsDebuggingEnabled(ConstantDef.DGM);
        }
        if (Build.VERSION.SDK_INT >= 17)
        {
            getSettings().setMediaPlaybackRequiresUserGesture(false);
        }
    }

    public void loadWebUrl(String url, String googlePlayAppID)
    {
        mAllowedSitesKeyWords.clear();
        mAllowedSitesKeyWords.add(getDomain(url));
        mAllowedSitesKeyWords.add(url);
        mGooglePlayAppId = googlePlayAppID;
        loadUrl(url);
    }

    public void loadWebUrl(String[] allowedDomains, String url, String googlePlayAppID)
    {
        mLastOverloadUrl = "";
        mAllowedSitesKeyWords.clear();
        if (allowedDomains != null)
        {
            mAllowedSitesKeyWords.addAll(Arrays.asList(allowedDomains));
        }
        mAllowedSitesKeyWords.add(getDomain(url));
        mAllowedSitesKeyWords.add(url);
        mGooglePlayAppId = googlePlayAppID;
        loadUrl(url);
    }

    @Override
    public void onResume()
    {
        super.onResume();
    }

    @Override
    public void onPause()
    {

        stopLoading();
        super.onPause();
    }

    public synchronized void onDestroy()
    {
        stopLoading();
        clearHistory();
        removeAllViews();
        destroyDrawingCache();
    }

    private boolean checkIfSafeUrl(String url)
    {
        boolean result = false;
        if (url != null)
        {
            if (url.contains(KIDOZ_DOMAIN) == true)
            {
                result = true;
            }
            else
            {
                String urlStringWithoutPrefix = removePrefix(url);
                String fixedUrlString = urlStringWithoutPrefix != null ? urlStringWithoutPrefix.toLowerCase(Locale.US) : null;
                com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, "checkIfSafeUrl: fixedUrlString = " + fixedUrlString);
                if (mAllowedSitesKeyWords != null)
                {
                    for (int i = 0; i < mAllowedSitesKeyWords.size(); i++)
                    {
                        String originalUrl = removePrefix(mAllowedSitesKeyWords.get(i));
                        com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, "checkIfSafeUrl: originalUrl = " + originalUrl);
                        if (fixedUrlString.contains(originalUrl != null ? originalUrl.toLowerCase(Locale.US) : null))
                        {
                            com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, "********************************");
                            com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, "checkIfSafeUrl: found a match:fixedUrlString = " + fixedUrlString);
                            com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, "checkIfSafeUrl: found a match:originalUrl = " + originalUrl);
                            com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, "********************************");
                            result = true;
                            break;
                        }
                    }
                }
                else
                {
                    com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, "Web Player  Not allowed link!");
                }
            }
        }
        return result;
    }

    private String getDomain(String url)
    {
        SDKLogger.printDebbugLog(TAG, "getDomain: URL = " + url);

        String result = null;
        if (url != null)
        {
            String urlStringWithoutPrefix = removePrefix(url);
            SDKLogger.printDebbugLog(TAG, "getDomain: urlStringWithoutPrefix = " + urlStringWithoutPrefix);
            String[] domainPartsArray = urlStringWithoutPrefix.split("/");
            if (domainPartsArray != null)
            {
                result = domainPartsArray[0];
                SDKLogger.printDebbugLog(TAG, "getDomain: result = " + result);
            }
            else
            {
                SDKLogger.printDebbugLog(TAG, "getDomain: domainPartsArray = null");
            }
        }
        return result;
    }

    /**
     * @param urlString - The WEBSITE_URL string we want modified
     * @return A modified WEBSITE_URL without all the prefixes:
     * will all return globes.co.il
     */
    private static String removePrefix(String urlString)
    {
        try
        {
            String modifiedString = urlString.toLowerCase();
            for (String prefixToRemove : PREFIXES_TO_REMOVE)
            {
                if (modifiedString.startsWith(prefixToRemove))
                {
                    int lenght = modifiedString.length();
                    modifiedString = modifiedString.substring(prefixToRemove.length(), lenght);
                }
            }
            return modifiedString;
        }
        catch (Exception e)
        {
            return null;
        }
    }

    public synchronized void stopAndReleaseWebView()
    {
        loadUrl("");
        stopLoading();
        clearHistory();
        removeAllViews();
        destroyDrawingCache();
    }

    /**
     * Listener for notifing the the webview holder that the webiew is lounching and external application
     */
    public interface IOnLounchExternalAppFromRedirectListener
    {
        void onLounch();
    }

    /**
     * Set on lounch external application listener*
     *
     * @param listener click listener
     */
    public void setOnLounchExternalAppFromRedirect(IOnLounchExternalAppFromRedirectListener listener)
    {
        onLounchExternalAppFromRedirectListener = listener;
    }
}