/*
 * Copyright (C) 2013 Square, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.squareup.imagelib;

import android.content.Context;
import android.net.Uri;
import android.os.Environment;

import java.io.File;
import java.io.IOException;
import java.util.concurrent.TimeUnit;

import okhttp3.Cache;
import okhttp3.CacheControl;
import okhttp3.Call;
import okhttp3.OkHttpClient;
import okhttp3.ResponseBody;

/**
 * A {@link Downloader} which uses OkHttp to download images.
 */
public class OkHttpDownloader implements Downloader
{
    private static OkHttpClient defaultOkHttpClient(final File cacheDir, final long maxSize)
    {
        OkHttpClient result;
        okhttp3.Cache cache = null;
        if (cacheDir != null && maxSize != -1)
        {
            File file = Environment.getDownloadCacheDirectory();
            if (file != null)
            {
                file = new File(file.getAbsolutePath() + File.separator + "SDKhttpCacheDir");
                if (file != null)
                {
                    if (file.exists() == false)
                    {
                        file.mkdirs();
                    }

                    int cacheSize = 10 * 1024 * 1024; // 10 MiB
                    cache = new Cache(file, cacheSize);
                }
            }
        }

        if (cache != null)
        {
            result = new OkHttpClient.Builder().connectTimeout(10, TimeUnit.SECONDS).writeTimeout(10, TimeUnit.SECONDS).readTimeout(20, TimeUnit.SECONDS).cache(cache).build();
        }
        else
        {
            result = new OkHttpClient.Builder().connectTimeout(10, TimeUnit.SECONDS).writeTimeout(10, TimeUnit.SECONDS).readTimeout(20, TimeUnit.SECONDS).build();
        }
        return result;
    }

    private OkHttpClient client;

    /**
     * Create new downloader that uses OkHttp. This will install an image cache into your application
     * cache directory.
     */
    public OkHttpDownloader(final Context context)
    {
        this(Utils.createDefaultCacheDir(context));
    }

    /**
     * Create new downloader that uses OkHttp. This will install an image cache into the specified
     * directory.
     *
     * @param cacheDir The directory in which the cache should be stored
     */
    public OkHttpDownloader(final File cacheDir)
    {
        this(cacheDir, Utils.calculateDiskCacheSize(cacheDir));
    }

    /**
     * Create new downloader that uses OkHttp. This will install an image cache into your application
     * cache directory.
     *
     * @param maxSize The size limit for the cache.
     */
    public OkHttpDownloader(final Context context, final long maxSize)
    {
        this(Utils.createDefaultCacheDir(context), maxSize);
    }

    /**
     * Create new downloader that uses OkHttp. This will install an image cache into the specified
     * directory.
     *
     * @param cacheDir The directory in which the cache should be stored
     * @param maxSize  The size limit for the cache.
     */
    public OkHttpDownloader(final File cacheDir, final long maxSize)
    {
        this(defaultOkHttpClient(cacheDir, maxSize));
    }

    /**
     * Create a new downloader that uses the specified OkHttp instance. A response cache will not be
     * automatically configured.
     */
    public OkHttpDownloader(OkHttpClient client)
    {
        this.client = client;
    }

    protected final OkHttpClient getClient()
    {
        return client;
    }

    @Override
    public Response load(Uri uri, int networkPolicy) throws IOException
    {
        Response result = null;
        CacheControl cacheControl = null;
        if (networkPolicy != 0)
        {
            if (NetworkPolicy.isOfflineOnly(networkPolicy))
            {
                cacheControl = CacheControl.FORCE_CACHE;
            }
            else
            {
                CacheControl.Builder builder = new CacheControl.Builder();
                if (!NetworkPolicy.shouldReadFromDiskCache(networkPolicy))
                {
                    builder.noCache();
                }
                if (!NetworkPolicy.shouldWriteToDiskCache(networkPolicy))
                {
                    builder.noStore();
                }
                cacheControl = builder.build();
            }
        }

        okhttp3.Request request;
        if (cacheControl != null)
        {
            request = new okhttp3.Request.Builder().cacheControl(cacheControl).url(uri.toString()).build();
        }
        else
        {
            request = new okhttp3.Request.Builder().url(uri.toString()).build();
        }

        Call call = client.newCall(request);
        boolean fromCache = false;
        if (call != null)
        {
            okhttp3.Response response = call.execute();
            int responseCode = response.code();
            if (responseCode >= 300)
            {
                response.body().close();
                throw new ResponseException(responseCode + " " + response.message(), networkPolicy, responseCode);
            }
            fromCache = response.cacheResponse() != null;
            ResponseBody responseBody = response.body();
            result = new Response(responseBody.byteStream(), fromCache, responseBody.contentLength());
        }
        return result;
    }

    @Override
    public void shutdown()
    {
        Cache cache = client.cache();
        if (cache != null)
        {
            try
            {
                cache.close();
            }
            catch (IOException ignored)
            {
            }
        }
    }
}
