package com.kidoz.sdk.api.ui_views.html_view;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.graphics.Point;
import android.net.Uri;
import android.os.Build;
import android.os.Looper;
import android.os.Message;
import android.webkit.WebChromeClient;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.RelativeLayout;

import com.kidoz.events.Event;
import com.kidoz.events.EventManager;
import com.kidoz.events.EventParameters;
import com.kidoz.sdk.api.KidozSDK;
import com.kidoz.sdk.api.dialogs.AboutKidozDialog;
import com.kidoz.sdk.api.dialogs.ParentalLockDialog;
import com.kidoz.sdk.api.general.ContentExecutionHandler;
import com.kidoz.sdk.api.general.EventMessage;
import com.kidoz.sdk.api.general.WidgetEventMessage;
import com.kidoz.sdk.api.general.enums.ContentType;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.general.utils.ConstantDef;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.ScreenUtils;
import com.kidoz.sdk.api.general.utils.SdkCookieManager;
import com.kidoz.sdk.api.general.utils.SdkDeviceUtils;
import com.kidoz.sdk.api.general.utils.StorageLife;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.structure.ContentItem;
import com.kidoz.sdk.api.ui_views.loading_progress_view.LoadingProgressView;
import com.kidoz.sdk.api.ui_views.one_item_view.ItemViewPagerAdapter;

import org.greenrobot.eventbus.EventBus;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.lang.ref.SoftReference;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Locale;

public class HtmlViewWrapper extends RelativeLayout implements HtmlJavaScriptInterafce
{
    protected static final float LOADING_PROGRESS_DEFAULT_RATIO = 0.35f;

    protected static final int ON_LOAD_FINISHED = 0;
    protected static final int ON_LOAD_STARTED = 1;
    protected static final int ON_VIEW_READY = 2;
    protected static final int ON_CLOSE = 3;
    protected static final int ON_RECEIVED_ERROR = 4;
    protected static final int ON_NOTIFY_VAST_AD_READY = 5;
    protected static final int ON_REWARDED = 6;
    protected static final int ON_REWARDED_STARTED = 7;
    protected static final int ON_AD_STATE_CHANGE = 8;

    public enum AdState
    {
        AD_PLAYING, AD_STOPED
    }

    protected static final String TAG = HtmlViewWrapper.class.getSimpleName();

    public HtmlFiveWebView mWebView;

    protected String mStyleId;
    protected String mWidgetType = "";

    protected boolean mAllowClickHandling = true;

    protected ContentItem mContentItem;
    protected String mHtmlPageUrl;

    // Used for tacking the last reloaded/redirect url in case of an error
    private String mLastOverloadUrl = "";

    protected IOnHtmlWebViewInterface mHtmlWeViewListener;
    protected IOnInitFinishedListener mSdkInitListener;

    protected boolean mIsShowClose = true;

    protected LoadingProgressView mLoadingProgressView;
    protected Utils.StaticHandler mEventHandler;
    protected ItemViewPagerAdapter.ViewPagerItemClickListener mViewPagerItemClickListener;
    protected Utils.StaticHandler mExecutionHandler;

    protected SoftReference<Context> mInFocusActivityContext;
    protected AdState mAdState = AdState.AD_STOPED;

    public HtmlViewWrapper(Context context, boolean clearCache)
    {
        super(context);
        mExecutionHandler = new Utils.StaticHandler(Looper.getMainLooper());
        initView(clearCache);
    }

    public void setWidgetType(String widgetType)
    {
        mWidgetType = widgetType;
    }

    public String getWidgetType()
    {
        return mWidgetType;
    }

    public void setStyleID(String styleID)
    {
        mStyleId = styleID;
    }

    public void setData(ContentItem contentItem)
    {
        mContentItem = contentItem;
        mHtmlPageUrl = contentItem.getData();

        setAndApplyExternalProperties(contentItem.getExtraParameters());
    }

    /**
     * Set current in focus activity context
     *
     * @param context
     */
    public void setInFocusActivityContext(Context context)
    {
        mInFocusActivityContext = new SoftReference<Context>(context);
    }

    /**
     * Set webview properties
     */
    private void setAndApplyExternalProperties(JSONObject jsonObject)
    {
        if (jsonObject != null)
        {
            if (jsonObject.has("webview_properties") && jsonObject.isNull("webview_properties") == false)
            {
                JSONObject mwbViewPropertiesMap = jsonObject.optJSONObject("webview_properties");
                if (mwbViewPropertiesMap != null)
                {
                    mWebView.applyProperties(mwbViewPropertiesMap);
                }
            }

            mIsShowClose = jsonObject.optBoolean("showClose", false);
        }
    }

    public void reloadHtml()
    {
        loadHtml(mHtmlPageUrl);
    }

    /**
     * Start HTML Widget/Page loading
     */
    public void loadHtml(String link)
    {
        mLastOverloadUrl = "";
        mHtmlPageUrl = link;
        if (mHtmlPageUrl != null)
        {
            StringBuilder stringBuilder = populateMustHaveParams(getContext(), mHtmlPageUrl);
            stringBuilder.append("&widget_type=").append(mWidgetType);

            mWebView.loadUrl(stringBuilder.toString());
        }
    }

    /**
     * This params must be passed for each html loaded url in every webview
     */
    public static StringBuilder populateMustHaveParams(Context context, String mainUrl)
    {

        StringBuilder stringBuilder = new StringBuilder(mainUrl);

        if (!mainUrl.contains("?"))
        {
            stringBuilder.append("?");
        }
        else
        {
            stringBuilder.append("&");
        }

        String appVersionCode = "";
        String appVersionName = "";
        try
        {
            PackageManager packageManager = context.getPackageManager();
            PackageInfo packageInfo = packageManager.getPackageInfo(context.getPackageName(), 0);
            appVersionCode = String.valueOf(packageInfo.versionCode);
            appVersionName = packageInfo.versionName;
        }
        catch (Exception ex)
        {
            SDKLogger.printErrorLog(TAG, "Error when trying to get app version name and code: " + ex.getMessage());
        }

        stringBuilder.append("publisher_id=").append(KidozSDK.getPublisherID());
        stringBuilder.append("&auth_token=").append(KidozSDK.getAuthToken());
        stringBuilder.append("&package_id=").append(context.getPackageName());
        stringBuilder.append("&sdk_version=" + ConstantDef.SDK_VERSION);
        stringBuilder.append("&actual_sdk_version=" + ConstantDef.ACTUAL_SDK_VERSION);
        stringBuilder.append("&os_version=").append(Build.VERSION.SDK_INT);
        stringBuilder.append("&os_type=" + "android");
        stringBuilder.append("&extension_type=").append(String.valueOf(ConstantDef.SDK_EXTENSION_TYPE));

        stringBuilder.append("&app_version_code=").append(appVersionCode);
        stringBuilder.append("&app_version_name=").append(appVersionName);

        if (ScreenUtils.getIsTablet(context))
        {
            stringBuilder.append("&device_type=" + "2");
        }
        else
        {
            stringBuilder.append("&device_type=" + "1");
        }

        stringBuilder.append("&device_lang=").append(Locale.getDefault().getLanguage());

        String webviewVersion = "w";
        String useragent = Utils.getDefaultUserAgent();
        if (useragent != null && useragent.contains("Chrome"))
        {
            int index = useragent.indexOf("Chrome");
            String sub = useragent.substring(index, useragent.length());
            String[] res = sub.split(" ");
            if (res != null && res.length > 0)
            {
                String[] tempVal = res[0].split("/");
                if (tempVal != null && tempVal.length > 1)
                {
                    webviewVersion = "c_" + tempVal[1];
                }
            }
        }

        stringBuilder.append("&webview_version=").append(webviewVersion);
        stringBuilder.append("&device_hash=").append(Utils.generateUniqeDeviceID(context.getPackageName(), KidozSDK.getPublisherID()));
        stringBuilder.append("&gid=").append(SdkDeviceUtils.getGoogleAdvertisingID(context));

        try
        {
            Point point = ScreenUtils.getScreenSize(context);
            stringBuilder.append("&resolution_height=").append(point.y);
            stringBuilder.append("&resolution_width=").append(point.x);
            stringBuilder.append("&screen_size=").append(ScreenUtils.getDeviceScreenSizeInInches(context));
            stringBuilder.append("&dpi=").append(context.getResources().getDisplayMetrics().density);
            stringBuilder.append("&screen_category=").append(context.getResources().getConfiguration().screenLayout & Configuration.SCREENLAYOUT_SIZE_MASK);
        }
        catch (Exception e)
        {

        }

        return stringBuilder;
    }

    private void initView(boolean clearCache)
    {
        initHandler();
        initWebView(clearCache);

        initProgressLoadingView();
    }

    private void initHandler()
    {
        mEventHandler = new Utils.StaticHandler(Looper.getMainLooper())
        {
            @Override
            public void handleMessage(Message msg)
            {
                super.handleMessage(msg);

                onHandleLocalUiThreadedEvents(msg);
            }
        };
    }

    /**
     * Handles UI threaded local events
     */
    protected void onHandleLocalUiThreadedEvents(Message msg)
    {
        switch (msg.what)
        {
            case ON_LOAD_STARTED:
            {
                if (mHtmlWeViewListener != null)
                {
                    mHtmlWeViewListener.onHtmlStartLoading();
                }
                break;
            }

            case ON_LOAD_FINISHED:
            {
                if (mHtmlWeViewListener != null)
                {
                    mHtmlWeViewListener.onHtmlFinishedLoading();
                }

                if (mSdkInitListener != null)
                {
                    mSdkInitListener.onInitFinished();
                    mSdkInitListener = null;
                }
                break;
            }

            case ON_VIEW_READY:
            {
                if (mHtmlWeViewListener != null)
                {
                    mHtmlWeViewListener.onViewReady();
                }
                break;
            }

            case ON_CLOSE:
            {
                if (mHtmlWeViewListener != null)
                {
                    mHtmlWeViewListener.onClose();
                }
                WidgetEventMessage eventMessage = null;
                if (mWidgetType != null && mWidgetType.equals(WidgetType.WIDGET_TYPE_INTERSTITIAL_REWARDED.getStringValue()))
                {
                    eventMessage = new WidgetEventMessage(EventMessage.MessageType.INTERSTITIAL_AD_CLOSE, WidgetType.WIDGET_TYPE_INTERSTITIAL_REWARDED);
                }
                else
                {
                    eventMessage = new WidgetEventMessage(EventMessage.MessageType.INTERSTITIAL_AD_CLOSE, WidgetType.WIDGET_TYPE_INTERSTITIAL);
                }
                EventBus.getDefault().post(eventMessage);
                break;
            }

            case ON_RECEIVED_ERROR:
            {
                if (mHtmlWeViewListener != null)
                {
                    mHtmlWeViewListener.onErrorReceived();
                }
                break;
            }

            case ON_NOTIFY_VAST_AD_READY:
            {
                if (mHtmlWeViewListener != null)
                {
                    JSONObject jsonObject = null;

                    if (msg.obj != null && msg.obj instanceof String)
                    {
                        try
                        {
                            jsonObject = new JSONObject((String) msg.obj);
                        }
                        catch (JSONException e)
                        {
                            SDKLogger.printErrorLog(HtmlViewWrapper.class.getCanonicalName(), "Error parsing vast properties!");
                        }
                    }

                    if (msg.arg1 == 1)
                    {
                        mHtmlWeViewListener.onNotifyVastReady(true, jsonObject);
                    }
                    else
                    {
                        mHtmlWeViewListener.onNotifyVastReady(false, jsonObject);
                    }
                }
                break;
            }

            case ON_REWARDED:
            {
                if (mHtmlWeViewListener != null)
                {
                    mHtmlWeViewListener.onRewarded();
                }
                break;
            }
            case ON_REWARDED_STARTED:
            {
                if (mHtmlWeViewListener != null)
                {
                    mHtmlWeViewListener.onRewardedVideoStarted();
                }
                break;
            }

            case ON_AD_STATE_CHANGE:
            {
                try
                {
                    mAdState = AdState.values()[msg.arg1];
                }
                catch (Exception e)
                {
                }

                if (mHtmlWeViewListener != null)
                {
                    mHtmlWeViewListener.onAdStateChanged(mAdState);
                }
                break;
            }
        }
    }

    /**
     * Initiate main webview
     */
    private void initWebView(boolean clearCache)
    {
        mWebView = new HtmlFiveWebView(getContext());
        if (clearCache)
        {
            clearCache();
        }

        // Apply java script interfaces to the webView
        applyJavaScriptInterfaces();

        mWebView.setWebViewClient(new WebViewClient()
        {
            @Override
            public void onPageStarted(WebView view, String url, Bitmap favicon)
            {
                super.onPageStarted(view, url, favicon);
                mEventHandler.sendEmptyMessage(ON_LOAD_STARTED);
            }

            @Override
            public void onPageFinished(WebView view, String url)
            {
                super.onPageFinished(view, url);
                mEventHandler.sendEmptyMessage(ON_LOAD_FINISHED);
            }

            @Override
            public boolean shouldOverrideUrlLoading(WebView view, final String url)
            {
                mLastOverloadUrl = url;
                if (mContentItem != null && mContentItem.getContentType() != null && mContentItem.getContentType() == ContentType.ROVIO_ITEM)
                {
                    return super.shouldOverrideUrlLoading(view, url);
                }
                else
                {
                    return handleOverrideUrlLoading(view, url);
                }
            }

            @Override
            public void onReceivedError(WebView view, int errorCode, String description, String failingUrl)
            {
                Event event = new Event();
                if (mContentItem != null)
                {
                    event.addParameterToJsonObject(EventParameters.ITEM_ID, mContentItem.getId());
                    event.addParameterToJsonObject(EventParameters.ADVERTISER_ID, mContentItem.getAdvertiserID());
                }

                EventManager.getInstance(getContext()).logEvent(getContext(), mWidgetType, mStyleId, EventManager.LOG_CRITICAL_LEVEL, event, EventParameters.CATEGORY_WEB_VIEW_ERROR, String.valueOf(errorCode).concat(": ").concat(description), failingUrl);
                EventManager.getInstance(getContext()).startEventsSync(view.getContext());
            }
        });

        mWebView.setWebChromeClient(new WebChromeClient()
        {
            @Override
            public void onProgressChanged(WebView view, int newProgress)
            {
                super.onProgressChanged(view, newProgress);
            }
        });
        addView(mWebView, new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.MATCH_PARENT, RelativeLayout.LayoutParams.MATCH_PARENT));
    }

    @SuppressLint("JavascriptInterface")
    protected void applyJavaScriptInterfaces()
    {
        mWebView.addJavascriptInterface(this, "KidozAndroid");
    }

    /**
     * Handles the call for override url loading and aplies correct logic depends on type and other params
     */
    protected boolean handleOverrideUrlLoading(WebView view, final String url)
    {
        /*boolean isForwardingToGoogle = false;
        if (url.contains("market://") || url.contains("play.google.com"))
        {
            isForwardingToGoogle = true;
        } else if (url.startsWith("https://m.youtube") || url.startsWith("http://m.youtube") || url.startsWith("https://www.youtube") || url.startsWith("http://www.youtube"))
        {
            // TODO Verify what is done here
        } else
        {
            isForwardingToGoogle = true;
        }

        if (isForwardingToGoogle == true)
        {
            openGooglePlayStore(url);
        }*/

        openGooglePlayStore(url);

        return true;
    }

    /**
     * Open correct google play store
     */
    protected void openGooglePlayStore(String url)
    {
        try
        {
            if (mContentItem != null && mContentItem.getContentType() != null)
            {
                switch (mContentItem.getContentType())
                {
                    case PROMOTED_PLAY_APPLICATION:
                    {
                        openGooglePlayInBackground(url);
                        break;
                    }
                    case HTML:
                    {
                        if (mContentItem.getJSONitem() != null)
                        {
                            try
                            {
                                JSONArray jsonArray = mContentItem.getJSONitem();
                                JSONObject extraJsonObject = jsonArray.getJSONObject(11);
                                if (extraJsonObject.optBoolean("cpi_play", false) == true)
                                {
                                    openGooglePlayInBackground(url);
                                }
                                else
                                {
                                    openGooglePlayInForground(url);
                                }
                            }
                            catch (Exception ex)
                            {
                                openGooglePlayInForground(url);
                            }
                        }
                        else
                        {
                            openGooglePlayInForground(url);
                        }
                        break;
                    }
                    default:
                    {
                        openGooglePlayInForground(url);
                    }
                }
            }
            else
            {
                openGooglePlayInForground(url);
            }
        }
        catch (Exception ex)
        {
            SDKLogger.printErrorLog(TAG, "Error when trying to open google play for promoted app: \n" + ex.getMessage());
        }
    }

    protected void openGooglePlayInForground(final String url)
    {
        Context context = mInFocusActivityContext.get();
        if (context == null)
        {
            context = getContext();
        }

        final Context finalContext = context;
        ContentExecutionHandler.checkForParentalLock(context, new ContentExecutionHandler.IOnParentalLockStatusListener()
        {
            @Override
            public void onLockActive(boolean isPasswordCorrect)
            {
                if (isPasswordCorrect == true)
                {
                    Intent intent = new Intent(Intent.ACTION_VIEW);
                    intent.setData(Uri.parse(url));
                    if (finalContext != null)
                    {
                        if (mHtmlWeViewListener != null)
                        {
                            mHtmlWeViewListener.onClose();
                        }

                        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                        finalContext.startActivity(intent);
                    }
                }
                else
                {
                    EventManager.getInstance(getContext()).logEvent(getContext(), mWidgetType, mStyleId, EventManager.LOG_NORMAL_LEVEL, null, EventParameters.CATEGORY_SPONSORED_CONTENT, EventParameters.ACTION_PARENTAL_INCORRECT_PASSWORD, mContentItem.getId());
                }
            }

            @Override
            public void onLockNotActive()
            {
                Intent intent = new Intent(Intent.ACTION_VIEW);
                intent.setData(Uri.parse(url));
                if (finalContext != null)
                {
                    if (mHtmlWeViewListener != null)
                    {
                        mHtmlWeViewListener.onClose();
                    }
                    intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                    finalContext.startActivity(intent);
                }
            }
        });
    }

    protected void openGooglePlayInBackground(String url)
    {
        Context context = mInFocusActivityContext.get();
        if (context == null)
        {
            context = getContext();
        }

        if (mHtmlWeViewListener != null)
        {
            mHtmlWeViewListener.onClose();
        }

        ContentItem contentItem = new ContentItem();
        contentItem.setContentType(ContentType.PROMOTED_PLAY_APPLICATION);
        contentItem.setData(url);
        contentItem.setId(mContentItem.getId());
        ContentExecutionHandler.handleContentItemClick(context, contentItem, mWidgetType, mStyleId, 0, false, null);
    }

    private void initProgressLoadingView()
    {
        mLoadingProgressView = new LoadingProgressView(getContext());

        Point point = Utils.getScreenSize(getContext());
        int size = (int) (Math.min(point.x, point.y) * LOADING_PROGRESS_DEFAULT_RATIO);
        RelativeLayout.LayoutParams prParams = new RelativeLayout.LayoutParams(size, size);
        prParams.addRule(RelativeLayout.CENTER_IN_PARENT);

        addView(mLoadingProgressView, prParams);
    }

    /**
     * Handles impression served action
     */
    protected void invokeOnImpressionServed(final String item_id, final String item_name, final String ad_id, final String view_index)
    {
        //SDKLogger.printToast(getContext(), "JS : onImpressionServed");
        mExecutionHandler.post(new Runnable()
        {
            @Override
            public void run()
            {
                int positionIndexAsInteger = 0;

                try
                {
                    positionIndexAsInteger = Integer.parseInt(view_index);
                }
                catch (Exception ex)
                {
                    SDKLogger.printErrorLog(TAG, "Error when trying to parse positionIndex: " + ex.getMessage());
                }
                EventManager.getInstance(getContext()).logSponsoredContentImpressionEvent(getContext(), mWidgetType, mStyleId, EventParameters.ACTION_IMPRESSION, item_name, ad_id, item_id, positionIndexAsInteger);
            }
        });
    }

    /**
     * Handles simulate click action
     */
    protected void invokeOnSimulateClick(final String json, final int positionIndex)
    {
        mExecutionHandler.post(new Runnable()
        {
            @Override
            public void run()
            {
                ContentItem item = null;
                try
                {
                    JSONArray jsonArray = new JSONArray(json);
                    if (jsonArray != null && jsonArray.length() > 1)
                    {
                        JSONArray keysArray = jsonArray.getJSONArray(0);
                        HashMap<String, Integer> keysMap = new LinkedHashMap<String, Integer>();
                        if (keysArray != null && keysArray.length() > 0)
                        {
                            for (int i = 0; i < keysArray.length(); i++)
                            {
                                keysMap.put(keysArray.getString(i), i);
                            }
                        }

                        if (keysMap.isEmpty() == false && jsonArray.getJSONArray(1) != null)
                        {
                            item = new ContentItem(jsonArray.getJSONArray(1), keysMap);
                        }
                    }
                }
                catch (JSONException e)
                {
                }

                if (item != null)
                {
                    if (mAllowClickHandling)
                    {
                        mAllowClickHandling = false;
                        if (mInFocusActivityContext != null && mInFocusActivityContext.get() != null)
                        {
                            ContentExecutionHandler.handleContentItemClick(mInFocusActivityContext.get(), item, mWidgetType, mStyleId, positionIndex, true, new ContentExecutionHandler.IOnHandleClickListener()
                            {
                                @Override
                                public void onRestoreClick()
                                {
                                    mAllowClickHandling = true;
                                }
                            });
                        }
                    }
                }
            }
        });
    }

    /**
     * Handles on about click action
     */
    protected void invokeAboutClick()
    {
        //SDKLogger.printToast(getContext(), "JS : invokeAboutClick");
        mExecutionHandler.post(new Runnable()
        {
            @Override
            public void run()
            {
                startAboutKidozDialog();
            }
        });
    }

    /**
     * Handles on parental icon click action
     */
    protected void invokeParentalClick()
    {
        //SDKLogger.printToast(getContext(), "JS : invokeParentalClick");
        mExecutionHandler.post(new Runnable()
        {
            @Override
            public void run()
            {
                startParentalLockDialog();
            }
        });
    }

    /**
     * Handles on ,aximized click action
     */
    protected void invokeMaximizedClick()
    {
        //SDKLogger.printToast(getContext(), "JS : onMaximize");
        mExecutionHandler.post(new Runnable()
        {
            @Override
            public void run()
            {
                if (mHtmlWeViewListener != null)
                {
                    mHtmlWeViewListener.onOpenMaximized();
                }
            }
        });
    }

    /**
     * Handles on initiate web view properties action
     */
    protected void invokeInitiateWebViewProperties(final String properties)
    {
        //SDKLogger.printToast(getContext(), "JS : onUpdateWebViewProperties " + properties);
        mExecutionHandler.post(new Runnable()
        {
            @Override
            public void run()
            {
                try
                {
                    mWebView.applyProperties(new JSONObject(properties));
                }
                catch (JSONException e)
                {

                }
            }
        });
    }

    /**
     * Handles on conversion served
     */
    protected void onInvokeConversionServed(final String item_id, final String item_name, final String ad_id, final String view_index, final String rewardKey, final String rewardValue)
    {
        //SDKLogger.printToast(getContext(), "JS : onInvokeConversionServed");
        mExecutionHandler.post(new Runnable()
        {
            @Override
            public void run()
            {
                int positionIndexAsInteger = 0;
                int rewardKeyAsInteger = 0;
                int rewardValueAsInteger = 0;

                try
                {
                    positionIndexAsInteger = Integer.parseInt(view_index);
                }
                catch (Exception ex)
                {
                    SDKLogger.printErrorLog(TAG, "Error when trying to parse positionIndex: " + ex.getMessage());
                }
                try
                {
                    rewardKeyAsInteger = Integer.parseInt(rewardKey);
                }
                catch (Exception ex)
                {
                    SDKLogger.printErrorLog(TAG, "Error when trying to parse rewardKey: " + ex.getMessage());
                }
                try
                {
                    rewardValueAsInteger = Integer.parseInt(rewardValue);
                }
                catch (Exception ex)
                {
                    SDKLogger.printErrorLog(TAG, "Error when trying to parse rewardValue: " + ex.getMessage());
                }
                Event event = new Event();
                EventManager.getInstance(getContext()).logSponsoredConversionEvent(getContext(), mWidgetType, mStyleId, event, item_name, ad_id, item_id, positionIndexAsInteger, rewardKeyAsInteger, rewardValueAsInteger);
            }
        });
    }

    /**
     * Handles store local params tp a local storage
     */
    protected void onRequestStoreLocalParam(final String key, final String value, final String scope_type)
    {
        //SDKLogger.printToast(getContext(), "JS : onStoreLocalParam");
        mExecutionHandler.post(new Runnable()
        {
            @Override
            public void run()
            {
                SdkCookieManager.saveData(getContext(), mWidgetType, mHtmlPageUrl, key, value, StorageLife.valueOf(scope_type));
            }
        });
    }

    /**
     * Handles requesting local params from local storage
     */
    protected void onGetLocalParam(final String function, final String key)
    {
        final String value = SdkCookieManager.loadData(getContext(), mWidgetType, mHtmlPageUrl, key);

        String defaultName = "kidozReturnedValue";
        if (function != null)
        {
            defaultName = function;
        }
        final String finalDefaultName = defaultName;
        mEventHandler.post(new Runnable()
        {
            @Override
            public void run()
            {
                invokeJSfunction("javascript:" + finalDefaultName + "('" + key + "','" + value + "'  );");
            }
        });
    }

    /**
     * Handles on done playback event sent from html
     */
    private void onGetDonePlayback()
    {
        //SDKLogger.printToast(getContext(), "JS : getDonePlayback");
        mExecutionHandler.post(new Runnable()
        {
            @Override
            public void run()
            {
                if (mHtmlWeViewListener != null)
                {
                    mHtmlWeViewListener.onDonePlayBack();
                }
            }
        });
    }

    /**
     * Handles request from parental lock state from client to html
     */
    private void onGetParentalLockState(final String function)
    {
        //SDKLogger.printToast(getContext(), "JS : getParentalLockStatus : FucntionName: " + function);
        final boolean isLockActive = ParentalLockDialog.isParentalLockActive(getContext());

        mExecutionHandler.post(new Runnable()
        {
            @Override
            public void run()
            {
                invokeJSfunction("javascript:" + function + "('" + isLockActive + "');");
            }
        });
    }

    /**
     * Handles send event form HTML
     */
    protected void onForwardToGooglePlay(final String advertiserID, final String appID, final String googlePlayLink, final String widgetType, final String styleID, final String itemIndex)
    {
        mExecutionHandler.post(new Runnable()
        {
            @Override
            public void run()
            {
                int itemIndexAsInteger = 0;
                try
                {
                    itemIndexAsInteger = Integer.parseInt(itemIndex);
                }
                catch (Exception ex)
                {
                    SDKLogger.printErrorLog(TAG, "Error when trying to parse item index: " + ex.getMessage());
                }
                ContentItem contentItem = new ContentItem();
                contentItem.setId(appID);
                contentItem.setName("");
                contentItem.setData(googlePlayLink);
                contentItem.setContentType(ContentType.PROMOTED_PLAY_APPLICATION);
                contentItem.setAdvertiserID(advertiserID);
                contentItem.setIsPromoted(true);

                if (mViewPagerItemClickListener != null)
                {
                    mViewPagerItemClickListener.onClickEnd(contentItem, itemIndexAsInteger);
                }
                else
                {
                    if (mInFocusActivityContext != null && mInFocusActivityContext.get() != null)
                    {
                        ContentExecutionHandler.handleContentItemClick(mInFocusActivityContext.get(), contentItem, widgetType, styleID, itemIndexAsInteger, false, null);
                    }
                }
            }
        });
    }

    protected void invokeSetDeviceOrientation(final int orientation)
    {
        if (mInFocusActivityContext != null && mInFocusActivityContext.get() != null && mInFocusActivityContext.get() instanceof Activity)
        {
            mExecutionHandler.post(new Runnable()
            {
                @Override
                public void run()
                {
                    ((Activity) mInFocusActivityContext.get()).setRequestedOrientation(orientation);
                }
            });
        }
    }

    /**
     * Handles send event form HTML
     */
    protected void onInvokeSendEvent(final String type, final String category, final String action, final String label, final String jsonData)
    {
        mExecutionHandler.post(new Runnable()
        {
            @Override
            public void run()
            {
                SDKLogger.printDebbugLog("RONY", "type = " + type);
                SDKLogger.printDebbugLog("RONY", "category = " + category);
                SDKLogger.printDebbugLog("RONY", "action = " + action);
                SDKLogger.printDebbugLog("RONY", "label = " + label);
                SDKLogger.printDebbugLog("RONY", "jsonData = " + jsonData);
                SDKLogger.printDebbugLog("RONY", "mStyleId = " + mStyleId);
                SDKLogger.printDebbugLog("RONY", "mWidgetType = " + mWidgetType);

                Event event = new Event();
                event.setLogLevel(EventManager.LOG_NORMAL_LEVEL);
                event.addParameterToJsonObject(jsonData);
                EventManager.getInstance(getContext()).logEvent(getContext(), mWidgetType, mStyleId, EventManager.LOG_NORMAL_LEVEL, event, category, action, label);
            }
        });
    }

    /**
     * Handles toggle loading state
     */
    protected void onToggleLoadingState(final boolean isLoading)
    {
        if (mLoadingProgressView != null)
        {
            mExecutionHandler.post(new Runnable()
            {
                @Override
                public void run()
                {
                    if (isLoading == true)
                    {
                        mLoadingProgressView.startLoadingAnimation();
                    }
                    else
                    {
                        mLoadingProgressView.stopLoadingAnimation();
                    }
                }
            });
        }
    }

    /**
     * Handles notify vast ready event
     */
    protected void onNotifyIsVastAdReady(final boolean isVastReady, String vastSettings)
    {
        Message message = Message.obtain();
        message.what = ON_NOTIFY_VAST_AD_READY;
        if (isVastReady)
        {
            message.arg1 = 1;
        }
        else
        {
            message.arg1 = 0;
        }

        if (vastSettings != null)
        {
            message.obj = vastSettings;
        }
        mEventHandler.sendMessage(message);
    }

    /**
     * Handles notify vast ready event
     */
    protected void onNotifyIsVastAdReady(final boolean isVastReady)
    {
        Message message = Message.obtain();
        message.what = ON_NOTIFY_VAST_AD_READY;
        if (isVastReady)
        {
            message.arg1 = 1;
        }
        else
        {
            message.arg1 = 0;
        }
        mEventHandler.sendMessage(message);
    }

    /**
     * Set parental lock state, update html icon view
     */
    protected void setParentalLockState(boolean locked)
    {
        invokeJSfunction("javascript:kidozOnParentalLockStateChanged('" + locked + "');");
    }

    /**
     * Start parental lock dialog
     */
    protected void startParentalLockDialog()
    {
        if (mInFocusActivityContext != null && mInFocusActivityContext.get() != null)
        {
            ParentalLockDialog.startParentalDialog(mInFocusActivityContext.get(), false, 0.5f, 0.5f, new ParentalLockDialog.IOnParentalDialogListener()
            {
                @Override
                public void onCloseDialog()
                {
                    //Refresh the parental lock icon
                    updateLockIcon();
                }

                @Override
                public void onInputPass(boolean isCorrectPass)
                {
                    updateLockIcon();
                }
            });
            updateLockIcon();
        }
    }

    /**
     * Start about kidoz dialog
     */
    protected void startAboutKidozDialog()
    {
        int[] popLocation = new int[2];
        popLocation[0] = (int) (ScreenUtils.getScreenSize(getContext(), true) * 0.5f);
        popLocation[1] = (int) (ScreenUtils.getScreenSize(getContext(), false) * 0.5f);
        if (mInFocusActivityContext != null && mInFocusActivityContext.get() != null)
        {
            AboutKidozDialog aboutKidozDialog = new AboutKidozDialog(mInFocusActivityContext.get(), popLocation);
            aboutKidozDialog.openDialog();
        }
    }

    /**
     * Update lock icon at HTML view depending on current lock state
     */
    protected void updateLockIcon()
    {
        try
        {
            boolean isLockActive = ParentalLockDialog.isParentalLockActive(getContext());;
            if (isLockActive == true)
            {
                setParentalLockState(true);
            }
            else
            {
                setParentalLockState(false);
            }
        }
        catch (Exception ex)
        {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(TAG, "Error when trying to load parental lock image: " + ex.getMessage());
        }
    }

    /**
     * Request focus ON item
     */
    public void requestFocusOn(boolean isFullScreen)
    {
        SDKLogger.printWarningLog("###@@@###  Focus ON");

        invokeJSfunction("javascript:toonsWebApi.appForeground()");
        if (mContentItem != null)
        {
            JSONArray jSONArray = mContentItem.getJSONitem();
            try
            {
                String itemName = jSONArray.getString(1);
                itemName = itemName.replace("\"", "");
                jSONArray.put(1, itemName);
            }
            catch (Exception ex)
            {
            }
            String json = jSONArray.toString();
            String func = "javascript:kidozOnFocusOn('" + mContentItem.getId() + "','" + json + "','" + isFullScreen + "','" + mWidgetType + "','" + mContentItem.getRealViewIndex() + "');";

            invokeJSfunction(func);
        }
    }

    /**
     * Request for prepare content item to get ready for focus on
     */
   /* public void requestPrepareFocusOn(boolean isFullScreen)
    {
        if (mContentItem != null)
        {
            String json = mContentItem.getJSONitem().toString();
            String func = "javascript:kidozOnPrepareFocusOn('" + mContentItem.getId() + "','" + json + "','" + isFullScreen + "','" + mWidgetType + "','" + mContentItem.getRealViewIndex() + "');";

            invokeJSfunction(func);
        }
    }*/

    /**
     * Request focus ON item
     */
    public void requestFocusOff()
    {
        SDKLogger.printWarningLog("###@@@###  Focus OFF");

        invokeJSfunction("javascript:toonsWebApi.appBackground()");
        if (mContentItem != null)
        {
            invokeJSfunction("javascript:kidozOnFocusOff('" + mContentItem.getId() + "');");
        }
    }

    /**
     * Make <b>Maximize</b> request to an html view
     */
    public void requestMaximize()
    {
        String func = "javascript:kidozOnMaximize();";
        invokeJSfunction(func);
    }

    /**
     * Make <b>Minimize</b> request to an html view
     */
    public void requestMinimize()
    {
        invokeJSfunction("javascript:kidozOnMinimize();");
    }

    /**
     * Request widget open
     */
    public void requestWidgetOpen(String widgetType)
    {
        if (widgetType != null)
        {
            invokeJSfunction("javascript:kidozOnWidgetOpen('" + widgetType + "');");
        }
    }

    /**
     * Request widget close
     */
    public void requestWidgetClose(String widgetType)
    {
        if (widgetType != null)
        {
            invokeJSfunction("javascript:kidozOnWidgetClose('" + widgetType + "');");
        }
    }

    /**
     * Request vast ads
     */
    public void requestVastAds(int adType)
    {
        invokeJSfunction("javascript:requestAds('" + adType + "');");
    }

    /**
     * Start vast ad playback
     */
    public void startVastAd()
    {
        invokeJSfunction("javascript:startVastAd();");
    }

    /**
     * Stop vast ad playback
     */
    public void stopVastAd()
    {
        invokeJSfunction("javascript:stopVastAd();");
    }

    /**
     * Pause vast ad playback
     */
    public void pauseVastAd()
    {
        invokeJSfunction("javascript:pauseVastAd();");
    }

    /**
     * Pause vast ad playback
     */
    public void resumeVastAd()
    {
        invokeJSfunction("javascript:resumeVastAd();");
    }

    /**
     * Clear the content of HTML view
     */
    public void clearHtmlView()
    {
        mWebView.loadUrl("");
        //mWebView.stopLoading();
    }

    public void setHtmlWebViewListener(IOnHtmlWebViewInterface htmlWeViewListener)
    {
        mHtmlWeViewListener = htmlWeViewListener;
    }

    public void setSdkInitListener(IOnInitFinishedListener listener)
    {
        mSdkInitListener = listener;
    }

    public void invokeJSfunction(final String function)
    {
        try
        {
            mExecutionHandler.post(new Runnable()
            {
                @Override
                public void run()
                {
                    if (Build.VERSION.SDK_INT >= 19)
                    {
                        mWebView.evaluateJavascript(function, null);
                    }
                    else
                    {
                        mWebView.loadUrl(function);
                    }
                }
            });
        }
        catch (Exception ex)
        {
        }
    }

    /**
     * Is show close button
     */
    public boolean isShowClose()
    {
        return mIsShowClose;
    }

    /**
     * Show loading circle progress bar view
     */
    public void showLoadingProgressView()
    {
        if (mLoadingProgressView != null)
        {
            mLoadingProgressView.startLoadingAnimation();
        }
    }

    /**
     * Hide loading circle progress bar view
     */
    public void hideLoadingProgressView()
    {
        if (mLoadingProgressView != null)
        {
            mLoadingProgressView.stopLoadingAnimation();
        }
    }

    /**
     * Stop and release webview instance
     */
    public void stopAndReleaseWebView()
    {
        if (mWebView != null)
        {
            mWebView.stopAndReleaseWebView();
        }
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh)
    {
        super.onSizeChanged(w, h, oldw, oldh);

        int size = (int) (Math.min(w, h) * LOADING_PROGRESS_DEFAULT_RATIO);
        if (mLoadingProgressView != null)
        {
            mLoadingProgressView.getLayoutParams().height = size;
            mLoadingProgressView.getLayoutParams().width = size;
            mLoadingProgressView.setCircleWidthRelativeToSize(size);
        }
    }

    public void setViewPagerItemClickListener(ItemViewPagerAdapter.ViewPagerItemClickListener viewPagerItemClickListener)
    {
        mViewPagerItemClickListener = viewPagerItemClickListener;
    }

    public void clearCache()
    {
        if (mWebView != null)
        {
            mWebView.clearCache(true);
            mWebView.clearHistory();
        }
    }

    public String getLastOverloadUrl() {
        return mLastOverloadUrl;
    }

    public interface IOnInitFinishedListener
    {
        void onInitFinished();
    }

    /**
     * JAVA SCRIPT INTERFACE IMPLEMENTATION
     */
    @Override
    @android.webkit.JavascriptInterface
    public void onViewReady()
    {
        mEventHandler.sendEmptyMessage(ON_VIEW_READY);
    }

    @Override
    @android.webkit.JavascriptInterface
    public void simulateClick(String contentItemJson, int positionIndex)
    {
        try
        {
            invokeOnSimulateClick(contentItemJson, positionIndex);
        }
        catch (Exception e)
        {
        }
    }

    @Override
    @android.webkit.JavascriptInterface
    public void onSendImpressionEvent(String item_id, String item_name, String ad_id, String view_index)
    {
        try
        {
            invokeOnImpressionServed(item_id, item_name, ad_id, view_index);
        }
        catch (Exception e)
        {
        }
    }

    @Override
    @android.webkit.JavascriptInterface
    public void onInvokeAboutClick()
    {
        invokeAboutClick();
    }

    @Override
    @android.webkit.JavascriptInterface
    public void onInvokeParentalClick()
    {
        invokeParentalClick();
    }

    @Override
    @android.webkit.JavascriptInterface
    public void onInvokeCloseClick()
    {
        mEventHandler.sendEmptyMessage(ON_CLOSE);
    }

    @Override
    @android.webkit.JavascriptInterface
    public void toggleWidgetState(final boolean isEnabled)
    {
        mEventHandler.postDelayed(new Runnable()
        {
            @Override
            public void run()
            {
                if (mHtmlWeViewListener != null)
                {
                    mHtmlWeViewListener.toggleWidgetState(isEnabled);
                }
            }
        }, 0);
    }

    @Override
    @android.webkit.JavascriptInterface
    public void onInvokeMaximize()
    {
        invokeMaximizedClick();
    }

    @Override
    @android.webkit.JavascriptInterface
    public void onInitWebViewWithProperties(String properties)
    {
        try
        {
            invokeInitiateWebViewProperties(properties);
        }
        catch (Exception e)
        {
        }
    }

    @Override
    @android.webkit.JavascriptInterface
    public void onSendConversionEvent(String item_id, String item_name, String ad_id, String view_index, String rewardKey, String rewardValue)
    {
        try
        {
            onInvokeConversionServed(item_id, item_name, ad_id, view_index, rewardKey, rewardValue);
        }
        catch (Exception e)
        {
        }
    }

    @Override
    @android.webkit.JavascriptInterface
    public void onStoreLocalParameter(String key, String value, String scope_type)
    {
        try
        {
            onRequestStoreLocalParam(key, value, scope_type);
        }
        catch (Exception e)
        {
        }
    }

    @Override
    @android.webkit.JavascriptInterface
    public void getLocalParameter(final String function, final String key)
    {
        try
        {
            onGetLocalParam(function, key);
        }
        catch (Exception e)
        {
        }
    }

    @Override
    @android.webkit.JavascriptInterface
    public void onDone()
    {
        try
        {
            onGetDonePlayback();
        }
        catch (Exception e)
        {
        }
    }

    @Override
    @android.webkit.JavascriptInterface
    public void getParentalLockStatus(String function)
    {
        try
        {
            onGetParentalLockState(function);
        }
        catch (Exception e)
        {
        }
    }

    @Override
    @android.webkit.JavascriptInterface
    public void forwardToGooglePlay(final String advertiserID, final String appID, final String googlePlayLink, final String widjetType, final String styleID, final String itemIndex)
    {
        onForwardToGooglePlay(advertiserID, appID, googlePlayLink, widjetType, styleID, itemIndex);
    }

    @Override
    @android.webkit.JavascriptInterface
    public void sendEvent(String type, String category, String action, String label, String jsonData)
    {
        onInvokeSendEvent(type, category, action, label, jsonData);
    }

    @Override
    @android.webkit.JavascriptInterface
    public void toggleLoadingState(String isLoading)
    {
        try
        {
            boolean isLoadingAsBoolean = Boolean.parseBoolean(isLoading);
            onToggleLoadingState(isLoadingAsBoolean);
        }
        catch (Exception ex)
        {
            SDKLogger.printErrorLog(TAG, "Error when trying to parse isLoading parameter: " + ex.getMessage());
        }
    }

    @Override
    @android.webkit.JavascriptInterface
    public void notifyIsVastAdReady(boolean isVastReady, String vastSettings)
    {
        onNotifyIsVastAdReady(isVastReady, vastSettings);
    }

    @Override
    @android.webkit.JavascriptInterface
    public void notifyIsVastAdReady(boolean isVastReady)
    {
        onNotifyIsVastAdReady(isVastReady);
    }

    @Override
    @android.webkit.JavascriptInterface
    public void onRewarded()
    {
        Message message = Message.obtain();
        message.what = ON_REWARDED;
        mEventHandler.sendMessage(message);
    }

    @Override
    @android.webkit.JavascriptInterface
    public void onRewardedVideoStarted()
    {
        Message message = Message.obtain();
        message.what = ON_REWARDED_STARTED;
        mEventHandler.sendMessage(message);
    }

    @Override
    @android.webkit.JavascriptInterface
    public void onAdStateChanged(int state)
    {
        Message message = Message.obtain();
        message.what = ON_AD_STATE_CHANGE;
        message.arg1 = state;
        mEventHandler.sendMessage(message);
    }

    @Override
    @android.webkit.JavascriptInterface
    public void setDeviceOrientation(final int orientation)
    {
        invokeSetDeviceOrientation(orientation);
    }

}
