package com.kidoz.sdk.api.server_connect;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.content.ContentValues;
import android.content.Context;
import android.os.AsyncTask;
import android.os.AsyncTask.Status;
import android.os.Build;

import com.kidoz.sdk.api.general.utils.PropertiesObj;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.structure.ContentData;

import org.json.JSONObject;

import java.io.BufferedInputStream;
import java.util.HashMap;
import java.util.Map.Entry;

import okhttp3.Call;
import okhttp3.Response;

@SuppressLint("NewApi")
class BaseAPIManager extends BaseConnectionClient
{
    private static final String TAG = BaseAPIManager.class.getSimpleName();
    private HashMap<SdkRequestType, RequestAsyncTask> runningTaskList = new HashMap<>();

    /**
     * Launch specific API request
     */
    @TargetApi(Build.VERSION_CODES.HONEYCOMB)
    protected void startServerConnection(Context context, String serverUrl, CONNECTION_TYPE type, SdkRequestType requestType, ContentValues contentValues, int numOfReconnecsOnFail, ApiResultCallback<?> resultCallback, boolean devLog, boolean removePrevRequest)
    {
        if (removePrevRequest)
        {
            try
            {
                if (runningTaskList.containsKey(requestType))
                {
                    if (runningTaskList.get(requestType) != null)
                    {
                        RequestAsyncTask task = runningTaskList.get(requestType);
                        if (task.getStatus() != Status.FINISHED)
                        {
                            task.cancel(true);
                            task.closeCurrentConnection();
                        } else
                        {
                            runningTaskList.remove(requestType);
                        }
                    }
                }
            } catch (Exception ex)
            {
                com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(TAG, " \n Unable to finish Running Request asyncTask ! \n\n " + ex.getMessage());
            }
        }

        RequestAsyncTask requestAsyncTask = new RequestAsyncTask(context, serverUrl, type, requestType, contentValues, numOfReconnecsOnFail, resultCallback, devLog);
        runningTaskList.put(requestType, requestAsyncTask);

        if (Build.VERSION.SDK_INT < 11)
        {
            requestAsyncTask.execute();
        } else
        {
            requestAsyncTask.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR);
        }
    }

    /**
     * Cancel all running task
     */
    public void cancelAllRunningRequests()
    {
        for (Entry<SdkRequestType, RequestAsyncTask> entry : runningTaskList.entrySet())
        {
            RequestAsyncTask asyncTask = entry.getValue();
            if (asyncTask != null)
            {
                if (asyncTask.getStatus() == Status.RUNNING && asyncTask.getStatus() == Status.PENDING)
                {
                    asyncTask.cancel(true);
                    asyncTask.closeCurrentConnection();
                }
            }
        }
        runningTaskList.clear();
    }

    /**
     * Background Async Task executor for HTTP server requests
     */
    class RequestAsyncTask extends AsyncTask<Void, Void, ResultData<?>>
    {

        private CONNECTION_TYPE restRequestType = CONNECTION_TYPE.POST;
        private ApiResultCallback<?> mResultCallback;
        private int numOfReconnectionsOnFail = 0;
        private SdkRequestType sdkRequestType;
        private ContentValues mContentValues;
        private Call mCall;
        private Context mContext;
        private boolean mDeveloperLogging = false;
        private String mServerUrl;

        public RequestAsyncTask(Context context, String serverUrl, CONNECTION_TYPE type, SdkRequestType requestType, ContentValues contentValues, int numOfReconnecsOnFail, ApiResultCallback<?> resultCallback, boolean devLog)
        {
            mContext = context;
            restRequestType = type;
            sdkRequestType = requestType;
            mContentValues = contentValues;
            mResultCallback = resultCallback;
            numOfReconnectionsOnFail = numOfReconnecsOnFail;
            mDeveloperLogging = devLog;
            mServerUrl = serverUrl;
        }

        @Override
        protected ResultData<?> doInBackground(Void... params)
        {
            String dataResponce = null;
            ResultData<?> webServiceResult = null;

            int mReconnectionTries = 0;
            if (!isCancelled())
            {
                Response response = null;
                while (mReconnectionTries <= numOfReconnectionsOnFail) // make reconnection tries
                {
                    mReconnectionTries = mReconnectionTries + 1;
                    if (!isCancelled())
                    {
                        try
                        {
                            if (restRequestType == CONNECTION_TYPE.POST)
                            {
                                mCall = makePostConnection(mServerUrl, mContentValues, sdkRequestType.name());
                                if (mCall != null)
                                {
                                    response = mCall.execute();
                                }
                            } else if (restRequestType == CONNECTION_TYPE.GET)
                            {
                                mCall = makeGetConnection(mServerUrl, mContentValues, sdkRequestType.name());
                                if (mCall != null)
                                {
                                    response = mCall.execute();
                                }
                            }
                        } catch (Exception e)
                        {
                            if (sdkRequestType != null)
                            {
                                com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(" \n IO Exception On [" + sdkRequestType.name() + "] request! \n" + e.getMessage());
                            }
                        }
                    } else
                    {
                        break;
                    }

                    if (response != null)
                    {
                        if (!isCancelled())
                        {
                            if (response.isSuccessful())
                            {
                                try
                                {
                                    com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(" \n Successful connection ! , Code :  " + response.code());
                                    dataResponce = StreamToStringConverter.readStream(new BufferedInputStream(response.body().byteStream()), this, true);
                                    response.body().close();
                                    break;
                                } catch (Exception e)
                                {
                                    com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(" \n Unable to convertByte Stream to String! : \n\n" + e.getMessage());
                                }
                            } else
                            {
                                com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(" \n Server connectivity Error!  Code : " + response.code());
                            }
                        } else
                        {
                            break;
                        }
                    }
                    try
                    {
                        if (!isCancelled())
                        {
                            Thread.sleep(300 * mReconnectionTries * 2);
                        } else
                        {
                            break;
                        }
                    } catch (Exception e)
                    {
                        break;
                    }
                }
            }

            // Print server response string before parsing data response
            if (sdkRequestType != null)
            {
                if (restRequestType == CONNECTION_TYPE.POST)
                {
                    SDKLogger.printPostRequestDebugLog(mServerUrl, mContentValues, sdkRequestType.name());
                } else if (restRequestType == CONNECTION_TYPE.GET)
                {
                    SDKLogger.printGetRequestDebugLog(mServerUrl, mContentValues, sdkRequestType.name());
                }

                SDKLogger.printResponse(dataResponce, sdkRequestType.name());
            }

            if (isCancelled())
            {
                return null;
            }

            if (dataResponce != null)
            {
                if (!isCancelled())
                {
                    if (mContext != null)
                    {
                        try
                        {
                            webServiceResult = parseWebServiceResponse(mContext, sdkRequestType, dataResponce, mDeveloperLogging);
                        } catch (Exception ex)
                        {
                            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(TAG, "Error when trying to parse service response: " + ex.getMessage());
                        }
                    }
                }
                return webServiceResult;
            } else
            {
                return null;
            }
        }

        @Override
        protected void onPostExecute(ResultData<?> result)
        {
            if (!isCancelled())
            {
                if (result == null)
                {
                    if (mResultCallback != null)
                    {
                        mResultCallback.onFailed();
                    }
                } else
                {
                    if (mResultCallback != null)
                    {
                        mResultCallback.onServerResult(result);
                    }
                }
            }
            runningTaskList.remove(sdkRequestType);
        }

        @Override
        protected void onCancelled()
        {
            super.onCancelled();
            if (sdkRequestType != null && runningTaskList != null)
            {
                runningTaskList.remove(sdkRequestType);
            }
        }

        public void closeCurrentConnection()
        {
            if (mCall != null)
            {
                if (mCall.isCanceled() == false)
                {
                    mCall.cancel();
                }
            }
        }
    }

    /**
     * Handle the paesing of web service requests
     */
    private ResultData<?> parseWebServiceResponse(Context context, SdkRequestType sdkRequestType, String dataResponce, boolean devLog)
    {

        ResultData<?> webServiceResult = null;
        if (sdkRequestType != null && context != null)
        {
            switch (sdkRequestType)
            {
                case GET_COUNTRY_CODE:
                {
                    try
                    {
                        ResponseStatus responseStatus = new ResponseStatus(dataResponce);
                        ResultData<String> resultData = new ResultData<>();
                        resultData.setResponseStatus(responseStatus);
                        if (responseStatus.getIsSuccessful() == true)
                        {
                            JSONObject jSONObject = new JSONObject(dataResponce);
                            JSONObject countryCodeJson = jSONObject.optJSONObject("data");
                            resultData.setData(countryCodeJson.optString("country_code"));
                        }
                        webServiceResult = resultData;
                    } catch (Exception ex)
                    {
                        com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(TAG, "Error when trying to parse GET_COUNTRY_CODE: " + ex.getMessage());
                    }
                    break;
                }

                //Parsers
                case LOAD_SDK_CONTENT:
                {
                    ContentData contentData = new ContentData();
                    contentData.decodeResponse(dataResponce);
                    ResultData<ContentData> resultData = new ResultData<>();
                    resultData.setData(contentData);
                    webServiceResult = resultData;
                    break;
                }
                case VALIDATE_SDK:
                {
                    try
                    {
                        ResponseStatus responseStatus = new ResponseStatus(dataResponce);
                        ResultData<PropertiesObj> resultData = new ResultData<>();
                        resultData.setResponseStatus(responseStatus);
                        if (responseStatus.getIsSuccessful() == true)
                        {
                            JSONObject jSONObject = new JSONObject(dataResponce);
                            PropertiesObj propertiesObj = new PropertiesObj(jSONObject.optJSONObject("data"));
                            resultData.setData(propertiesObj);
                        }
                        webServiceResult = resultData;
                    } catch (Exception ex)
                    {
                        com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(TAG, "Error when trying to parse validate SDK: " + ex.getMessage());
                    }
                    break;
                }
            }
        }

        return webServiceResult;
    }

    /**
     * Cancel all running task
     *
     * @param requestType type
     */
    public void cancelRequest(SdkRequestType requestType)
    {
        if (runningTaskList.containsKey(requestType))
        {
            if (runningTaskList.get(requestType) != null && runningTaskList.get(requestType).getStatus() != Status.FINISHED)
            {
                runningTaskList.get(requestType).cancel(true);
                runningTaskList.get(requestType).closeCurrentConnection();
            }
        }
    }
}
