package com.kidoz.sdk.api;

import android.content.Context;
import android.graphics.Color;
import android.util.AttributeSet;
import android.view.View;
import android.widget.RelativeLayout;

import com.kidoz.sdk.api.general.EventMessage;
import com.kidoz.sdk.api.general.assets_handling.AssetUtil;
import com.kidoz.sdk.api.general.database.DatabaseManager;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.interfaces.AnotherInterface;
import com.kidoz.sdk.api.interfaces.IOnPanelViewEventListener;
import com.kidoz.sdk.api.interfaces.KidozPlayerEventHelper;
import com.kidoz.sdk.api.interfaces.KidozPlayerListener;
import com.kidoz.sdk.api.ui_views.panel_view.AbstractPanelView;
import com.kidoz.sdk.api.ui_views.panel_view.FamilyPanelView;
import com.kidoz.sdk.api.ui_views.panel_view.HANDLE_POSITION;
import com.kidoz.sdk.api.ui_views.panel_view.PANEL_TYPE;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;
import org.json.JSONObject;

import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * KIDOZ PanelView widget class
 */
public class PanelView extends RelativeLayout
{
    public static final String TAG = PanelView.class.getSimpleName();

    private AbstractPanelView mAbstractPanelView;
    private PANEL_TYPE mPanelType = PANEL_TYPE.BOTTOM;
    private HANDLE_POSITION mHandlePosition = HANDLE_POSITION.START;
    private IOnPanelViewEventListener mIOnPanelViewEventListener = null;

    private KidozPlayerEventHelper mKidozPlayerEventHelper;
    private KidozPlayerListener mKidozPlayerListener;
    private Lock raceConditionLock = new ReentrantLock();

    public PanelView(Context context)
    {
        super(context);
        initView();
    }

    public PanelView(Context context, AttributeSet attrs)
    {
        super(context, attrs);
        initView();
    }

    public PanelView(Context context, AttributeSet attrs, int defStyleAttr)
    {
        super(context, attrs, defStyleAttr);
        initView();
    }

    /**
     * Set kidoz player event listeners.
     * Listen for events generated by any launched player (Unique for each widget)
     *
     * @param kidozPlayerListener event listener
     */
    public void setKidozPlayerListener(KidozPlayerListener kidozPlayerListener)
    {
        mKidozPlayerListener = kidozPlayerListener;
    }

    private void initView()
    {
        if (EventBus.getDefault().isRegistered(this) == false)
        {
            EventBus.getDefault().register(this);
        }

        setBackgroundColor(Color.TRANSPARENT);
        setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT));

        // We check if SDK initiation has successfully finished( Sdk Config data accepted)
        if (KidozSDK.isInitialised() == true)
        {
            validateLayoutAndAssets();
        }

        mKidozPlayerEventHelper = new KidozPlayerEventHelper(new KidozPlayerEventHelper.IPlayerListener()
        {
            @Override
            public void onOpen(boolean isMaximizedPlayer)
            {
                if (mKidozPlayerListener != null)
                {
                    mKidozPlayerListener.onPlayerOpen();
                }

                if (mAbstractPanelView != null && mAbstractPanelView instanceof FamilyPanelView)
                {
                    ((FamilyPanelView) mAbstractPanelView).pauseAction(isMaximizedPlayer);
                }
            }

            @Override
            public void onClose(boolean isMaximizedPlayer)
            {
                if (mKidozPlayerListener != null)
                {
                    mKidozPlayerListener.onPlayerClose();
                }

                if (mAbstractPanelView != null && mAbstractPanelView instanceof FamilyPanelView)
                {
                    ((FamilyPanelView) mAbstractPanelView).resumeAction(isMaximizedPlayer);
                }
            }
        }, WidgetType.WIDGET_TYPE_PANEL);


    }

    // long time = 0;
    private void validateLayoutAndAssets()
    {
        if (EventBus.getDefault().isRegistered(this))
        {
            EventBus.getDefault().unregister(this);
        }

        if (raceConditionLock.tryLock())
        {
            try
            {
                AssetUtil.parseStyleAsync(getContext(), AssetUtil.ParserAsyncTask.StyleType.PANEL_STYLE, new AssetUtil.ParserAsyncTask.IOnStyleParseListener()
                {
                    @Override
                    public void onParseFinished(boolean success)
                    {
                        if (success)
                        {
                            buildPanelView();
                        }
                    }
                });
            }
            finally
            {
                raceConditionLock.unlock();
            }
        }
    }

    /**
     * Build panel view according to accepted parameters from server
     */
    private void buildPanelView()
    {
        JSONObject jsonObject = DatabaseManager.getInstance(getContext()).getConfigTable().loadProperties(PanelView.TAG);
        if (jsonObject != null)
        {
            //String panelType = jsonObject.optString(PanelView.TAG);
            mAbstractPanelView = new FamilyPanelView(getContext(), jsonObject);

            if (mPanelType != null && mHandlePosition != null)
            {
                mAbstractPanelView.setPanelConfiguration(mPanelType, mHandlePosition);
            }

            if (mIOnPanelViewEventListener != null)
            {
                mAbstractPanelView.setOnPanelViewEventListener(mIOnPanelViewEventListener);
            }
        }

        if (mKidozPlayerEventHelper != null)
        {
            mKidozPlayerEventHelper.register();
        }
        addView(mAbstractPanelView);
        setAnotherInterface();
    }

    private void setAnotherInterface()
    {
        try
        {
            mAbstractPanelView.mOneItemFamilyView.setAnotherInterface(new AnotherInterface()
            {
                @Override
                public void toggleWidgetState(boolean isEnabled)
                {
                    mAbstractPanelView.mAboutKidozButton.setClickable(isEnabled);
                    mAbstractPanelView.mParentalLockButton.setClickable(isEnabled);
                    ((FamilyPanelView) mAbstractPanelView).mNextButton.setClickable(isEnabled);
                    mAbstractPanelView.mOneItemFamilyView.mViewPager.mIsBlocked = !isEnabled;
                    mAbstractPanelView.mPanelHandle.setEnabled(isEnabled);
                }
            });
        }
        catch (Exception ex)
        {
        }
    }

    // Event bus callback
    @Subscribe
    public void onHandleEvent(EventMessage event)
    {
        if (event.getMessageType() == EventMessage.MessageType.INIT_SDK)
        {
            validateLayoutAndAssets();
        }
    }

    /**
     * Set panel color
     *
     * @param color
     * @deprecated Deprecated from version 0.6.0 will be removed from version 0.7.2
     */
    @Deprecated
    public void setPanelColor(int color)
    {

    }

    /**
     * Collapse panel view
     */
    public void collapsePanelView()
    {
        if (mAbstractPanelView != null)
        {
            mAbstractPanelView.collapsePanelView();
        }
    }

    /**
     * Expand (Open) Panel view
     */
    public void expandPanelView()
    {
        if (mAbstractPanelView != null)
        {
            mAbstractPanelView.expandPanelView();
        }
    }

    /**
     * Set panel view configuration
     *
     * @param panelType      panel view type (Possible values <code> PANEL_TYPE.TOP, PANEL_TYPE.BOTTOM, PANEL_TYPE.LEFT ,PANEL_TYPE.RIGHT </code>
     * @param handlePosition handle position  (Possible values <code> HANDLE_POSITION.START, HANDLE_POSITION.CENTER, HANDLE_POSITION.END </code>
     */
    public void setPanelConfiguration(PANEL_TYPE panelType, HANDLE_POSITION handlePosition)
    {
        mPanelType = panelType;
        mHandlePosition = handlePosition;

        if (mAbstractPanelView != null)
        {
            if (panelType == PANEL_TYPE.TOP || panelType == PANEL_TYPE.BOTTOM)
            {
                mAbstractPanelView.setPanelConfiguration(panelType, handlePosition);
            }
        }
    }

    /**
     * Set Automatically expand panel for a period of time on being added to view
     *
     * @param startDelay delay in seconds before automatic invocation of panel expand , pass -1 to  disable
     * @param showPeriod period in seconds to show the panel before closing it, pass -1 to  disable
     * @deprecated Deprecated from version 0.6.0 will be removed from version 0.7.2
     */
    @Deprecated
    public void setPanelAutoExpand(float startDelay, float showPeriod)
    {

    }

    /**
     * Set Automatically expand panel for a period of time on being added to view
     *
     * @param startDelay      delay in seconds before automatic invocation of panel expand , pass -1 to  disable
     * @param showPeriod      period in seconds to show the panel before closing it, pass -1 to  disable
     * @param repeatTimeFrame time in hours , before repeating the invocation of panel expand on each "first lounch" , pass -1 to invoke panel on each "first lounch"
     * @deprecated Deprecated from version 0.6.0 will be removed from version 0.7.2
     */
    @Deprecated
    public void setPanelAutoExpand(float startDelay, float showPeriod, float repeatTimeFrame)
    {

    }

    /**
     * Set panel events listener
     *
     * @param iOnPanelViewEventListener listener
     */
    public void setOnPanelViewEventListener(IOnPanelViewEventListener iOnPanelViewEventListener)
    {
        mIOnPanelViewEventListener = iOnPanelViewEventListener;
        if (mAbstractPanelView != null)
        {
            mAbstractPanelView.setOnPanelViewEventListener(iOnPanelViewEventListener);
        }
    }

    /**
     * Get panel view state
     *
     * @return is panel expanded pr closed
     */
    public boolean getIsPanelViewExpanded()
    {
        if (mAbstractPanelView != null)
        {
            return mAbstractPanelView.getPanelViewState() == AbstractPanelView.PanelViewState.OPEN ? true : false;
        }
        else
        {
            return false;
        }
    }

    @Override
    protected void onDetachedFromWindow()
    {
        try
        {
            if (EventBus.getDefault().isRegistered(this))
            {
                EventBus.getDefault().unregister(this);
            }
        }
        catch (Exception e)
        {
        }

        if (mKidozPlayerEventHelper != null)
        {
            mKidozPlayerEventHelper.unRegister();
        }
        super.onDetachedFromWindow();
    }

    @Override
    protected void onAttachedToWindow()
    {
        super.onAttachedToWindow();
        if (mKidozPlayerEventHelper != null)
        {
            mKidozPlayerEventHelper.register();
        }
    }

    @Override
    public void setVisibility(int visibility)
    {
        if (visibility == View.GONE)
        {
            visibility = View.INVISIBLE;
        }
        super.setVisibility(visibility);

        if(visibility == View.INVISIBLE) {
            mAbstractPanelView.setPanelEnabled(false);
        }else {
            mAbstractPanelView.setPanelEnabled(true);
        }

    }
}
